/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.app.appsearch;

import android.content.Context;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

/**
 * Provides access to the centralized AppSearch index maintained by the system.
 *
 * <p>AppSearch is an offline, on-device search library for managing structured data featuring:
 *
 * <ul>
 *   <li>APIs to index and retrieve data via full-text search.
 *   <li>An API for applications to explicitly grant read-access permission of their data to other
 *       applications. <b>See: {@link android.app.appsearch.SetSchemaRequest.Builder#setSchemaTypeVisibilityForPackage        }</b>
 *   <li>An API for applications to opt into or out of having their data displayed on System UI
 *       surfaces by the System-designated global querier. <b>See: {@link android.app.appsearch.SetSchemaRequest.Builder#setSchemaTypeDisplayedBySystem        }</b>
 * </ul>
 *
 * <p>Applications create a database by opening an {@link android.app.appsearch.AppSearchSession AppSearchSession}.
 *
 * <p>Example:
 *
 * <pre>
 * AppSearchManager appSearchManager = context.getSystemService(AppSearchManager.class);
 *
 * AppSearchManager.SearchContext searchContext = new AppSearchManager.SearchContext.Builder().
 *    setDatabaseName(dbName).build());
 * appSearchManager.createSearchSession(searchContext, mExecutor, appSearchSessionResult -&gt; {
 *      mAppSearchSession = appSearchSessionResult.getResultValue();
 * });</pre>
 *
 * <p>After opening the session, a schema must be set in order to define the organizational
 * structure of data. The schema is set by calling {@link android.app.appsearch.AppSearchSession#setSchema AppSearchSession.setSchema}. The schema is
 * composed of a collection of {@link android.app.appsearch.AppSearchSchema AppSearchSchema} objects, each of which defines a unique type
 * of data.
 *
 * <p>Example:
 *
 * <pre>
 * AppSearchSchema emailSchemaType = new AppSearchSchema.Builder("Email")
 *     .addProperty(new StringPropertyConfig.Builder("subject")
 *        .setCardinality(PropertyConfig.CARDINALITY_OPTIONAL)
 *        .setIndexingType(PropertyConfig.INDEXING_TYPE_PREFIXES)
 *        .setTokenizerType(PropertyConfig.TOKENIZER_TYPE_PLAIN)
 *    .build()
 * ).build();
 *
 * SetSchemaRequest request = new SetSchemaRequest.Builder().addSchema(emailSchemaType).build();
 * mAppSearchSession.set(request, mExecutor, appSearchResult -&gt; {
 *      if (appSearchResult.isSuccess()) {
 *           // Schema has been successfully set.
 *      }
 * });</pre>
 *
 * <p>The basic unit of data in AppSearch is represented as a {@link android.app.appsearch.GenericDocument GenericDocument} object,
 * containing an ID, namespace, time-to-live, score, and properties. A namespace organizes a logical
 * group of documents. For example, a namespace can be created to group documents on a per-account
 * basis. An ID identifies a single document within a namespace. The combination of namespace and ID
 * uniquely identifies a {@link android.app.appsearch.GenericDocument GenericDocument} in the database.
 *
 * <p>Once the schema has been set, {@link android.app.appsearch.GenericDocument GenericDocument} objects can be put into the database and
 * indexed by calling {@link android.app.appsearch.AppSearchSession#put AppSearchSession.put}.
 *
 * <p>Example:
 *
 * <pre>
 * // Although for this example we use GenericDocument directly, we recommend extending
 * // GenericDocument to create specific types (i.e. Email) with specific setters/getters.
 * GenericDocument email = new GenericDocument.Builder<>(NAMESPACE, ID, EMAIL_SCHEMA_TYPE)
 *     .setPropertyString(“subject”, EMAIL_SUBJECT)
 *     .setScore(EMAIL_SCORE)
 *     .build();
 *
 * PutDocumentsRequest request = new PutDocumentsRequest.Builder().addGenericDocuments(email)
 *     .build();
 * mAppSearchSession.put(request, mExecutor, appSearchBatchResult -&gt; {
 *      if (appSearchBatchResult.isSuccess()) {
 *           // All documents have been successfully indexed.
 *      }
 * });</pre>
 *
 * <p>Searching within the database is done by calling {@link android.app.appsearch.AppSearchSession#search AppSearchSession.search} and providing
 * the query string to search for, as well as a {@link android.app.appsearch.SearchSpec SearchSpec}.
 *
 * <p>Alternatively, {@link android.app.appsearch.AppSearchSession#getByDocumentId AppSearchSession.getByDocumentId} can be called to retrieve documents by
 * namespace and ID.
 *
 * <p>Document removal is done either by time-to-live expiration, or explicitly calling a remove
 * operation. Remove operations can be done by namespace and ID via {@link android.app.appsearch.AppSearchSession#remove(android.app.appsearch.RemoveByDocumentIdRequest,java.util.concurrent.Executor,android.app.appsearch.BatchResultCallback)  }, or by query
 * via {@link android.app.appsearch.AppSearchSession#remove(java.lang.String,android.app.appsearch.SearchSpec,java.util.concurrent.Executor,java.util.function.Consumer) AppSearchSession.remove(String, SearchSpec, Executor, Consumer)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class AppSearchManager {

AppSearchManager() { throw new RuntimeException("Stub!"); }

/**
 * Creates a new {@link android.app.appsearch.EnterpriseGlobalSearchSession EnterpriseGlobalSearchSession}
 *
 * <p>EnterpriseGlobalSearchSession queries data from the user’s work profile, allowing apps
 * running on the personal profile to access a limited subset of work profile data. Enterprise
 * access must be explicitly enabled on schemas, and schemas may also specify additional
 * permissions required for enterprise access.
 *
 * <p>This process requires an AppSearch native indexing file system. If it's not created, the
 * initialization process will create one under the user's credential encrypted directory.
 *
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback The {@link android.app.appsearch.AppSearchResult AppSearchResult}&lt;{@link android.app.appsearch.EnterpriseGlobalSearchSession EnterpriseGlobalSearchSession}&gt; of
 *     performing this operation. Or a {@link android.app.appsearch.AppSearchResult AppSearchResult} with failure reason code and
 *     error information.

 * This value cannot be {@code null}.
 */

public void createEnterpriseGlobalSearchSession(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.EnterpriseGlobalSearchSession>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new {@link android.app.appsearch.GlobalSearchSession GlobalSearchSession}.
 *
 * <p>This process requires an AppSearch native indexing file system. If it's not created, the
 * initialization process will create one under the user's credential encrypted directory.
 *
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback The {@link android.app.appsearch.AppSearchResult AppSearchResult}&lt;{@link android.app.appsearch.GlobalSearchSession GlobalSearchSession}&gt; of performing
 *     this operation. Or a {@link android.app.appsearch.AppSearchResult AppSearchResult} with failure reason code and error
 *     information.

 * This value cannot be {@code null}.
 */

public void createGlobalSearchSession(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.GlobalSearchSession>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new {@link android.app.appsearch.AppSearchSession AppSearchSession}.
 *
 * <p>This process requires an AppSearch native indexing file system. If it's not created, the
 * initialization process will create one under the user's credential encrypted directory.
 *
 * @param searchContext The {@link android.app.appsearch.AppSearchManager.SearchContext SearchContext} contains all information to create a new
 *     {@link android.app.appsearch.AppSearchSession AppSearchSession}
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback The {@link android.app.appsearch.AppSearchResult AppSearchResult}&lt;{@link android.app.appsearch.AppSearchSession AppSearchSession}&gt; of performing
 *     this operation. Or a {@link android.app.appsearch.AppSearchResult AppSearchResult} with failure reason code and error
 *     information.

 * This value cannot be {@code null}.
 */

public void createSearchSession(@android.annotation.NonNull android.app.appsearch.AppSearchManager.SearchContext searchContext, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.AppSearchSession>> callback) { throw new RuntimeException("Stub!"); }
/** Contains information about how to create the search session. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class SearchContext {

SearchContext() { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of the database to create or open.
 *
 * <p>Databases with different names are fully separate with distinct types, namespaces, and
 * data.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.lang.String getDatabaseName() { throw new RuntimeException("Stub!"); }
/** Builder for {@link android.app.appsearch.AppSearchManager.SearchContext SearchContext} objects. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

/**
 * Creates a new {@link android.app.appsearch.AppSearchManager.SearchContext.Builder SearchContext.Builder}.
 *
 * <p>{@link android.app.appsearch.AppSearchSession AppSearchSession} will create or open a database under the given name.
 *
 * <p>Databases with different names are fully separate with distinct types, namespaces,
 * and data.
 *
 * <p>Database name cannot contain {@code '/'}.
 *
 * @param databaseName The name of the database.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the databaseName contains {@code '/'}.
 */

public Builder(@android.annotation.NonNull java.lang.String databaseName) { throw new RuntimeException("Stub!"); }

/**
 * Builds a {@link android.app.appsearch.AppSearchManager.SearchContext SearchContext} instance.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.AppSearchManager.SearchContext build() { throw new RuntimeException("Stub!"); }
}

}

}

