/*
 * Copyright 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.app.appsearch;


/**
 * A migrator class to translate {@link android.app.appsearch.GenericDocument GenericDocument} from different version of {@link android.app.appsearch.AppSearchSchema AppSearchSchema}
 *
 * <p>Make non-backwards-compatible changes will delete all stored documents in old schema. You can
 * save your documents by setting {@link android.app.appsearch.Migrator Migrator} via the {@link android.app.appsearch.SetSchemaRequest.Builder#setMigrator  } for each type and target version you want to save.
 *
 * <p>{@link #onDowngrade} or {@link #onUpgrade} will be triggered if the version number of the
 * schema stored in AppSearch is different with the version in the request.
 *
 * <p>If any error or Exception occurred in the {@link #onDowngrade} or {@link #onUpgrade}, all the
 * setSchema request will be rejected unless the schema changes are backwards-compatible, and stored
 * documents won't have any observable changes.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class Migrator {

public Migrator() { throw new RuntimeException("Stub!"); }

/**
 * Migrates {@link android.app.appsearch.GenericDocument GenericDocument} to an older version of {@link android.app.appsearch.AppSearchSchema AppSearchSchema}.
 *
 * <p>This method will be invoked only if the {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest} is setting a lower
 * version number than the current {@link android.app.appsearch.AppSearchSchema AppSearchSchema} saved in AppSearch.
 *
 * <p>If this {@link android.app.appsearch.Migrator Migrator} is provided to cover a compatible schema change via {@link android.app.appsearch.AppSearchSession#setSchema  }, documents under the old version won't be removed unless you use
 * the same document ID.
 *
 * <p>This method will be invoked on the background worker thread.
 *
 * <br>
 * This method may take several seconds to complete, so it should
 * only be called from a worker thread.
 * @param currentVersion The current version of the document's schema.
 * @param finalVersion The final version that documents need to be migrated to.
 * @param document The {@link android.app.appsearch.GenericDocument GenericDocument} need to be translated to new version.
 * @return A {@link android.app.appsearch.GenericDocument GenericDocument} in new version.
 */

@android.annotation.NonNull
public abstract android.app.appsearch.GenericDocument onDowngrade(int currentVersion, int finalVersion, @android.annotation.NonNull android.app.appsearch.GenericDocument document);

/**
 * Migrates {@link android.app.appsearch.GenericDocument GenericDocument} to a newer version of {@link android.app.appsearch.AppSearchSchema AppSearchSchema}.
 *
 * <p>This method will be invoked only if the {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest} is setting a higher
 * version number than the current {@link android.app.appsearch.AppSearchSchema AppSearchSchema} saved in AppSearch.
 *
 * <p>If this {@link android.app.appsearch.Migrator Migrator} is provided to cover a compatible schema change via {@link android.app.appsearch.AppSearchSession#setSchema  }, documents under the old version won't be removed unless you use
 * the same document ID.
 *
 * <p>This method will be invoked on the background worker thread provided via {@link android.app.appsearch.AppSearchSession#setSchema  }.
 *
 * <br>
 * This method may take several seconds to complete, so it should
 * only be called from a worker thread.
 * @param currentVersion The current version of the document's schema.
 * @param finalVersion The final version that documents need to be migrated to.
 * @param document The {@link android.app.appsearch.GenericDocument GenericDocument} need to be translated to new version.
 * @return A {@link android.app.appsearch.GenericDocument GenericDocument} in new version.
 */

@android.annotation.NonNull
public abstract android.app.appsearch.GenericDocument onUpgrade(int currentVersion, int finalVersion, @android.annotation.NonNull android.app.appsearch.GenericDocument document);

/**
 * Returns {@code true} if this migrator's source type needs to be migrated to update from
 * currentVersion to finalVersion.
 *
 * <p>Migration won't be triggered if currentVersion is equal to finalVersion even if {@link
 * #shouldMigrate} return true;
 */

public abstract boolean shouldMigrate(int currentVersion, int finalVersion);
}

