/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.graphics.pdf;

import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.pdf.content.PdfPageGotoLinkContent;
import android.graphics.pdf.content.PdfPageImageContent;
import android.graphics.pdf.content.PdfPageTextContent;
import android.graphics.pdf.models.FormEditRecord;
import android.graphics.pdf.models.FormWidgetInfo;
import android.graphics.pdf.models.PageMatchBounds;
import android.graphics.pdf.models.selection.PageSelection;
import android.graphics.pdf.models.selection.SelectionBoundary;
import android.os.ParcelFileDescriptor;
import java.io.IOException;
import java.util.List;

/**
 * <p>
 * This class enables rendering a PDF document and selecting, searching, fast scrolling,
 * annotations, etc. from Android V. This class is thread safe and can be called by
 * multiple threads however only one thread will be executed at a time as the access is
 * synchronized by internal locking.
 * <p>
 * If you want to render a PDF, you create a renderer and for every page you want
 * to render, you open the page, render it, and close the page. After you are done
 * with rendering, you close the renderer. After the renderer is closed it should not
 * be used anymore. Note that the pages are rendered one by one, i.e. you can have
 * only a single page opened at any given time.
 * <p>
 * A typical use of the APIs to render a PDF looks like this:
 * <pre>
 * // create a new renderer
 * PdfRenderer renderer = new PdfRenderer(getSeekableFileDescriptor());
 *
 * // let us just render all pages
 * final int pageCount = renderer.getPageCount();
 * for (int i = 0; i < pageCount; i++) {
 *     Page page = renderer.openPage(i);
 *
 *     // say we render for showing on the screen
 *     page.render(mBitmap, null, null, Page.RENDER_MODE_FOR_DISPLAY);
 *
 *     // do stuff with the bitmap
 *
 *     // close the page
 *     page.close();
 * }
 *
 * // close the renderer
 * renderer.close();
 * </pre>
 *
 * <h3>Print preview and print output</h3>
 * <p>
 * If you are using this class to rasterize a PDF for printing or show a print
 * preview, it is recommended that you respect the following contract in order
 * to provide a consistent user experience when seeing a preview and printing,
 * i.e. the user sees a preview that is the same as the printout.
 * <ul>
 * <li>
 * Respect the property whether the document would like to be scaled for printing
 * as per {@link #shouldScaleForPrinting()}.
 * </li>
 * <li>
 * When scaling a document for printing the aspect ratio should be preserved.
 * </li>
 * <li>
 * Do not inset the content with any margins from the {@link android.print.PrintAttributes}
 * as the application is responsible to render it such that the margins are respected.
 * </li>
 * <li>
 * If document page size is greater than the printed media size the content should
 * be anchored to the upper left corner of the page for left-to-right locales and
 * top right corner for right-to-left locales.
 * </li>
 * </ul>
 *
 * @see #close()
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class PdfRenderer implements java.lang.AutoCloseable {

/**
 * Creates a new instance.
 *
 * <p><strong>Note:</strong> The provided file descriptor must be <strong>seekable</strong>,
 * i.e. its data being randomly accessed, e.g. pointing to a file.
 * <p>
 * <strong>Note:</strong> This class takes ownership of the passed in file descriptor
 * and is responsible for closing it when the renderer is closed.
 * <p>
 * If the file is from an untrusted source it is recommended to run the renderer in a separate,
 * isolated process with minimal permissions to limit the impact of security exploits.
 * <strong>Note:</strong> The constructor should be instantiated on the
 * {@link android.annotation.WorkerThread} as it can be long-running while loading the
 * document.
 *
 * @param fileDescriptor Seekable file descriptor to read from.
 * This value cannot be {@code null}.
 * @throws java.io.IOException         If an error occurs while reading the file.
 * @throws java.lang.SecurityException If the file requires a password or
 *                                     the security scheme is not supported.
 * @throws java.lang.IllegalArgumentException    If the {@link android.os.ParcelFileDescriptor ParcelFileDescriptor} is not seekable.
 * @throws java.lang.NullPointerException        If the file descriptor is null.
 */

public PdfRenderer(@android.annotation.NonNull android.os.ParcelFileDescriptor fileDescriptor) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Creates a new instance of PdfRenderer class.
 * <p>
 * <strong>Note:</strong> The provided file descriptor must be <strong>seekable</strong>,
 * i.e. its data being randomly accessed, e.g. pointing to a file. If the password passed in
 * {@link android.graphics.pdf.LoadParams} is incorrect, the
 * {@link android.graphics.pdf.PdfRenderer} will throw a {@link java.lang.SecurityException SecurityException}.
 * <p>
 * <strong>Note:</strong> This class takes ownership of the passed in file descriptor
 * and is responsible for closing it when the renderer is closed.
 * <p>
 * If the file is from an untrusted source it is recommended to run the renderer in a separate,
 * isolated process with minimal permissions to limit the impact of security exploits.
 * <strong>Note:</strong> The constructor should be instantiated on the
 * {@link android.annotation.WorkerThread} as it can be long-running while loading the document.
 *
 * @param fileDescriptor Seekable file descriptor to read from.
 * This value cannot be {@code null}.
 * @param params         Instance of {@link android.graphics.pdf.LoadParams LoadParams} specifying params for loading PDF
 *                       document.
 * This value cannot be {@code null}.
 * @throws java.io.IOException         If an error occurs while reading the file.
 * @throws java.lang.SecurityException If the file requires a password or
 *                                     the security scheme is not supported by the renderer.
 * @throws java.lang.IllegalArgumentException    If the {@link android.os.ParcelFileDescriptor ParcelFileDescriptor} is not seekable.
 * @throws java.lang.NullPointerException        If the file descriptor or load params is null.
 */

public PdfRenderer(@android.annotation.NonNull android.os.ParcelFileDescriptor fileDescriptor, @android.annotation.NonNull android.graphics.pdf.LoadParams params) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Closes this renderer. You should not use this instance
 * after this method is called.
 *
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this or if any
 *                               page is opened and not closed
 * @see android.graphics.pdf.PdfRenderer.Page#close()
 */

public void close() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }

/**
 * Gets the type of the PDF document.
 *
 * @return The PDF document type.
 * Value is {@link android.graphics.pdf.PdfRenderer#DOCUMENT_LINEARIZED_TYPE_NON_LINEARIZED}, or {@link android.graphics.pdf.PdfRenderer#DOCUMENT_LINEARIZED_TYPE_LINEARIZED}
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public int getDocumentLinearizationType() { throw new RuntimeException("Stub!"); }

/**
 * Gets the number of pages in the document.
 *
 * @return The page count.
 * Value is 0 or greater
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public int getPageCount() { throw new RuntimeException("Stub!"); }

/**
 * Returns the form type of the loaded PDF
 *
 * @throws java.lang.IllegalStateException    if the renderer is closed
 * @throws java.lang.IllegalArgumentException if an unexpected PDF form type is returned

 * @return Value is {@link android.graphics.pdf.PdfRenderer#PDF_FORM_TYPE_NONE}, {@link android.graphics.pdf.PdfRenderer#PDF_FORM_TYPE_ACRO_FORM}, {@link android.graphics.pdf.PdfRenderer#PDF_FORM_TYPE_XFA_FULL}, or {@link android.graphics.pdf.PdfRenderer#PDF_FORM_TYPE_XFA_FOREGROUND}
 */

public int getPdfFormType() { throw new RuntimeException("Stub!"); }

/**
 * Opens a page for rendering.
 *
 * @param index The page index to open, starting from index 0.
 * Value is 0 or greater
 * @return A page that can be rendered.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException    If {@link #close()} is called before invoking this.
 * @throws java.lang.IllegalArgumentException If the page number is less than 0 or greater than or equal
 *                                  to the total page count.
 * @see android.graphics.pdf.PdfRenderer.Page#close()
 */

@android.annotation.NonNull
public android.graphics.pdf.PdfRenderer.Page openPage(int index) { throw new RuntimeException("Stub!"); }

/**
 * Gets whether the document prefers to be scaled for printing.
 * You should take this info account if the document is rendered
 * for printing and the target media size differs from the page
 * size.
 *
 * @return If to scale the document.
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public boolean shouldScaleForPrinting() { throw new RuntimeException("Stub!"); }

/**
 * Saves the current state of the loaded PDF document to the given writable
 * {@link android.os.ParcelFileDescriptor ParcelFileDescriptor}. If the document is password-protected then setting
 * {@code removePasswordProtection} removes the protection before saving. The PDF document
 * should already be decrypted with the correct password before writing. Useful for printing or
 * sharing.
 * <strong>Note:</strong> This method closes the provided file descriptor.
 *
 * @param destination              The writable {@link android.os.ParcelFileDescriptor ParcelFileDescriptor}
 * This value cannot be {@code null}.
 * @param removePasswordProtection If true, removes password protection from the PDF before
 *                                 saving.
 * @throws java.io.IOException           If there's a write error, or if 'removePasswordSecurity' is
 *                               {@code true} but the document remains encrypted.
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public void write(@android.annotation.NonNull android.os.ParcelFileDescriptor destination, boolean removePasswordProtection) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/** Represents a linearized PDF document. */

public static final int DOCUMENT_LINEARIZED_TYPE_LINEARIZED = 1; // 0x1

/** Represents a non-linearized PDF document. */

public static final int DOCUMENT_LINEARIZED_TYPE_NON_LINEARIZED = 0; // 0x0

/** Represents a PDF with form fields specified using the AcroForm spec */

public static final int PDF_FORM_TYPE_ACRO_FORM = 1; // 0x1

/** Represents a PDF without form fields */

public static final int PDF_FORM_TYPE_NONE = 0; // 0x0

/** Represents a PDF with form fields specified using the XFAF subset of the XFA spec */

public static final int PDF_FORM_TYPE_XFA_FOREGROUND = 3; // 0x3

/** Represents a PDF with form fields specified using the entire XFA spec */

public static final int PDF_FORM_TYPE_XFA_FULL = 2; // 0x2
/**
 * This class represents a PDF document page for rendering.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class Page implements java.lang.AutoCloseable {

Page() { throw new RuntimeException("Stub!"); }

/**
 * Applies a {@link android.graphics.pdf.models.FormEditRecord FormEditRecord} to the PDF.
 *
 * <p>Apps must call {@link #render(android.graphics.Bitmap,android.graphics.Rect,android.graphics.Matrix,android.graphics.pdf.RenderParams)} to render new
 * bitmaps for the corresponding areas of the page.
 *
 * <p>For click type {@link android.graphics.pdf.models.FormEditRecord FormEditRecord}s, performs a click on {@link android.graphics.pdf.models.FormEditRecord#getClickPoint()  }
 *
 * <p>For set text type {@link android.graphics.pdf.models.FormEditRecord FormEditRecord}s, sets the text value of the form widget.
 *
 * <p>For set indices type {@link android.graphics.pdf.models.FormEditRecord FormEditRecord}s, sets the {@link android.graphics.pdf.models.FormEditRecord#getSelectedIndices()  } as selected and all others as unselected for the
 * form widget indicated by the record.
 *
 * @param editRecord the {@link android.graphics.pdf.models.FormEditRecord FormEditRecord} to be applied
 * This value cannot be {@code null}.
 * @return Rectangular areas of the page bitmap that have been invalidated by this action.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided {@link android.graphics.pdf.models.FormEditRecord FormEditRecord} cannot be
 *                                  applied to the widget indicated by the index, or if the
 *                                  index does not correspond to a widget on the page.
 * @throws java.lang.IllegalStateException    If the document is already closed.
 * @throws java.lang.IllegalStateException    If the page is already closed.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.Rect> applyEdit(@android.annotation.NonNull android.graphics.pdf.models.FormEditRecord editRecord) { throw new RuntimeException("Stub!"); }

/**
 * Closes this page.
 *
 * @see android.graphics.pdf.PdfRenderer#openPage(int)
 */

public void close() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }

/**
 * Returns information about the widget with {@code widgetIndex}.
 *
 * @param widgetIndex the index of the widget within the page's "Annot" array in the PDF
 *                    document, available on results of previous calls to {@link
 *                    #getFormWidgetInfos(int[])} or
 *                    {@link #getFormWidgetInfoAtPosition(int,int)} via
 *                    {@link android.graphics.pdf.models.FormWidgetInfo#getWidgetIndex() FormWidgetInfo.getWidgetIndex()}.
 * Value is 0 or greater
 * @throws java.lang.IllegalArgumentException if there is no form widget at the provided index.
 * @throws java.lang.IllegalStateException    if the renderer or page is closed
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.graphics.pdf.models.FormWidgetInfo getFormWidgetInfoAtIndex(int widgetIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns information about the widget at the given point.
 *
 * @param x the x position of the widget on the page, in points
 * @param y the y position of the widget on the page, in points
 * @throws java.lang.IllegalArgumentException if there is no form widget at the provided position.
 * @throws java.lang.IllegalStateException    if the renderer or page is closed

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.graphics.pdf.models.FormWidgetInfo getFormWidgetInfoAtPosition(int x, int y) { throw new RuntimeException("Stub!"); }

/**
 * Returns information about all form widgets on the page, or an empty list if there are no
 * form widgets on the page.
 *
 * @throws java.lang.IllegalStateException if the renderer or page is closed
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.models.FormWidgetInfo> getFormWidgetInfos() { throw new RuntimeException("Stub!"); }

/**
 * Returns information about all form widgets of the specified types on the page, or an
 * empty list if there are no form widgets of the specified types on the page.
 *
 * @param types the types of form widgets to return, or an empty array to return all widgets
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException if the renderer or page is closed
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.models.FormWidgetInfo> getFormWidgetInfos(@android.annotation.NonNull int[] types) { throw new RuntimeException("Stub!"); }

/**
 * Gets bookmarks and goto links present on the page of a pdf document. Goto Links
 * are the internal navigation links which directs the user to different location
 * within the same document.
 *
 * @return list of all goto links {@link android.graphics.pdf.content.PdfPageGotoLinkContent PdfPageGotoLinkContent} on a page, ordered
 * left to right and top to bottom.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageGotoLinkContent> getGotoLinks() { throw new RuntimeException("Stub!"); }

/**
 * Returns the height of the {@link android.graphics.pdf.PdfRenderer.Page PdfRenderer.Page} object in points (1/72"). It is
 * not guaranteed that all pages will have the same height and the viewport should be
 * resized to the page height.
 *
 * @return height of the page
 * Value is 0 or greater
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

public int getHeight() { throw new RuntimeException("Stub!"); }

/**
 * Return list of {@link android.graphics.pdf.content.PdfPageImageContent PdfPageImageContent} found on the page, ordered left to right
 * and top to bottom. It contains all the content associated with images found on the
 * page including alt text. The list will be empty if there are no results found.
 * Currently, localisation does not have any impact on the order in which
 * {@link android.graphics.pdf.content.PdfPageImageContent PdfPageImageContent} is returned.
 *
 * @return list of image content found on the page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageImageContent> getImageContents() { throw new RuntimeException("Stub!"); }

/**
 * Gets the page index.
 *
 * @return The index.

 * Value is 0 or greater
 */

public int getIndex() { throw new RuntimeException("Stub!"); }

/**
 * Get the bounds and URLs of all the links on the page.
 *
 * @return list of all links on the page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageLinkContent> getLinkContents() { throw new RuntimeException("Stub!"); }

/**
 * Return list of {@link android.graphics.pdf.content.PdfPageTextContent PdfPageTextContent} found on the page, ordered left to right
 * and top to bottom. It contains all the content associated with text found on the page.
 * The list will be empty if there are no results found. Currently, localisation does
 * not have any impact on the order in which {@link android.graphics.pdf.content.PdfPageTextContent PdfPageTextContent} is returned.
 *
 * @return list of text content found on the page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageTextContent> getTextContents() { throw new RuntimeException("Stub!"); }

/**
 * Returns the width of the {@link android.graphics.pdf.PdfRenderer.Page PdfRenderer.Page} object in points (1/72"). It is
 * not guaranteed that all pages will have the same width and the viewport should be resized
 * to the page width.
 *
 * @return width of the page
 * Value is 0 or greater
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

public int getWidth() { throw new RuntimeException("Stub!"); }

/**
 * Renders a page to a bitmap. In case of default zoom, the {@link android.graphics.Bitmap Bitmap} dimensions will
 * be equal to the page dimensions. In this case, {@link android.graphics.Rect Rect} parameter can be null.
 *
 * <p>In case of zoom, the {@link android.graphics.Rect Rect} parameter needs to be specified which represents
 * the offset from top and left for tile generation purposes. In this case, the
 * {@link android.graphics.Bitmap Bitmap} dimensions should be equal to the tile dimensions.
 * <p>
 * <strong>Note:</strong> The method will take care of closing the bitmap. Should be
 * invoked
 * on the {@link android.annotation.WorkerThread} as it is long-running task.
 *
 * @param destination Destination bitmap to write to.
 * This value cannot be {@code null}.
 * @param destClip    If null, default zoom is applied. In case the value is non-null, the
 *                    value specifies the top top-left corner of the tile.
 * @param transform   Applied to scale the bitmap up/down from default 1/72 points.
 * This value may be {@code null}.
 * @param params      Render params for the changing display mode and/or annotations.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

public void render(@android.annotation.NonNull android.graphics.Bitmap destination, @android.annotation.Nullable android.graphics.Rect destClip, @android.annotation.Nullable android.graphics.Matrix transform, @android.annotation.NonNull android.graphics.pdf.RenderParams params) { throw new RuntimeException("Stub!"); }

/**
 * Renders a page to a bitmap.
 * <p>
 * You may optionally specify a rectangular clip in the bitmap bounds. No rendering
 * outside the clip will be performed, hence it is your responsibility to initialize
 * the bitmap outside the clip.
 * <p>
 * You may optionally specify a matrix to transform the content from page coordinates
 * which are in points (1/72") to bitmap coordinates which are in pixels. If this
 * matrix is not provided this method will apply a transformation that will fit the
 * whole page to the destination clip if provided or the destination bitmap if no
 * clip is provided.
 * <p>
 * The clip and transformation are useful for implementing tile rendering where the
 * destination bitmap contains a portion of the image, for example when zooming.
 * Another useful application is for printing where the size of the bitmap holding
 * the page is too large and a client can render the page in stripes.
 * <p>
 * <strong>Note: </strong> The destination bitmap format must be
 * {@link android.graphics.Bitmap.Config#ARGB_8888 ARGB}.
 * <p>
 * <strong>Note: </strong> The optional transformation matrix must be affine as per
 * {@link android.graphics.Matrix#isAffine() Matrix.isAffine()}. Hence, you can specify
 * rotation, scaling, translation but not a perspective transformation.
 *
 * @param destination Destination bitmap to which to render.
 * This value cannot be {@code null}.
 * @param destClip    Optional clip in the bitmap bounds.
 * This value may be {@code null}.
 * @param transform   Optional transformation to apply when rendering.
 * This value may be {@code null}.
 * @param renderMode  The render mode.
 * Value is {@link android.graphics.pdf.PdfRenderer.Page#RENDER_MODE_FOR_DISPLAY}, or {@link android.graphics.pdf.PdfRenderer.Page#RENDER_MODE_FOR_PRINT}
 * @see #RENDER_MODE_FOR_DISPLAY
 * @see #RENDER_MODE_FOR_PRINT
 */

public void render(@android.annotation.NonNull android.graphics.Bitmap destination, @android.annotation.Nullable android.graphics.Rect destClip, @android.annotation.Nullable android.graphics.Matrix transform, int renderMode) { throw new RuntimeException("Stub!"); }

/**
 * Search for the given string on the page and returns the bounds of all the matches. The
 * list will be empty if there are no matches on the page. If this function was
 * invoked previously for any page, it will wait for that operation to
 * complete before this operation is started.
 * <p>
 * <strong>Note:</strong> Should be invoked on the {@link android.annotation.WorkerThread}
 * as it is long-running task.
 *
 * @param query plain search string for querying the document
 * This value cannot be {@code null}.
 * @return List of {@link android.graphics.pdf.models.PageMatchBounds PageMatchBounds} representing the bounds of each match on the
 * page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.models.PageMatchBounds> searchText(@android.annotation.NonNull java.lang.String query) { throw new RuntimeException("Stub!"); }

/**
 * Return a {@link android.graphics.pdf.models.selection.PageSelection PageSelection} which represents the selected content that spans between
 * the two boundaries. The boundaries can be either exactly defined with text indexes, or
 * approximately defined with points on the page. The resulting selection will also be
 * exactly defined with both indexes and points. If the start and stop boundary are both at
 * the same point, selects the word at that point. In case the selection from the given
 * boundaries result in an empty space, then the method returns {@code null}. The start and
 * stop {@link android.graphics.pdf.models.selection.SelectionBoundary SelectionBoundary} in {@link android.graphics.pdf.models.selection.PageSelection PageSelection} resolves to the "nearest" index
 * when returned.
 * <p>
 * <strong>Note:</strong> Should be invoked on a {@link android.annotation.WorkerThread}
 * as it is long-running task.
 *
 * @param start boundary where the selection starts (inclusive)
 * This value cannot be {@code null}.
 * @param stop  boundary where the selection stops (exclusive)
 * This value cannot be {@code null}.
 * @return collection of the selected content for text, images, etc.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.Nullable
public android.graphics.pdf.models.selection.PageSelection selectContent(@android.annotation.NonNull android.graphics.pdf.models.selection.SelectionBoundary start, @android.annotation.NonNull android.graphics.pdf.models.selection.SelectionBoundary stop) { throw new RuntimeException("Stub!"); }

/**
 * Mode to render the content for display on a screen.
 */

public static final int RENDER_MODE_FOR_DISPLAY = 1; // 0x1

/**
 * Mode to render the content for printing.
 */

public static final int RENDER_MODE_FOR_PRINT = 2; // 0x2
}

}

