/*
 * Copyright (C) 2025 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net;

import android.os.Build;

/**
 * A {@link android.net.NetworkSpecifier NetworkSpecifier} used to identify an L2CAP network over BLE.
 *
 * An L2CAP network is not symmetrical, meaning there exists both a server (Bluetooth peripheral)
 * and a client (Bluetooth central) node. This specifier contains the information required to
 * request a client L2CAP network using {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager.requestNetwork} while specifying
 * the remote MAC address, and Protocol/Service Multiplexer (PSM). It can also contain information
 * allocated by the system when reserving a server network using {@link android.net.ConnectivityManager#reserveNetwork  } such as the Protocol/Service Multiplexer (PSM). In both
 * cases, the header compression option must be specified.
 *
 * An L2CAP server network allocates a Protocol/Service Multiplexer (PSM) to be advertised to the
 * client. A new server network must always be reserved using {@code
 * ConnectivityManager#reserveNetwork}. The subsequent {@link android.net.ConnectivityManager.NetworkCallback#onReserved(android.net.NetworkCapabilities)  } callback includes an {@code
 * L2CapNetworkSpecifier}. The {@link getPsm()} method will return the Protocol/Service Multiplexer
 * (PSM) of the reserved network so that the server can advertise it to the client and the client
 * can connect.
 * An L2CAP server network is backed by a {@link android.bluetooth.BluetoothServerSocket} which can,
 * in theory, accept many connections. However, before SDK version {@link
 * Build.VERSION_CODES.VANILLA_ICE_CREAM} Bluetooth APIs do not expose the channel ID, so these
 * connections are indistinguishable. In practice, this means that the network matching semantics in
 * ConnectivityService will tear down all but the first connection.
 *
 * When the connection between client and server completes, a {@link android.net.Network Network} whose capabilities
 * satisfy this {@code L2capNetworkSpecifier} will connect and the usual callbacks, such as {@link
 * NetworkCallback#onAvailable}, will be called on the callback object passed to {@code
 * ConnectivityManager#reserveNetwork} or {@code ConnectivityManager#requestNetwork}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class L2capNetworkSpecifier extends android.net.NetworkSpecifier implements android.os.Parcelable {

L2capNetworkSpecifier() { throw new RuntimeException("Stub!"); }

/** @hide */

public boolean canBeSatisfiedBy(android.net.NetworkSpecifier other) { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

/** @hide */

public boolean equals(java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * Returns the compression mechanism for this network.
 * @return Value is {@link android.net.L2capNetworkSpecifier#HEADER_COMPRESSION_ANY}, {@link android.net.L2capNetworkSpecifier#HEADER_COMPRESSION_NONE}, or {@link android.net.L2capNetworkSpecifier#HEADER_COMPRESSION_6LOWPAN}
 */

public int getHeaderCompression() { throw new RuntimeException("Stub!"); }

/** Returns the Protocol/Service Multiplexer (PSM) for this network to connect to. */

public int getPsm() { throw new RuntimeException("Stub!"); }

/**
 * Returns the remote MAC address for this network to connect to.
 *
 * The remote address is only meaningful for networks that have ROLE_CLIENT.
 *
 * When receiving this {@link android.net.L2capNetworkSpecifier L2capNetworkSpecifier} from Connectivity APIs such as a {@link android.net.ConnectivityManager.NetworkCallback ConnectivityManager.NetworkCallback}, the MAC address is redacted.

 * @return This value may be {@code null}.
 */

@android.annotation.Nullable
public android.net.MacAddress getRemoteAddress() { throw new RuntimeException("Stub!"); }

/**
 * Returns the role to be used for this network.
 * @return Value is {@link android.net.L2capNetworkSpecifier#ROLE_ANY}, {@link android.net.L2capNetworkSpecifier#ROLE_CLIENT}, or {@link android.net.L2capNetworkSpecifier#ROLE_SERVER}
 */

public int getRole() { throw new RuntimeException("Stub!"); }

/** @hide */

public int hashCode() { throw new RuntimeException("Stub!"); }

/** @hide */

@android.annotation.Nullable
public android.net.NetworkSpecifier redact() { throw new RuntimeException("Stub!"); }

/** @hide */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param dest This value cannot be {@code null}.
 */

public void writeToParcel(@android.annotation.NonNull android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.L2capNetworkSpecifier> CREATOR;
static { CREATOR = null; }

/** Use 6lowpan header compression as specified in rfc6282. */

public static final int HEADER_COMPRESSION_6LOWPAN = 2; // 0x2

/**
 * Accept any form of header compression.
 *
 * This option is only meaningful in {@link android.net.NetworkRequest NetworkRequest}s. Specifiers for actual L2CAP
 * networks never have this option set.
 */

public static final int HEADER_COMPRESSION_ANY = 0; // 0x0

/** Do not compress packets on this network. */

public static final int HEADER_COMPRESSION_NONE = 1; // 0x1

/**
 * Match any Protocol/Service Multiplexer (PSM).
 *
 * This PSM value is only meaningful in {@link android.net.NetworkRequest NetworkRequest}s. Specifiers for actual L2CAP
 * networks never have this value set.
 */

public static final int PSM_ANY = 0; // 0x0

/**
 * Match any role.
 *
 * This role is only meaningful in {@link android.net.NetworkRequest NetworkRequest}s. Specifiers for actual L2CAP
 * networks never have this role set.
 */

public static final int ROLE_ANY = 0; // 0x0

/** Specifier describes a client network, i.e., the device is the Bluetooth central. */

public static final int ROLE_CLIENT = 1; // 0x1

/** Specifier describes a server network, i.e., the device is the Bluetooth peripheral. */

public static final int ROLE_SERVER = 2; // 0x2
/** A builder class for L2capNetworkSpecifier. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Create the L2capNetworkSpecifier object.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.L2capNetworkSpecifier build() { throw new RuntimeException("Stub!"); }

/**
 * Set the header compression mechanism to use for this network.
 *
 * If not set, defaults to {@link HEADER_COMPRESSION_ANY}. This option must be specified
 * (i.e. must not be set to {@link HEADER_COMPRESSION_ANY}) when requesting or reserving a
 * new network.
 *
 * @param headerCompression the header compression mechanism to use.
 
 * Value is {@link android.net.L2capNetworkSpecifier#HEADER_COMPRESSION_ANY}, {@link android.net.L2capNetworkSpecifier#HEADER_COMPRESSION_NONE}, or {@link android.net.L2capNetworkSpecifier#HEADER_COMPRESSION_6LOWPAN}
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.L2capNetworkSpecifier.Builder setHeaderCompression(int headerCompression) { throw new RuntimeException("Stub!"); }

/**
 * Set the Protocol/Service Multiplexer (PSM) for the client to connect to.
 *
 * If not set, defaults to {@link PSM_ANY}.
 *
 * @param psm the Protocol/Service Multiplexer (PSM) to connect to.
 
 * Value is between 0 and 255 inclusive
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.L2capNetworkSpecifier.Builder setPsm(int psm) { throw new RuntimeException("Stub!"); }

/**
 * Set the remote address for the client to connect to.
 *
 * Only valid for client networks. If not set, the specifier matches any MAC address.
 *
 * @param remoteAddress the MAC address to connect to, or null to match any MAC address.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.L2capNetworkSpecifier.Builder setRemoteAddress(@android.annotation.Nullable android.net.MacAddress remoteAddress) { throw new RuntimeException("Stub!"); }

/**
 * Set the role to use for this network.
 *
 * If not set, defaults to {@link ROLE_ANY}.
 *
 * @param role the role to use.
 
 * Value is {@link android.net.L2capNetworkSpecifier#ROLE_ANY}, {@link android.net.L2capNetworkSpecifier#ROLE_CLIENT}, or {@link android.net.L2capNetworkSpecifier#ROLE_SERVER}
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.L2capNetworkSpecifier.Builder setRole(int role) { throw new RuntimeException("Stub!"); }
}

}

