// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.


package android.net.http;

import android.content.Context;
import java.io.IOException;
import java.net.URL;
import java.net.URLStreamHandlerFactory;
import java.util.concurrent.Executor;
import javax.net.ssl.HttpsURLConnection;

/**
 * An engine to process {@link android.net.http.UrlRequest UrlRequest}s, which uses the best HTTP stack available on the current
 * platform. An instance of this class can be created using {@link android.net.http.HttpEngine.Builder Builder}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class HttpEngine {

HttpEngine() { throw new RuntimeException("Stub!"); }

/**
 * Binds the engine to the specified network. All requests created through this engine
 * will use the network associated to this handle. If this network disconnects all requests will
 * fail, the exact error will depend on the stage of request processing when the network
 * disconnects.
 *
 * @param network the network to bind the engine to. Specify {@code null} to unbind.
 */

public void bindToNetwork(@android.annotation.Nullable android.net.Network network) { throw new RuntimeException("Stub!"); }

/**
 * Creates a {@link java.net.URLStreamHandlerFactory URLStreamHandlerFactory} to handle HTTP and HTTPS
 * traffic. An instance of this class can be installed via
 * {@link java.net.URL#setURLStreamHandlerFactory URL.setURLStreamHandlerFactory} thus using this {@link android.net.http.HttpEngine HttpEngine} by default for
 * all requests created via {@link java.net.URL#openConnection URL.openConnection}.
 * <p>
 * This {@link java.net.HttpURLConnection} implementation does not implement all features
 * offered by the API:
 * <ul>
 * <li>the HTTP cache installed via
 *     {@link android.net.http.HttpResponseCache#install(java.io.File,long) HttpResponseCache.install()}</li>
 * <li>the HTTP authentication method installed via
 *     {@link java.net.Authenticator#setDefault}</li>
 * <li>the HTTP cookie storage installed via {@link java.net.CookieHandler#setDefault}</li>
 * </ul>
 * <p>
 * While we support and encourages requests using the HTTPS protocol, we don't provide support
 * for the {@link javax.net.ssl.HttpsURLConnection HttpsURLConnection} API. This lack of support also includes not using certain
 * HTTPS features provided via {@link javax.net.ssl.HttpsURLConnection HttpsURLConnection}:
 * <ul>
 * <li>the HTTPS hostname verifier installed via {@link javax.net.ssl.HttpsURLConnection#setDefaultHostnameVerifier(javax.net.ssl.HostnameVerifier)      HttpsURLConnection.setDefaultHostnameVerifier()}</li>
 * <li>the HTTPS socket factory installed via {@link javax.net.ssl.HttpsURLConnection#setDefaultSSLSocketFactory(javax.net.ssl.SSLSocketFactory)      HttpsURLConnection.setDefaultSSLSocketFactory()}</li>
 * </ul>
 *
 * @return an {@link java.net.URLStreamHandlerFactory URLStreamHandlerFactory} instance implemented by this
 *         {@link android.net.http.HttpEngine HttpEngine}.
 */

@android.annotation.NonNull
public abstract java.net.URLStreamHandlerFactory createUrlStreamHandlerFactory();

/**
 * @return a human-readable version string of the engine.
 */

@android.annotation.NonNull
public static java.lang.String getVersionString() { throw new RuntimeException("Stub!"); }

/**
 * Creates a builder for {@link android.net.http.BidirectionalStream BidirectionalStream} objects. All callbacks for
 * generated {@code BidirectionalStream} objects will be invoked on
 * {@code executor}. {@code executor} must not run tasks on the
 * current thread, otherwise the networking operations may block and exceptions
 * may be thrown at shutdown time.
 *
 * @param url URL for the generated streams.
 * @param executor the {@link java.util.concurrent.Executor Executor} on which {@code callback} methods will be invoked.
 * @param callback the {@link android.net.http.BidirectionalStream.Callback BidirectionalStream.Callback} object that gets invoked upon
 * different events occurring.
 *
 * @return the created builder.
 */

@android.annotation.NonNull
public abstract android.net.http.BidirectionalStream.Builder newBidirectionalStreamBuilder(@android.annotation.NonNull java.lang.String url, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.http.BidirectionalStream.Callback callback);

/**
 * Creates a builder for {@link android.net.http.UrlRequest UrlRequest}. All callbacks for
 * generated {@link android.net.http.UrlRequest UrlRequest} objects will be invoked on
 * {@code executor}'s threads. {@code executor} must not run tasks on the
 * thread calling {@link java.util.concurrent.Executor#execute Executor.execute} to prevent blocking networking
 * operations and causing exceptions during shutdown.
 *
 * @param url URL for the generated requests.
 * @param executor {@link java.util.concurrent.Executor Executor} on which all callbacks will be invoked.
 * @param callback callback object that gets invoked on different events.
 */

@android.annotation.NonNull
public abstract android.net.http.UrlRequest.Builder newUrlRequestBuilder(@android.annotation.NonNull java.lang.String url, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.http.UrlRequest.Callback callback);

/**
 * Establishes a new connection to the resource specified by the {@link java.net.URL URL} {@code url}.
 * <p>
 * <b>Note:</b> This {@link java.net.HttpURLConnection} implementation is subject to certain
 * limitations, see {@link #createUrlStreamHandlerFactory} for details.
 *
 * @param url URL of resource to connect to.
 * @return an {@link java.net.HttpURLConnection} instance implemented
 *     by this {@link android.net.http.HttpEngine HttpEngine}.
 * @throws java.io.IOException if an error occurs while opening the connection.
 */

@android.annotation.NonNull
public abstract java.net.URLConnection openConnection(@android.annotation.NonNull java.net.URL url) throws java.io.IOException;

/**
 * Shuts down the {@link android.net.http.HttpEngine HttpEngine} if there are no active requests,
 * otherwise throws an exception.
 *
 * Cannot be called on network thread - the thread the HTTP stack calls into
 * Executor on (which is different from the thread the Executor invokes
 * callbacks on). May block until all the {@link android.net.http.HttpEngine HttpEngine} resources have been cleaned up.
 */

public abstract void shutdown();
/**
 * A builder for {@link android.net.http.HttpEngine HttpEngine}s, which allows runtime configuration of
 * {@link android.net.http.HttpEngine HttpEngine}. Configuration options are set on the builder and
 * then {@link #build} is called to create the {@link android.net.http.HttpEngine HttpEngine}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class Builder {

/**
 * Constructs a {@link android.net.http.HttpEngine.Builder Builder} object that facilitates creating a
 * {@link android.net.http.HttpEngine HttpEngine}. The default configuration enables HTTP/2 and
 * QUIC, but disables the HTTP cache.
 *
 * @param context Android {@link android.content.Context Context}, which is used by {@link android.net.http.HttpEngine.Builder Builder} to retrieve the
 * application context. A reference to only the application context will be kept, so as to
 * avoid extending the lifetime of {@code context} unnecessarily.
 */

public Builder(@android.annotation.NonNull android.content.Context context) { throw new RuntimeException("Stub!"); }

/**
 * Pins a set of public keys for a given host. By pinning a set of public keys, {@code
 * pinsSha256}, communication with {@code hostName} is required to authenticate with a
 * certificate with a public key from the set of pinned ones. An app can pin the public key
 * of the root certificate, any of the intermediate certificates or the end-entry
 * certificate. Authentication will fail and secure communication will not be established if
 * none of the public keys is present in the host's certificate chain, even if the host
 * attempts to authenticate with a certificate allowed by the device's trusted store of
 * certificates.
 *
 * <p>Calling this method multiple times with the same host name overrides the previously
 * set pins for the host.
 *
 * <p>More information about the public key pinning can be found in <a
 * href="https://tools.ietf.org/html/rfc7469">RFC 7469</a>.
 *
 * @param hostName name of the host to which the public keys should be pinned. A host that
 * consists only of digits and the dot character is treated as invalid.
 * @param pinsSha256 a set of pins. Each pin is the SHA-256 cryptographic hash of the
 * DER-encoded ASN.1 representation of the Subject Public Key Info (SPKI) of the host's
 * X.509 certificate. Use {@link java.security.cert.Certificate#getPublicKey()
 * Certificate.getPublicKey()} and {@link java.security.Key#getEncoded() Key.getEncoded()}
 * to obtain DER-encoded ASN.1 representation of the SPKI. Although, the method does not
 * mandate the presence of the backup pin that can be used if the control of the primary
 * private key has been lost, it is highly recommended to supply one.
 * @param includeSubdomains indicates whether the pinning policy should be applied to
 *                          subdomains of {@code hostName}.
 * @param expirationInstant specifies the expiration instant for the pins.
 * @return the builder to facilitate chaining.
 * @throws java.lang.NullPointerException if any of the input parameters are {@code null}.
 * @throws java.lang.IllegalArgumentException if the given host name is invalid or {@code pinsSha256}
 * contains a byte array that does not represent a valid SHA-256 hash.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder addPublicKeyPins(@android.annotation.NonNull java.lang.String hostName, @android.annotation.NonNull java.util.Set<byte[]> pinsSha256, boolean includeSubdomains, @android.annotation.NonNull java.time.Instant expirationInstant) { throw new RuntimeException("Stub!"); }

/**
 * Adds hint that {@code host} supports QUIC.
 * Note that {@link #setEnableHttpCache enableHttpCache}
 * ({@link #HTTP_CACHE_DISK}) is needed to take advantage of 0-RTT
 * connection establishment between sessions.
 *
 * @param host hostname of the server that supports QUIC.
 * @param port host of the server that supports QUIC.
 * @param alternatePort alternate port to use for QUIC.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder addQuicHint(@android.annotation.NonNull java.lang.String host, int port, int alternatePort) { throw new RuntimeException("Stub!"); }

/**
 * Build a {@link android.net.http.HttpEngine HttpEngine} using this builder's configuration.
 *
 * @return constructed {@link android.net.http.HttpEngine HttpEngine}.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine build() { throw new RuntimeException("Stub!"); }

/**
 * Constructs a default User-Agent string including the system build version, model and id,
 * and the HTTP stack version.
 *
 * @return User-Agent string.
 */

@android.annotation.NonNull
public java.lang.String getDefaultUserAgent() { throw new RuntimeException("Stub!"); }

/**
 * Configures the behavior of connection migration. For more details, see documentation
 * of {@link android.net.http.ConnectionMigrationOptions ConnectionMigrationOptions} and the individual methods of {@link android.net.http.ConnectionMigrationOptions.Builder ConnectionMigrationOptions.Builder}.
 *
 * <p>Only relevant if {@link #setEnableQuic(boolean)} is enabled.
 *
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setConnectionMigrationOptions(@android.annotation.NonNull android.net.http.ConnectionMigrationOptions connectionMigrationOptions) { throw new RuntimeException("Stub!"); }

/**
 * Configures the behavior of hostname lookup. For more details, see documentation
 * of {@link android.net.http.DnsOptions DnsOptions} and the individual methods of {@link android.net.http.DnsOptions.Builder DnsOptions.Builder}.
 *
 * <p>Only relevant if {@link #setEnableQuic(boolean)} is enabled.
 *
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setDnsOptions(@android.annotation.NonNull android.net.http.DnsOptions dnsOptions) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether <a href="https://tools.ietf.org/html/rfc7932">Brotli</a> compression is
 * enabled. If enabled, Brotli will be advertised in Accept-Encoding request headers.
 * Defaults to disabled.
 *
 * @param value {@code true} to enable Brotli, {@code false} to disable.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setEnableBrotli(boolean value) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether <a href="https://tools.ietf.org/html/rfc7540">HTTP/2</a> protocol is
 * enabled. Defaults to enabled.
 *
 * @param value {@code true} to enable HTTP/2, {@code false} to disable.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setEnableHttp2(boolean value) { throw new RuntimeException("Stub!"); }

/**
 * Enables or disables caching of HTTP data and other information like QUIC server
 * information.
 *
 * @param cacheMode control location and type of cached data. Must be one of {@link
 * #HTTP_CACHE_DISABLED HTTP_CACHE_*}.
 * @param maxSize maximum size in bytes used to cache data (advisory and maybe exceeded at
 * times).
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setEnableHttpCache(int cacheMode, long maxSize) { throw new RuntimeException("Stub!"); }

/**
 * Enables or disables public key pinning bypass for local trust anchors. Disabling the
 * bypass for local trust anchors is highly discouraged since it may prohibit the app from
 * communicating with the pinned hosts. E.g., a user may want to send all traffic through an
 * SSL enabled proxy by changing the device proxy settings and adding the proxy certificate
 * to the list of local trust anchor. Disabling the bypass will most likely prevent the app
 * from sending any traffic to the pinned hosts. For more information see 'How does key
 * pinning interact with local proxies and filters?' at
 * https://www.chromium.org/Home/chromium-security/security-faq
 *
 * @param value {@code true} to enable the bypass, {@code false} to disable.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setEnablePublicKeyPinningBypassForLocalTrustAnchors(boolean value) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether <a href="https://www.chromium.org/quic">QUIC</a> protocol
 * is enabled. Defaults to enabled.
 *
 * @param value {@code true} to enable QUIC, {@code false} to disable.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setEnableQuic(boolean value) { throw new RuntimeException("Stub!"); }

/**
 * Configures the behavior of the HTTP stack when using QUIC. For more details, see
 * documentation of {@link android.net.http.QuicOptions QuicOptions} and the individual methods
 * of {@link android.net.http.QuicOptions.Builder QuicOptions.Builder}.
 *
 * <p>Only relevant if {@link #setEnableQuic(boolean)} is enabled.
 *
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setQuicOptions(@android.annotation.NonNull android.net.http.QuicOptions quicOptions) { throw new RuntimeException("Stub!"); }

/**
 * Sets directory for HTTP Cache and Cookie Storage. The directory must
 * exist.
 * <p>
 * <b>NOTE:</b> Do not use the same storage directory with more than one
 * {@link android.net.http.HttpEngine HttpEngine} at a time. Access to the storage directory does
 * not support concurrent access by multiple {@link android.net.http.HttpEngine HttpEngine} instances.
 *
 * @param value path to existing directory.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setStoragePath(@android.annotation.NonNull java.lang.String value) { throw new RuntimeException("Stub!"); }

/**
 * Overrides the User-Agent header for all requests. An explicitly set User-Agent header
 * (set using {@link android.net.http.UrlRequest.Builder#addHeader UrlRequest.Builder.addHeader}) will override a value set using this
 * function.
 *
 * @param userAgent the User-Agent string to use for all requests.
 * @return the builder to facilitate chaining.
 */

@android.annotation.NonNull
public android.net.http.HttpEngine.Builder setUserAgent(@android.annotation.NonNull java.lang.String userAgent) { throw new RuntimeException("Stub!"); }

/**
 * Setting to disable HTTP cache. Some data may still be temporarily stored in memory.
 * Passed to {@link #setEnableHttpCache}.
 */

public static final int HTTP_CACHE_DISABLED = 0; // 0x0

/**
 * Setting to enable on-disk cache, including HTTP data.
 * {@link #setStoragePath} must be called prior to passing this constant to
 * {@link #setEnableHttpCache}.
 */

public static final int HTTP_CACHE_DISK = 3; // 0x3

/**
 * Setting to enable on-disk cache, excluding HTTP data.
 * {@link #setStoragePath} must be called prior to passing this constant to
 * {@link #setEnableHttpCache}.
 */

public static final int HTTP_CACHE_DISK_NO_HTTP = 2; // 0x2

/**
 * Setting to enable in-memory HTTP cache, including HTTP data.
 * Passed to {@link #setEnableHttpCache}.
 */

public static final int HTTP_CACHE_IN_MEMORY = 1; // 0x1
}

}

