/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.net.vcn;

import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.ipsec.ike.IkeTunnelConnectionParams;
import java.util.List;
import java.util.Set;

/**
 * This class represents a configuration for a connection to a Virtual Carrier Network gateway.
 *
 * <p>Each VcnGatewayConnectionConfig represents a single logical connection to a carrier gateway,
 * and may provide one or more telephony services (as represented by network capabilities). Each
 * gateway is expected to provide mobility for a given session as the device roams across {@link android.net.Network Network}s.
 *
 * <p>A VCN connection based on this configuration will be brought up dynamically based on device
 * settings, and filed NetworkRequests. Underlying Networks must provide INTERNET connectivity, and
 * must be part of the subscription group under which this configuration is registered (see {@link android.net.vcn.VcnManager#setVcnConfig  }).
 *
 * <p>As an abstraction of a cellular network, services that can be provided by a VCN network are
 * limited to services provided by cellular networks:
 *
 * <ul>
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS NetworkCapabilities.NET_CAPABILITY_MMS}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL NetworkCapabilities.NET_CAPABILITY_SUPL}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN NetworkCapabilities.NET_CAPABILITY_DUN}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA NetworkCapabilities.NET_CAPABILITY_FOTA}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS NetworkCapabilities.NET_CAPABILITY_IMS}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS NetworkCapabilities.NET_CAPABILITY_CBS}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_IA NetworkCapabilities.NET_CAPABILITY_IA}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS NetworkCapabilities.NET_CAPABILITY_RCS}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP NetworkCapabilities.NET_CAPABILITY_XCAP}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS NetworkCapabilities.NET_CAPABILITY_EIMS}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET NetworkCapabilities.NET_CAPABILITY_INTERNET}
 *   <li>{@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX NetworkCapabilities.NET_CAPABILITY_MCX}
 * </ul>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class VcnGatewayConnectionConfig {

VcnGatewayConnectionConfig() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param other This value may be {@code null}.
 */

public boolean equals(@android.annotation.Nullable java.lang.Object other) { throw new RuntimeException("Stub!"); }

/**
 * Returns all exposed capabilities.
 *
 * <p>The returned integer-value capabilities will not contain duplicates, and will be sorted in
 * ascending numerical order.
 *
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#addExposedCapability(int)
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#removeExposedCapability(int)

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public int[] getExposedCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Returns the configured Gateway Connection name.
 *
 * <p>This name is used by the configuring apps to distinguish between
 * VcnGatewayConnectionConfigs configured on a single {@link android.net.vcn.VcnConfig VcnConfig}. This will be used as
 * the identifier in VcnStatusCallback invocations.
 *
 * @see android.net.vcn.VcnManager.VcnStatusCallback#onGatewayConnectionError

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.lang.String getGatewayConnectionName() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the maximum MTU allowed for this Gateway Connection.
 *
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#setMaxMtu(int)

 * @return Value is MIN_MTU_V6 or greater
 */

public int getMaxMtu() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the maximum supported IKEv2/IPsec NATT keepalive timeout.
 *
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#setMinUdpPort4500NatTimeoutSeconds(int)
 */

public int getMinUdpPort4500NatTimeoutSeconds() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the configured retry intervals.
 *
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#setRetryIntervalsMillis(long[])

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public long[] getRetryIntervalsMillis() { throw new RuntimeException("Stub!"); }

/**
 * Retrieve the VcnUnderlyingNetworkTemplate list, or a default list if it is not configured.
 *
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#setVcnUnderlyingNetworkPriorities(List)

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.net.vcn.VcnUnderlyingNetworkTemplate> getVcnUnderlyingNetworkPriorities() { throw new RuntimeException("Stub!"); }

/**
 * Checks if the given VCN gateway option is enabled.
 *
 * @param option the option to check.
 * Value is {@link android.net.vcn.VcnGatewayConnectionConfig#VCN_GATEWAY_OPTION_ENABLE_DATA_STALL_RECOVERY_WITH_MOBILITY}
 * @throws java.lang.IllegalArgumentException if the provided option is invalid.
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#addGatewayOption(int)
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#removeGatewayOption(int)
 */

public boolean hasGatewayOption(int option) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Check whether safe mode is enabled
 *
 * @see android.net.vcn.VcnGatewayConnectionConfig.Builder#setSafeModeEnabled(boolean)
 */

public boolean isSafeModeEnabled() { throw new RuntimeException("Stub!"); }

/**
 * Minimum NAT timeout not set.
 *
 * <p>When the timeout is not set, the device will automatically choose a keepalive interval and
 * may reduce the keepalive frequency for power-optimization.
 */

public static final int MIN_UDP_PORT_4500_NAT_TIMEOUT_UNSET = -1; // 0xffffffff

/**
 * Perform mobility update to attempt recovery from suspected data stalls.
 *
 * <p>If set, the gateway connection will monitor the data stall detection of the VCN network.
 * When there is a suspected data stall, the gateway connection will attempt recovery by
 * performing a mobility update on the underlying IKE session.
 */

public static final int VCN_GATEWAY_OPTION_ENABLE_DATA_STALL_RECOVERY_WITH_MOBILITY = 0; // 0x0
/**
 * This class is used to incrementally build {@link android.net.vcn.VcnGatewayConnectionConfig VcnGatewayConnectionConfig} objects.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

/**
 * Construct a Builder object.
 *
 * @param gatewayConnectionName the String GatewayConnection name for this
 *     VcnGatewayConnectionConfig. Each VcnGatewayConnectionConfig within a {@link android.net.vcn.VcnConfig VcnConfig} must be given a unique name. This name is used by the caller to
 *     distinguish between VcnGatewayConnectionConfigs configured on a single {@link android.net.vcn.VcnConfig VcnConfig}. This will be used as the identifier in VcnStatusCallback invocations.
 * This value cannot be {@code null}.
 * @param tunnelConnectionParams the IKE tunnel connection configuration
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided IkeTunnelConnectionParams is not
 *     configured to support MOBIKE
 * @see android.net.ipsec.ike.IkeTunnelConnectionParams
 * @see android.net.vcn.VcnManager.VcnStatusCallback#onGatewayConnectionError
 */

public Builder(@android.annotation.NonNull java.lang.String gatewayConnectionName, @android.annotation.NonNull android.net.ipsec.ike.IkeTunnelConnectionParams tunnelConnectionParams) { throw new RuntimeException("Stub!"); }

/**
 * Add a capability that this VCN Gateway Connection will support.
 *
 * @param exposedCapability the app-facing capability to be exposed by this VCN Gateway
 *     Connection (i.e., the capabilities that this VCN Gateway Connection will support).
 * Value is {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET}, or {@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX}
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 * This value cannot be {@code null}.
 * @see android.net.vcn.VcnGatewayConnectionConfig for a list of capabilities may be exposed by a Gateway
 *     Connection
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder addExposedCapability(int exposedCapability) { throw new RuntimeException("Stub!"); }

/**
 * Enables the specified VCN gateway option.
 *
 * @param option the option to be enabled
 * Value is {@link android.net.vcn.VcnGatewayConnectionConfig#VCN_GATEWAY_OPTION_ENABLE_DATA_STALL_RECOVERY_WITH_MOBILITY}
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided option is invalid
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder addGatewayOption(int option) { throw new RuntimeException("Stub!"); }

/**
 * Builds and validates the VcnGatewayConnectionConfig.
 *
 * @return an immutable VcnGatewayConnectionConfig instance

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig build() { throw new RuntimeException("Stub!"); }

/**
 * Remove a capability that this VCN Gateway Connection will support.
 *
 * @param exposedCapability the app-facing capability to not be exposed by this VCN Gateway
 *     Connection (i.e., the capabilities that this VCN Gateway Connection will support)
 * Value is {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET}, or {@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX}
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 * This value cannot be {@code null}.
 * @see android.net.vcn.VcnGatewayConnectionConfig for a list of capabilities may be exposed by a Gateway
 *     Connection
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder removeExposedCapability(int exposedCapability) { throw new RuntimeException("Stub!"); }

/**
 * Resets (disables) the specified VCN gateway option.
 *
 * @param option the option to be disabled
 * Value is {@link android.net.vcn.VcnGatewayConnectionConfig#VCN_GATEWAY_OPTION_ENABLE_DATA_STALL_RECOVERY_WITH_MOBILITY}
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided option is invalid
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder removeGatewayOption(int option) { throw new RuntimeException("Stub!"); }

/**
 * Sets the maximum MTU allowed for this VCN Gateway Connection.
 *
 * <p>This MTU is applied to the VCN Gateway Connection exposed Networks, and represents the
 * MTU of the virtualized network.
 *
 * <p>The system may reduce the MTU below the maximum specified based on signals such as the
 * MTU of the underlying networks (and adjusted for Gateway Connection overhead).
 *
 * @param maxMtu the maximum MTU allowed for this Gateway Connection. Must be greater than
 *     the IPv6 minimum MTU of 1280. Defaults to 1500.
 * Value is MIN_MTU_V6 or greater
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder setMaxMtu(int maxMtu) { throw new RuntimeException("Stub!"); }

/**
 * Sets the maximum supported IKEv2/IPsec NATT keepalive timeout.
 *
 * <p>This is used as a power-optimization hint for other IKEv2/IPsec use cases (e.g. VPNs,
 * or IWLAN) to reduce the necessary keepalive frequency, thus conserving power and data.
 *
 * @param minUdpPort4500NatTimeoutSeconds the maximum keepalive timeout supported by the VCN
 *     Gateway Connection, generally the minimum duration a NAT mapping is cached on the VCN
 *     Gateway; or {@link MIN_UDP_PORT_4500_NAT_TIMEOUT_UNSET} to clear this value.
 * Value is MIN_UDP_PORT_4500_NAT_TIMEOUT_SECONDS or greater
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder setMinUdpPort4500NatTimeoutSeconds(int minUdpPort4500NatTimeoutSeconds) { throw new RuntimeException("Stub!"); }

/**
 * Set the retry interval between VCN establishment attempts upon successive failures.
 *
 * <p>The last retry interval will be repeated until safe mode is entered, or a connection
 * is successfully established, at which point the retry timers will be reset. For power
 * reasons, the last (repeated) retry interval MUST be at least 15 minutes.
 *
 * <p>Retry intervals MAY be subject to system power saving modes. That is to say that if
 * the system enters a power saving mode, the retry may not occur until the device leaves
 * the specified power saving mode. Intervals are sequential, and intervals will NOT be
 * skipped if system power saving results in delaying retries (even if it exceed multiple
 * retry intervals).
 *
 * <p>Each Gateway Connection will retry according to the retry intervals configured, but if
 * safe mode is enabled, all Gateway Connection(s) will be disabled.
 *
 * @param retryIntervalsMs an array of between 1 and 10 millisecond intervals after which
 *     the VCN will attempt to retry a session initiation. The last (repeating) retry
 *     interval must be at least 15 minutes. Defaults to: {@code [1s, 2s, 5s, 30s, 1m, 5m,
 *     15m]}
 * This value cannot be {@code null}.
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 * This value cannot be {@code null}.
 * @see android.net.vcn.VcnManager for additional discussion on fail-safe mode
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder setRetryIntervalsMillis(@android.annotation.NonNull long[] retryIntervalsMs) { throw new RuntimeException("Stub!"); }

/**
 * Enable/disable safe mode
 *
 * <p>If a VCN fails to provide connectivity within a system-provided timeout, it will enter
 * safe mode. In safe mode, the VCN Network will be torn down and the system will restore
 * connectivity by allowing underlying cellular or WiFi networks to be used as default. At
 * the same time, VCN will continue to retry until it succeeds.
 *
 * <p>When safe mode is disabled and VCN connection fails to provide connectivity, end users
 * might not have connectivity, and may not have access to carrier-owned underlying
 * networks.
 *
 * @param enabled whether safe mode should be enabled. Defaults to {@code true}

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder setSafeModeEnabled(boolean enabled) { throw new RuntimeException("Stub!"); }

/**
 * Set the list of templates to match underlying networks against, in high-to-low priority
 * order.
 *
 * <p>To select the VCN underlying network, the VCN connection will go through all the
 * network candidates and return a network matching the highest priority rule.
 *
 * <p>If multiple networks match the same rule, the VCN will prefer an already-selected
 * network as opposed to a new/unselected network. However, if both are new/unselected
 * networks, a network will be chosen arbitrarily amongst the networks matching the highest
 * priority rule.
 *
 * <p>If all networks fail to match the rules provided, a carrier-owned underlying network
 * will still be selected (if available, at random if necessary).
 *
 * @param underlyingNetworkTemplates a list of unique VcnUnderlyingNetworkTemplates that are
 *     ordered from most to least preferred, or an empty list to use the default
 *     prioritization. The default network prioritization order is Opportunistic cellular,
 *     Carrier WiFi and then Macro cellular.
 * This value cannot be {@code null}.
 * @return this {@link android.net.vcn.VcnGatewayConnectionConfig.Builder Builder} instance, for chaining
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.vcn.VcnGatewayConnectionConfig.Builder setVcnUnderlyingNetworkPriorities(@android.annotation.NonNull java.util.List<android.net.vcn.VcnUnderlyingNetworkTemplate> underlyingNetworkTemplates) { throw new RuntimeException("Stub!"); }
}

}

