/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.ranging;

import android.Manifest;
import android.os.CancellationSignal;
import android.os.RemoteException;
import android.ranging.raw.RawResponderRangingConfig;
import java.util.concurrent.Executor;

/**
 * Represents a session for performing ranging operations. A {@link android.ranging.RangingSession RangingSession} manages
 * the lifecycle of a ranging operation, including start, stop, and event callbacks.
 *
 * <p>All methods are asynchronous and rely on the provided {@link java.util.concurrent.Executor Executor} to invoke
 * callbacks on appropriate threads.
 *
 * <p>This class implements {@link java.lang.AutoCloseable AutoCloseable}, ensuring that resources can be
 * automatically released when the session is closed.
 *
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class RangingSession implements java.lang.AutoCloseable {

RangingSession() { throw new RuntimeException("Stub!"); }

/**
 * Adds a new device to an ongoing ranging session.
 * <p>
 * This method allows for adding a new device to an active ranging session using raw ranging
 * parameters. Only devices represented by {@link android.ranging.raw.RawResponderRangingConfig RawResponderRangingConfig} is supported.
 * If the provided {@link android.ranging.RangingConfig RangingConfig} does not match one of these types, the addition fails
 * and invokes {@link android.ranging.RangingSession.Callback#onOpenFailed(int) Callback.onOpenFailed(int)} with a reason of
 * {@link android.ranging.RangingSession.Callback#REASON_UNSUPPORTED Callback.REASON_UNSUPPORTED}.
 * </p>
 *
 * <br>
 * Requires android.Manifest.permission.RANGING
 * @param deviceRangingParams the ranging parameters for the device to be added,
 *                            which must be an instance of {@link android.ranging.raw.RawResponderRangingConfig RawResponderRangingConfig}
 *
 * This value cannot be {@code null}.
 * @apiNote If the underlying ranging technology cannot support this dynamic addition, failure
 * will be indicated via {@code Callback#onStartFailed(REASON_UNSUPPORTED, RangingDevice)}
 *
 */

public void addDeviceToRangingSession(@android.annotation.NonNull android.ranging.RangingConfig deviceRangingParams) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * Requires android.Manifest.permission.RANGING
 */

public void close() { throw new RuntimeException("Stub!"); }

/**
 * Reconfigures the ranging interval for the current session by setting the interval
 * skip count. The {@code intervalSkipCount} defines how many intervals should be skipped
 * between successive ranging rounds. Valid values range from 0 to 255.
 *
 * <br>
 * Requires android.Manifest.permission.RANGING
 * @param intervalSkipCount the number of intervals to skip, ranging from 0 to 255.

 * Value is between 0 and 255 inclusive
 */

public void reconfigureRangingInterval(int intervalSkipCount) { throw new RuntimeException("Stub!"); }

/**
 * Removes a specific device from an ongoing ranging session.
 * <p>
 * This method removes a specified device from the active ranging session, stopping
 * further ranging operations for that device. The operation is handled by the system
 * server and may throw a {@link android.os.RemoteException RemoteException} in case of server-side communication
 * issues.
 * </p>
 *
 * <br>
 * Requires android.Manifest.permission.RANGING
 * @param rangingDevice the device to be removed from the session.
 * This value cannot be {@code null}.
 * @apiNote Currently, this API is supported only for UWB multicast session if using
 * {@link android.ranging.RangingConfig#RANGING_SESSION_RAW RangingConfig.RANGING_SESSION_RAW}.
 *
 */

public void removeDeviceFromRangingSession(@android.annotation.NonNull android.ranging.RangingDevice rangingDevice) { throw new RuntimeException("Stub!"); }

/**
 * Starts the ranging session with the provided ranging preferences.
 * <p>The {@link android.ranging.RangingSession.Callback#onOpened() Callback.onOpened()} will be called when the session finishes starting.
 *
 * <p>The provided {@link android.ranging.RangingPreference RangingPreference} determines the configuration for the session.
 * A {@link android.os.CancellationSignal CancellationSignal} is returned to allow the caller to cancel the session
 * if needed. If the session is canceled, the {@link #close()} method will be invoked
 * automatically to release resources.
 *
 * <br>
 * Requires android.Manifest.permission.RANGING
 * @param rangingPreference {@link android.ranging.RangingPreference RangingPreference} the preferences for configuring the
 *                          ranging session.
 * This value cannot be {@code null}.
 * @return a {@link android.os.CancellationSignal CancellationSignal} to close the session.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.os.CancellationSignal start(@android.annotation.NonNull android.ranging.RangingPreference rangingPreference) { throw new RuntimeException("Stub!"); }

/**
 * Stops the ranging session.
 *
 * <p>This method releases any ongoing ranging operations. If the operation fails,
 * it will propagate a {@link android.os.RemoteException RemoteException} from the system server.

 * <br>
 * Requires android.Manifest.permission.RANGING
 */

public void stop() { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }
/**
 * Callback interface for receiving ranging session events.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface Callback {

/**
 * Called when the ranging session has closed.
 *
 * @param reason the reason why the session was closed, limited to values
 *               defined by {@link android.ranging.RangingSession.Callback.Reason Reason}.

 * Value is {@link android.ranging.RangingSession.Callback#REASON_UNKNOWN}, {@link android.ranging.RangingSession.Callback#REASON_LOCAL_REQUEST}, {@link android.ranging.RangingSession.Callback#REASON_REMOTE_REQUEST}, {@link android.ranging.RangingSession.Callback#REASON_UNSUPPORTED}, {@link android.ranging.RangingSession.Callback#REASON_SYSTEM_POLICY}, or {@link android.ranging.RangingSession.Callback#REASON_NO_PEERS_FOUND}
 */

public void onClosed(int reason);

/**
 * Called when the ranging session failed to open.
 *
 * @param reason the reason for the failure, limited to values defined by
 *               {@link android.ranging.RangingSession.Callback.Reason Reason}.

 * Value is {@link android.ranging.RangingSession.Callback#REASON_UNKNOWN}, {@link android.ranging.RangingSession.Callback#REASON_LOCAL_REQUEST}, {@link android.ranging.RangingSession.Callback#REASON_REMOTE_REQUEST}, {@link android.ranging.RangingSession.Callback#REASON_UNSUPPORTED}, {@link android.ranging.RangingSession.Callback#REASON_SYSTEM_POLICY}, or {@link android.ranging.RangingSession.Callback#REASON_NO_PEERS_FOUND}
 */

public void onOpenFailed(int reason);

/**
 * Called when the ranging session opens successfully.
 */

public void onOpened();

/**
 * Called when ranging data has been received from a peer.
 *
 * @param peer {@link android.ranging.RangingDevice RangingDevice} the peer from which ranging data was received.
 * This value cannot be {@code null}.
 * @param data {@link android.ranging.RangingData RangingData} the received.

 * This value cannot be {@code null}.
 */

public void onResults(@android.annotation.NonNull android.ranging.RangingDevice peer, @android.annotation.NonNull android.ranging.RangingData data);

/**
 * Called when ranging has started with a particular peer using a particular technology
 * during an ongoing session.
 *
 * @param peer       {@link android.ranging.RangingDevice RangingDevice} the peer with which ranging has started.
 * This value cannot be {@code null}.
 * @param technology {@link android.ranging.RangingManager.RangingTechnology}
 *                   the ranging technology that started.

 * Value is {@link android.ranging.RangingManager#UWB}, {@link android.ranging.RangingManager#BLE_CS}, {@link android.ranging.RangingManager#WIFI_NAN_RTT}, or {@link android.ranging.RangingManager#BLE_RSSI}
 */

public void onStarted(@android.annotation.NonNull android.ranging.RangingDevice peer, int technology);

/**
 * Called when ranging has stopped with a particular peer using a particular technology
 * during an ongoing session.
 *
 * @param peer       {@link android.ranging.RangingDevice RangingDevice} the peer with which ranging has stopped.
 * This value cannot be {@code null}.
 * @param technology {@link android.ranging.RangingManager.RangingTechnology}
 *                   the ranging technology that stopped.

 * Value is {@link android.ranging.RangingManager#UWB}, {@link android.ranging.RangingManager#BLE_CS}, {@link android.ranging.RangingManager#WIFI_NAN_RTT}, or {@link android.ranging.RangingManager#BLE_RSSI}
 */

public void onStopped(@android.annotation.NonNull android.ranging.RangingDevice peer, int technology);

/**
 * Indicates that the session was closed because {@link java.lang.AutoCloseable#close() AutoCloseable.close()} or
 * {@link android.ranging.RangingSession#stop() RangingSession.stop()} was called.
 */

public static final int REASON_LOCAL_REQUEST = 1; // 0x1

/**
 * Indicates that the session was closed because none of the specified peers were found.
 */

public static final int REASON_NO_PEERS_FOUND = 5; // 0x5

/**
 * Indicates that the session was closed at the request of a remote peer.
 */

public static final int REASON_REMOTE_REQUEST = 2; // 0x2

/**
 * Indicates that the local system policy forced the session to close, such
 * as power management policy, airplane mode etc.
 */

public static final int REASON_SYSTEM_POLICY = 4; // 0x4

/**
 * Indicates that the session was closed due to an unknown reason.
 */

public static final int REASON_UNKNOWN = 0; // 0x0

/**
 * Indicates that the session closed because the provided session parameters were not
 * supported.
 */

public static final int REASON_UNSUPPORTED = 3; // 0x3
}

}

