/*
 * Copyright (c) 1996, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.security;

import java.util.*;
import java.util.regex.*;

/**
 * This class provides a cryptographically strong random number
 * generator (RNG).
 *
 * <p>A cryptographically strong random number minimally complies with the
 * statistical random number generator tests specified in
 * <a href="http://nvlpubs.nist.gov/nistpubs/FIPS/NIST.FIPS.140-2.pdf">
 * <i>FIPS 140-2, Security Requirements for Cryptographic Modules</i></a>,
 * section 4.9.1.
 * Additionally, {@code SecureRandom} must produce non-deterministic output.
 * Therefore any seed material passed to a {@code SecureRandom} object must be
 * unpredictable, and all {@code SecureRandom} output sequences must be
 * cryptographically strong, as described in
 * <a href="http://tools.ietf.org/html/rfc4086">
 * <i>RFC 4086: Randomness Requirements for Security</i></a>.
 *
 * <p> Many {@code SecureRandom} implementations are in the form of a
 * pseudo-random number generator (PRNG, also known as deterministic random
 * bits generator or DRBG), which means they use a deterministic algorithm
 * to produce a pseudo-random sequence from a random seed.
 * Other implementations may produce true random numbers,
 * and yet others may use a combination of both techniques.
 *
 * <p>A caller obtains a {@code SecureRandom} instance via the
 * no-argument constructor or one of the {@code getInstance} methods.
 * For example:
 *
 * <blockquote><pre>
 * SecureRandom r1 = new SecureRandom();
 * SecureRandom r2 = SecureRandom.getInstance("NativePRNG");
 * SecureRandom r3 = SecureRandom.getInstance("DRBG",
 *         DrbgParameters.instantiation(128, RESEED_ONLY, null));</pre>
 * </blockquote>
 *
 * <p> The third statement above returns a {@code SecureRandom} object of the
 * specific algorithm supporting the specific instantiate parameters. The
 * implementation's effective instantiated parameters must match this minimum
 * request but is not necessarily the same. For example, even if the request
 * does not require a certain feature, the actual instantiation can provide
 * the feature. An implementation may lazily instantiate a {@code SecureRandom}
 * until it's actually used, but the effective instantiate parameters must be
 * determined right after it's created and {@link #getParameters()} should
 * always return the same result unchanged.
 *
 * <p> Typical callers of {@code SecureRandom} invoke the following methods
 * to retrieve random bytes:
 *
 * <blockquote><pre>
 * SecureRandom random = new SecureRandom();
 * byte[] bytes = new byte[20];
 * random.nextBytes(bytes);</pre>
 * </blockquote>
 *
 * <p> Callers may also invoke the {@link #generateSeed} method
 * to generate a given number of seed bytes (to seed other random number
 * generators, for example):
 *
 * <blockquote><pre>
 * byte[] seed = random.generateSeed(20);</pre>
 * </blockquote>
 *
 * <p> A newly created PRNG {@code SecureRandom} object is not seeded (except
 * if it is created by {@link #SecureRandom(byte[])}). The first call to
 * {@code nextBytes} will force it to seed itself from an implementation-
 * specific entropy source. This self-seeding will not occur if {@code setSeed}
 * was previously called.
 *
 * <p> A {@code SecureRandom} can be reseeded at any time by calling the
 * {@code reseed} or {@code setSeed} method. The {@code reseed} method
 * reads entropy input from its entropy source to reseed itself.
 * The {@code setSeed} method requires the caller to provide the seed.
 *
 * <p> Please note that {@code reseed} may not be supported by all
 * {@code SecureRandom} implementations.
 *
 * <p> Some {@code SecureRandom} implementations may accept a
 * {@link java.security.SecureRandomParameters SecureRandomParameters} parameter in its
 * {@link #nextBytes(byte[],java.security.SecureRandomParameters)} and
 * {@link #reseed(java.security.SecureRandomParameters)} methods to further
 * control the behavior of the methods.
 *
 * <p> Note: Depending on the implementation, the {@code generateSeed},
 * {@code reseed} and {@code nextBytes} methods may block as entropy is being
 * gathered, for example, if the entropy source is /dev/random on various
 * Unix-like operating systems.
 *
 * <h2> Thread safety </h2>
 * {@code SecureRandom} objects are safe for use by multiple concurrent threads.
 *
 * @implSpec
 * A {@code SecureRandom} service provider can advertise that it is thread-safe
 * by setting the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#service-attributes">service
 * provider attribute</a> "ThreadSafe" to "true" when registering the provider.
 * Otherwise, this class will instead synchronize access to the following
 * methods of the {@code SecureRandomSpi} implementation:
 * <ul>
 * <li>{@link java.security.SecureRandomSpi#engineSetSeed(byte[]) SecureRandomSpi.engineSetSeed(byte[])}
 * <li>{@link java.security.SecureRandomSpi#engineNextBytes(byte[]) SecureRandomSpi.engineNextBytes(byte[])}
 * <li>{@link java.security.SecureRandomSpi#engineNextBytes(byte[],java.security.SecureRandomParameters) SecureRandomSpi.engineNextBytes(byte[], SecureRandomParameters)}
 * <li>{@link java.security.SecureRandomSpi#engineGenerateSeed(int) SecureRandomSpi.engineGenerateSeed(int)}
 * <li>{@link java.security.SecureRandomSpi#engineReseed(java.security.SecureRandomParameters) SecureRandomSpi.engineReseed(SecureRandomParameters)}
 * </ul>
 *
 * The SHA1PRNG algorithm from the Crypto provider has been deprecated as it was insecure, and also
 * incorrectly used by some apps as a key derivation function. See
 * <a href="http://android-developers.blogspot.com/2016/06/security-crypto-provider-deprecated-in.html">
 * Security &quot;Crypto&quot; provider deprecated in Android N</a> for details.
 *
 * @see java.security.SecureRandomSpi
 * @see java.util.Random
 *
 * @author Benjamin Renaud
 * @author Josh Bloch
 * @since 1.1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class SecureRandom extends java.util.Random {

/**
 * Constructs a secure random number generator (RNG) implementing the
 * default random number algorithm.
 *
 * <p> This constructor traverses the list of registered security Providers,
 * starting with the most preferred Provider.
 * A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the first
 * Provider that supports a {@code SecureRandom} (RNG) algorithm is returned.
 * If none of the Providers support a RNG algorithm,
 * then an implementation-specific default is returned.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * <p> See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 */

public SecureRandom() { throw new RuntimeException("Stub!"); }

/**
 * Constructs a secure random number generator (RNG) implementing the
 * default random number algorithm.
 * The {@code SecureRandom} instance is seeded with the specified seed bytes.
 *
 * <p> This constructor traverses the list of registered security Providers,
 * starting with the most preferred Provider.
 * A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the first
 * Provider that supports a {@code SecureRandom} (RNG) algorithm is returned.
 * If none of the Providers support a RNG algorithm,
 * then an implementation-specific default is returned.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * <p> See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @param seed the seed.
 */

public SecureRandom(byte[] seed) { throw new RuntimeException("Stub!"); }

/**
 * Creates a {@code SecureRandom} object.
 *
 * @param secureRandomSpi the {@code SecureRandom} implementation.
 * @param provider the provider.
 */

protected SecureRandom(java.security.SecureRandomSpi secureRandomSpi, java.security.Provider provider) { throw new RuntimeException("Stub!"); }

/**
 * Returns the given number of seed bytes, computed using the seed
 * generation algorithm that this class uses to seed itself.  This
 * call may be used to seed other random number generators.
 *
 * @param numBytes the number of seed bytes to generate.
 * @throws java.lang.IllegalArgumentException if {@code numBytes} is negative
 * @return the seed bytes.
 */

public byte[] generateSeed(int numBytes) { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of the algorithm implemented by this
 * {@code SecureRandom} object.
 *
 * @return the name of the algorithm or {@code unknown}
 *          if the algorithm name cannot be determined.
 * @since 1.5
 */

public java.lang.String getAlgorithm() { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object that implements the specified
 * Random Number Generator (RNG) algorithm.
 *
 * <p> This method traverses the list of registered security Providers,
 * starting with the most preferred Provider.
 * A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the first
 * Provider that supports the specified algorithm is returned.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * @implNote
 * The JDK Reference Implementation additionally uses the
 * {@code jdk.security.provider.preferred}
 * {@link java.security.Security#getProperty(java.lang.String) Security} property to determine
 * the preferred provider order for the specified algorithm. This
 * may be different than the order of providers returned by
 * {@link java.security.Security#getProviders() Security.getProviders()}.
 *
 * @param algorithm the name of the RNG algorithm.
 * See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @return the new {@code SecureRandom} object
 *
 * @throws java.security.NoSuchAlgorithmException if no {@code Provider} supports a
 *         {@code SecureRandomSpi} implementation for the
 *         specified algorithm
 *
 * @throws java.lang.NullPointerException if {@code algorithm} is {@code null}
 *
 * @see java.security.Provider
 *
 * @since 1.2
 */

public static java.security.SecureRandom getInstance(java.lang.String algorithm) throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object that implements the specified
 * Random Number Generator (RNG) algorithm.
 *
 * <p> A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the specified provider
 * is returned.  The specified provider must be registered
 * in the security provider list.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * @param algorithm the name of the RNG algorithm.
 * See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @param provider the name of the provider.
 *
 * @return the new {@code SecureRandom} object
 *
 * @throws java.lang.IllegalArgumentException if the provider name is {@code null}
 *         or empty
 *
 * @throws java.security.NoSuchAlgorithmException if a {@code SecureRandomSpi}
 *         implementation for the specified algorithm is not
 *         available from the specified provider
 *
 * @throws java.security.NoSuchProviderException if the specified provider is not
 *         registered in the security provider list
 *
 * @throws java.lang.NullPointerException if {@code algorithm} is {@code null}
 *
 * @see java.security.Provider
 *
 * @since 1.2
 */

public static java.security.SecureRandom getInstance(java.lang.String algorithm, java.lang.String provider) throws java.security.NoSuchAlgorithmException, java.security.NoSuchProviderException { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object that implements the specified
 * Random Number Generator (RNG) algorithm.
 *
 * <p> A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the specified {@code Provider}
 * object is returned.  Note that the specified {@code Provider} object
 * does not have to be registered in the provider list.
 *
 * @param algorithm the name of the RNG algorithm.
 * See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @param provider the provider.
 *
 * @return the new {@code SecureRandom} object
 *
 * @throws java.lang.IllegalArgumentException if the specified provider is
 *         {@code null}
 *
 * @throws java.security.NoSuchAlgorithmException if a {@code SecureRandomSpi}
 *         implementation for the specified algorithm is not available
 *         from the specified {@code Provider} object
 *
 * @throws java.lang.NullPointerException if {@code algorithm} is {@code null}
 *
 * @see java.security.Provider
 *
 * @since 1.4
 */

public static java.security.SecureRandom getInstance(java.lang.String algorithm, java.security.Provider provider) throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object that implements the specified
 * Random Number Generator (RNG) algorithm and supports the specified
 * {@code SecureRandomParameters} request.
 *
 * <p> This method traverses the list of registered security Providers,
 * starting with the most preferred Provider.
 * A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the first
 * Provider that supports the specified algorithm and the specified
 * {@code SecureRandomParameters} is returned.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * @implNote
 * The JDK Reference Implementation additionally uses the
 * {@code jdk.security.provider.preferred} property to determine
 * the preferred provider order for the specified algorithm. This
 * may be different than the order of providers returned by
 * {@link java.security.Security#getProviders() Security.getProviders()}.
 *
 * @param algorithm the name of the RNG algorithm.
 * See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @param params the {@code SecureRandomParameters}
 *               the newly created {@code SecureRandom} object must support.
 *
 * @return the new {@code SecureRandom} object
 *
 * @throws java.lang.IllegalArgumentException if the specified params is
 *         {@code null}
 *
 * @throws java.security.NoSuchAlgorithmException if no Provider supports a
 *         {@code SecureRandomSpi} implementation for the specified
 *         algorithm and parameters
 *
 * @throws java.lang.NullPointerException if {@code algorithm} is {@code null}
 *
 * @see java.security.Provider
 *
 * @since 9
 */

public static java.security.SecureRandom getInstance(java.lang.String algorithm, java.security.SecureRandomParameters params) throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object that implements the specified
 * Random Number Generator (RNG) algorithm and supports the specified
 * {@code SecureRandomParameters} request.
 *
 * <p> A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the specified provider
 * is returned.  The specified provider must be registered
 * in the security provider list.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link java.security.Security#getProviders() Security.getProviders()} method.
 *
 * @param algorithm the name of the RNG algorithm.
 * See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @param params the {@code SecureRandomParameters}
 *               the newly created {@code SecureRandom} object must support.
 *
 * @param provider the name of the provider.
 *
 * @return the new {@code SecureRandom} object
 *
 * @throws java.lang.IllegalArgumentException if the provider name is {@code null}
 *         or empty, or params is {@code null}
 *
 * @throws java.security.NoSuchAlgorithmException if the specified provider does not
 *         support a {@code SecureRandomSpi} implementation for the
 *         specified algorithm and parameters
 *
 * @throws java.security.NoSuchProviderException if the specified provider is not
 *         registered in the security provider list
 *
 * @throws java.lang.NullPointerException if {@code algorithm} is {@code null}
 *
 * @see java.security.Provider
 *
 * @since 9
 */

public static java.security.SecureRandom getInstance(java.lang.String algorithm, java.security.SecureRandomParameters params, java.lang.String provider) throws java.security.NoSuchAlgorithmException, java.security.NoSuchProviderException { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object that implements the specified
 * Random Number Generator (RNG) algorithm and supports the specified
 * {@code SecureRandomParameters} request.
 *
 * <p> A new {@code SecureRandom} object encapsulating the
 * {@code SecureRandomSpi} implementation from the specified
 * {@code Provider} object is returned.  Note that the specified
 * {@code Provider} object does not have to be registered in the
 * provider list.
 *
 * @param algorithm the name of the RNG algorithm.
 * See the {@code SecureRandom} section in the <a href=
 * "https://docs.oracle.com/en/java/javase/17/docs/specs/security/standard-names.html#securerandom-number-generation-algorithms">
 * Java Security Standard Algorithm Names Specification</a>
 * for information about standard RNG algorithm names.
 *
 * @param params the {@code SecureRandomParameters}
 *               the newly created {@code SecureRandom} object must support.
 *
 * @param provider the provider.
 *
 * @return the new {@code SecureRandom} object
 *
 * @throws java.lang.IllegalArgumentException if the specified provider or params
 *         is {@code null}
 *
 * @throws java.security.NoSuchAlgorithmException if the specified provider does not
 *         support a {@code SecureRandomSpi} implementation for the
 *         specified algorithm and parameters
 *
 * @throws java.lang.NullPointerException if {@code algorithm} is {@code null}
 *
 * @see java.security.Provider
 *
 * @since 9
 */

public static java.security.SecureRandom getInstance(java.lang.String algorithm, java.security.SecureRandomParameters params, java.security.Provider provider) throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code SecureRandom} object.
 *
 * In Android this is equivalent to get a SHA1PRNG from AndroidOpenSSL.
 *
 * Some situations require strong random values, such as when
 * creating high-value/long-lived secrets like RSA public/private
 * keys.  To help guide applications in selecting a suitable strong
 * {@code SecureRandom} implementation, Java distributions
 * include a list of known strong {@code SecureRandom}
 * implementations in the {@code securerandom.strongAlgorithms}
 * Security property.
 * <p>
 * Every implementation of the Java platform is required to
 * support at least one strong {@code SecureRandom} implementation.
 *
 * @return a strong {@code SecureRandom} implementation
 *
 * @throws java.security.NoSuchAlgorithmException if no algorithm is available
 *
 * @see java.security.Security#getProperty(String)
 *
 * @since 1.8
 */

public static java.security.SecureRandom getInstanceStrong() throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns the effective {@link java.security.SecureRandomParameters SecureRandomParameters} for this
 * {@code SecureRandom} instance.
 * <p>
 * The returned value can be different from the
 * {@code SecureRandomParameters} object passed into a {@code getInstance}
 * method, but it cannot change during the lifetime of this
 * {@code SecureRandom} object.
 * <p>
 * A caller can use the returned value to find out what features this
 * {@code SecureRandom} supports.
 *
 * @return the effective {@link java.security.SecureRandomParameters SecureRandomParameters} parameters,
 * or {@code null} if no parameters were used.
 *
 * @since 9
 * @see java.security.SecureRandomSpi
 */

public java.security.SecureRandomParameters getParameters() { throw new RuntimeException("Stub!"); }

/**
 * Returns the provider of this {@code SecureRandom} object.
 *
 * @return the provider of this {@code SecureRandom} object.
 */

public final java.security.Provider getProvider() { throw new RuntimeException("Stub!"); }

/**
 * Returns the given number of seed bytes, computed using the seed
 * generation algorithm that this class uses to seed itself.  This
 * call may be used to seed other random number generators.
 *
 * <p>This method is only included for backwards compatibility.
 * The caller is encouraged to use one of the alternative
 * {@code getInstance} methods to obtain a {@code SecureRandom} object, and
 * then call the {@code generateSeed} method to obtain seed bytes
 * from that object.
 *
 * @param numBytes the number of seed bytes to generate.
 *
 * @throws java.lang.IllegalArgumentException if {@code numBytes} is negative
 * @return the seed bytes.
 *
 * @see #setSeed
 */

public static byte[] getSeed(int numBytes) { throw new RuntimeException("Stub!"); }

/**
 * Generates an integer containing the user-specified number of
 * pseudo-random bits (right justified, with leading zeros).  This
 * method overrides a {@code java.util.Random} method, and serves
 * to provide a source of random bits to all of the methods inherited
 * from that class (for example, {@code nextInt},
 * {@code nextLong}, and {@code nextFloat}).
 *
 * @param numBits number of pseudo-random bits to be generated, where
 * {@code 0 <= numBits <= 32}.
 *
 * @return an {@code int} containing the user-specified number
 * of pseudo-random bits (right justified, with leading zeros).
 */

protected final int next(int numBits) { throw new RuntimeException("Stub!"); }

/**
 * Generates a user-specified number of random bytes.
 *
 * @param bytes the array to be filled in with random bytes.
 */

public void nextBytes(byte[] bytes) { throw new RuntimeException("Stub!"); }

/**
 * Generates a user-specified number of random bytes with
 * additional parameters.
 *
 * @param bytes the array to be filled in with random bytes
 * @param params additional parameters
 * @throws java.lang.NullPointerException if {@code bytes} is null
 * @throws java.lang.UnsupportedOperationException if the underlying provider
 *         implementation has not overridden this method
 * @throws java.lang.IllegalArgumentException if {@code params} is {@code null},
 *         illegal or unsupported by this {@code SecureRandom}
 *
 * @since 9
 */

public void nextBytes(byte[] bytes, java.security.SecureRandomParameters params) { throw new RuntimeException("Stub!"); }

/**
 * Reseeds this {@code SecureRandom} with entropy input read from its
 * entropy source.
 *
 * @throws java.lang.UnsupportedOperationException if the underlying provider
 *         implementation has not overridden this method.
 *
 * @since 9
 */

public void reseed() { throw new RuntimeException("Stub!"); }

/**
 * Reseeds this {@code SecureRandom} with entropy input read from its
 * entropy source with additional parameters.
 * <p>
 * Note that entropy is obtained from an entropy source. While
 * some data in {@code params} may contain entropy, its main usage is to
 * provide diversity.
 *
 * @param params extra parameters
 * @throws java.lang.UnsupportedOperationException if the underlying provider
 *         implementation has not overridden this method.
 * @throws java.lang.IllegalArgumentException if {@code params} is {@code null},
 *         illegal or unsupported by this {@code SecureRandom}
 *
 * @since 9
 */

public void reseed(java.security.SecureRandomParameters params) { throw new RuntimeException("Stub!"); }

/**
 * Reseeds this random object with the given seed. The seed supplements,
 * rather than replaces, the existing seed. Thus, repeated calls are
 * guaranteed never to reduce randomness.
 * <p>
 * A PRNG {@code SecureRandom} will not seed itself automatically if
 * {@code setSeed} is called before any {@code nextBytes} or {@code reseed}
 * calls. The caller should make sure that the {@code seed} argument
 * contains enough entropy for the security of this {@code SecureRandom}.
 *
 * @param seed the seed.
 *
 * @see #getSeed
 */

public void setSeed(byte[] seed) { throw new RuntimeException("Stub!"); }

/**
 * Reseeds this random object, using the eight bytes contained
 * in the given {@code long seed}. The given seed supplements,
 * rather than replaces, the existing seed. Thus, repeated calls
 * are guaranteed never to reduce randomness.
 *
 * <p>This method is defined for compatibility with
 * {@code java.util.Random}.
 *
 * @param seed the seed.
 *
 * @see #getSeed
 */

public void setSeed(long seed) { throw new RuntimeException("Stub!"); }

/**
 * Returns a Human-readable string representation of this
 * {@code SecureRandom}.
 *
 * @return the string representation
 */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }
}

