/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * This file is available under and governed by the GNU General Public
 * License version 2 only, as published by the Free Software Foundation.
 * However, the following notice accompanied the original version of this
 * file:
 *
 * Written by Doug Lea with assistance from members of JCP JSR-166
 * Expert Group and released to the public domain, as explained at
 * http://creativecommons.org/publicdomain/zero/1.0/
 */


package java.util.concurrent;

import java.util.function.Consumer;

/**
 * A {@link java.util.concurrent.Flow.Publisher Flow.Publisher} that asynchronously issues submitted
 * (non-null) items to current subscribers until it is closed.  Each
 * current subscriber receives newly submitted items in the same order
 * unless drops or exceptions are encountered.  Using a
 * SubmissionPublisher allows item generators to act as compliant <a
 * href="http://www.reactive-streams.org/"> reactive-streams</a>
 * Publishers relying on drop handling and/or blocking for flow
 * control.
 *
 * <p>A SubmissionPublisher uses the {@link java.util.concurrent.Executor Executor} supplied in its
 * constructor for delivery to subscribers. The best choice of
 * Executor depends on expected usage. If the generator(s) of
 * submitted items run in separate threads, and the number of
 * subscribers can be estimated, consider using a {@link java.util.concurrent.Executors#newFixedThreadPool  }. Otherwise consider using the
 * default, normally the {@link java.util.concurrent.ForkJoinPool#commonPool ForkJoinPool.commonPool}.
 *
 * <p>Buffering allows producers and consumers to transiently operate
 * at different rates.  Each subscriber uses an independent buffer.
 * Buffers are created upon first use and expanded as needed up to the
 * given maximum. (The enforced capacity may be rounded up to the
 * nearest power of two and/or bounded by the largest value supported
 * by this implementation.)  Invocations of {@link java.util.concurrent.Flow.Subscription#request(long) request} do not directly result in
 * buffer expansion, but risk saturation if unfilled requests exceed
 * the maximum capacity.  The default value of {@link java.util.concurrent.Flow#defaultBufferSize()  } may provide a useful starting point for
 * choosing a capacity based on expected rates, resources, and usages.
 *
 * <p>A single SubmissionPublisher may be shared among multiple
 * sources. Actions in a source thread prior to publishing an item or
 * issuing a signal <a href="package-summary.html#MemoryVisibility">
 * <i>happen-before</i></a> actions subsequent to the corresponding
 * access by each subscriber. But reported estimates of lag and demand
 * are designed for use in monitoring, not for synchronization
 * control, and may reflect stale or inaccurate views of progress.
 *
 * <p>Publication methods support different policies about what to do
 * when buffers are saturated. Method {@link #submit(java.lang.Object) submit}
 * blocks until resources are available. This is simplest, but least
 * responsive.  The {@code offer} methods may drop items (either
 * immediately or with bounded timeout), but provide an opportunity to
 * interpose a handler and then retry.
 *
 * <p>If any Subscriber method throws an exception, its subscription
 * is cancelled.  If a handler is supplied as a constructor argument,
 * it is invoked before cancellation upon an exception in method
 * {@link java.util.concurrent.Flow.Subscriber#onNext onNext}, but exceptions in methods
 * {@link java.util.concurrent.Flow.Subscriber#onSubscribe onSubscribe},
 * {@link java.util.concurrent.Flow.Subscriber#onError(java.lang.Throwable) onError} and
 * {@link java.util.concurrent.Flow.Subscriber#onComplete() onComplete} are not recorded or
 * handled before cancellation.  If the supplied Executor throws
 * {@link java.util.concurrent.RejectedExecutionException RejectedExecutionException} (or any other RuntimeException
 * or Error) when attempting to execute a task, or a drop handler
 * throws an exception when processing a dropped item, then the
 * exception is rethrown. In these cases, not all subscribers will
 * have been issued the published item. It is usually good practice to
 * {@link #closeExceptionally closeExceptionally} in these cases.
 *
 * <p>Method {@link #consume(java.util.function.Consumer)} simplifies support for a
 * common case in which the only action of a subscriber is to request
 * and process all items using a supplied function.
 *
 * <p>This class may also serve as a convenient base for subclasses
 * that generate items, and use the methods in this class to publish
 * them.  For example here is a class that periodically publishes the
 * items generated from a supplier. (In practice you might add methods
 * to independently start and stop generation, to share Executors
 * among publishers, and so on, or use a SubmissionPublisher as a
 * component rather than a superclass.)
 *
 * <pre> {@code
 * class PeriodicPublisher<T> extends SubmissionPublisher<T> {
 *   final ScheduledFuture<?> periodicTask;
 *   final ScheduledExecutorService scheduler;
 *   PeriodicPublisher(Executor executor, int maxBufferCapacity,
 *                     Supplier<? extends T> supplier,
 *                     long period, TimeUnit unit) {
 *     super(executor, maxBufferCapacity);
 *     scheduler = new ScheduledThreadPoolExecutor(1);
 *     periodicTask = scheduler.scheduleAtFixedRate(
 *       () -> submit(supplier.get()), 0, period, unit);
 *   }
 *   public void close() {
 *     periodicTask.cancel(false);
 *     scheduler.shutdown();
 *     super.close();
 *   }
 * }}</pre>
 *
 * <p>Here is an example of a {@link java.util.concurrent.Flow.Processor Flow.Processor} implementation.
 * It uses single-step requests to its publisher for simplicity of
 * illustration. A more adaptive version could monitor flow using the
 * lag estimate returned from {@code submit}, along with other utility
 * methods.
 *
 * <pre> {@code
 * class TransformProcessor<S,T> extends SubmissionPublisher<T>
 *   implements Flow.Processor<S,T> {
 *   final Function<? super S, ? extends T> function;
 *   Flow.Subscription subscription;
 *   TransformProcessor(Executor executor, int maxBufferCapacity,
 *                      Function<? super S, ? extends T> function) {
 *     super(executor, maxBufferCapacity);
 *     this.function = function;
 *   }
 *   public void onSubscribe(Flow.Subscription subscription) {
 *     (this.subscription = subscription).request(1);
 *   }
 *   public void onNext(S item) {
 *     subscription.request(1);
 *     submit(function.apply(item));
 *   }
 *   public void onError(Throwable ex) { closeExceptionally(ex); }
 *   public void onComplete() { close(); }
 * }}</pre>
 *
 * @param <T> the published item type
 * @author Doug Lea
 * @since 9
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class SubmissionPublisher<T> implements java.lang.AutoCloseable, java.util.concurrent.Flow.Publisher<T> {

/**
 * Creates a new SubmissionPublisher using the {@link java.util.concurrent.ForkJoinPool#commonPool()  } for async delivery to subscribers
 * (unless it does not support a parallelism level of at least two,
 * in which case, a new Thread is created to run each task), with
 * maximum buffer capacity of {@link java.util.concurrent.Flow#defaultBufferSize Flow.defaultBufferSize}, and no
 * handler for Subscriber exceptions in method {@link java.util.concurrent.Flow.Subscriber#onNext(java.lang.Object) onNext}.
 */

public SubmissionPublisher() { throw new RuntimeException("Stub!"); }

/**
 * Creates a new SubmissionPublisher using the given Executor for
 * async delivery to subscribers, with the given maximum buffer size
 * for each subscriber, and no handler for Subscriber exceptions in
 * method {@link java.util.concurrent.Flow.Subscriber#onNext(java.lang.Object) onNext}.
 *
 * @param executor the executor to use for async delivery,
 * supporting creation of at least one independent thread
 * @param maxBufferCapacity the maximum capacity for each
 * subscriber's buffer (the enforced capacity may be rounded up to
 * the nearest power of two and/or bounded by the largest value
 * supported by this implementation; method {@link #getMaxBufferCapacity}
 * returns the actual value)
 * @throws java.lang.NullPointerException if executor is null
 * @throws java.lang.IllegalArgumentException if maxBufferCapacity not
 * positive
 */

public SubmissionPublisher(java.util.concurrent.Executor executor, int maxBufferCapacity) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new SubmissionPublisher using the given Executor for
 * async delivery to subscribers, with the given maximum buffer size
 * for each subscriber, and, if non-null, the given handler invoked
 * when any Subscriber throws an exception in method {@link java.util.concurrent.Flow.Subscriber#onNext(java.lang.Object) onNext}.
 *
 * @param executor the executor to use for async delivery,
 * supporting creation of at least one independent thread
 * @param maxBufferCapacity the maximum capacity for each
 * subscriber's buffer (the enforced capacity may be rounded up to
 * the nearest power of two and/or bounded by the largest value
 * supported by this implementation; method {@link #getMaxBufferCapacity}
 * returns the actual value)
 * @param handler if non-null, procedure to invoke upon exception
 * thrown in method {@code onNext}
 * @throws java.lang.NullPointerException if executor is null
 * @throws java.lang.IllegalArgumentException if maxBufferCapacity not
 * positive
 */

public SubmissionPublisher(java.util.concurrent.Executor executor, int maxBufferCapacity, java.util.function.BiConsumer<? super java.util.concurrent.Flow.Subscriber<? super T>,? super java.lang.Throwable> handler) { throw new RuntimeException("Stub!"); }

/**
 * Unless already closed, issues {@link java.util.concurrent.Flow.Subscriber#onComplete() onComplete} signals to current
 * subscribers, and disallows subsequent attempts to publish.
 * Upon return, this method does <em>NOT</em> guarantee that all
 * subscribers have yet completed.
 */

public void close() { throw new RuntimeException("Stub!"); }

/**
 * Unless already closed, issues {@link java.util.concurrent.Flow.Subscriber#onError(java.lang.Throwable) onError} signals to current
 * subscribers with the given error, and disallows subsequent
 * attempts to publish.  Future subscribers also receive the given
 * error. Upon return, this method does <em>NOT</em> guarantee
 * that all subscribers have yet completed.
 *
 * @param error the {@code onError} argument sent to subscribers
 * @throws java.lang.NullPointerException if error is null
 */

public void closeExceptionally(java.lang.Throwable error) { throw new RuntimeException("Stub!"); }

/**
 * Processes all published items using the given Consumer function.
 * Returns a CompletableFuture that is completed normally when this
 * publisher signals {@link java.util.concurrent.Flow.Subscriber#onComplete()  onComplete}, or completed exceptionally upon any error, or an
 * exception is thrown by the Consumer, or the returned
 * CompletableFuture is cancelled, in which case no further items
 * are processed.
 *
 * @param consumer the function applied to each onNext item
 * @return a CompletableFuture that is completed normally
 * when the publisher signals onComplete, and exceptionally
 * upon any error or cancellation
 * @throws java.lang.NullPointerException if consumer is null
 */

public java.util.concurrent.CompletableFuture<java.lang.Void> consume(java.util.function.Consumer<? super T> consumer) { throw new RuntimeException("Stub!"); }

/**
 * Returns an estimate of the maximum number of items produced but
 * not yet consumed among all current subscribers.
 *
 * @return the estimate
 */

public int estimateMaximumLag() { throw new RuntimeException("Stub!"); }

/**
 * Returns an estimate of the minimum number of items requested
 * (via {@link java.util.concurrent.Flow.Subscription#request(long) request}) but not
 * yet produced, among all current subscribers.
 *
 * @return the estimate, or zero if no subscribers
 */

public long estimateMinimumDemand() { throw new RuntimeException("Stub!"); }

/**
 * Returns the exception associated with {@link
 * #closeExceptionally(java.lang.Throwable) closeExceptionally}, or null if
 * not closed or if closed normally.
 *
 * @return the exception, or null if none
 */

public java.lang.Throwable getClosedException() { throw new RuntimeException("Stub!"); }

/**
 * Returns the Executor used for asynchronous delivery.
 *
 * @return the Executor used for asynchronous delivery
 */

public java.util.concurrent.Executor getExecutor() { throw new RuntimeException("Stub!"); }

/**
 * Returns the maximum per-subscriber buffer capacity.
 *
 * @return the maximum per-subscriber buffer capacity
 */

public int getMaxBufferCapacity() { throw new RuntimeException("Stub!"); }

/**
 * Returns the number of current subscribers.
 *
 * @return the number of current subscribers
 */

public int getNumberOfSubscribers() { throw new RuntimeException("Stub!"); }

/**
 * Returns a list of current subscribers for monitoring and
 * tracking purposes, not for invoking {@link java.util.concurrent.Flow.Subscriber Flow.Subscriber}
 * methods on the subscribers.
 *
 * @return list of current subscribers
 */

public java.util.List<java.util.concurrent.Flow.Subscriber<? super T>> getSubscribers() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if this publisher has any subscribers.
 *
 * @return true if this publisher has any subscribers
 */

public boolean hasSubscribers() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if this publisher is not accepting submissions.
 *
 * @return true if closed
 */

public boolean isClosed() { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the given Subscriber is currently subscribed.
 *
 * @param subscriber the subscriber
 * @return true if currently subscribed
 * @throws java.lang.NullPointerException if subscriber is null
 */

public boolean isSubscribed(java.util.concurrent.Flow.Subscriber<? super T> subscriber) { throw new RuntimeException("Stub!"); }

/**
 * Publishes the given item, if possible, to each current subscriber
 * by asynchronously invoking its {@link java.util.concurrent.Flow.Subscriber#onNext(java.lang.Object) onNext} method. The item may be
 * dropped by one or more subscribers if resource limits are
 * exceeded, in which case the given handler (if non-null) is
 * invoked, and if it returns true, retried once.  Other calls to
 * methods in this class by other threads are blocked while the
 * handler is invoked.  Unless recovery is assured, options are
 * usually limited to logging the error and/or issuing an {@link java.util.concurrent.Flow.Subscriber#onError(java.lang.Throwable) onError} signal to the
 * subscriber.
 *
 * <p>This method returns a status indicator: If negative, it
 * represents the (negative) number of drops (failed attempts to
 * issue the item to a subscriber). Otherwise it is an estimate of
 * the maximum lag (number of items submitted but not yet
 * consumed) among all current subscribers. This value is at least
 * one (accounting for this submitted item) if there are any
 * subscribers, else zero.
 *
 * <p>If the Executor for this publisher throws a
 * RejectedExecutionException (or any other RuntimeException or
 * Error) when attempting to asynchronously notify subscribers, or
 * the drop handler throws an exception when processing a dropped
 * item, then this exception is rethrown.
 *
 * @param item the (non-null) item to publish
 * @param onDrop if non-null, the handler invoked upon a drop to a
 * subscriber, with arguments of the subscriber and item; if it
 * returns true, an offer is re-attempted (once)
 * @return if negative, the (negative) number of drops; otherwise
 * an estimate of maximum lag
 * @throws java.lang.IllegalStateException if closed
 * @throws java.lang.NullPointerException if item is null
 * @throws java.util.concurrent.RejectedExecutionException if thrown by Executor
 */

public int offer(T item, java.util.function.BiPredicate<java.util.concurrent.Flow.Subscriber<? super T>,? super T> onDrop) { throw new RuntimeException("Stub!"); }

/**
 * Publishes the given item, if possible, to each current subscriber
 * by asynchronously invoking its {@link java.util.concurrent.Flow.Subscriber#onNext(java.lang.Object) onNext} method, blocking while
 * resources for any subscription are unavailable, up to the
 * specified timeout or until the caller thread is interrupted, at
 * which point the given handler (if non-null) is invoked, and if it
 * returns true, retried once. (The drop handler may distinguish
 * timeouts from interrupts by checking whether the current thread
 * is interrupted.)  Other calls to methods in this class by other
 * threads are blocked while the handler is invoked.  Unless
 * recovery is assured, options are usually limited to logging the
 * error and/or issuing an {@link java.util.concurrent.Flow.Subscriber#onError(java.lang.Throwable)  onError} signal to the subscriber.
 *
 * <p>This method returns a status indicator: If negative, it
 * represents the (negative) number of drops (failed attempts to
 * issue the item to a subscriber). Otherwise it is an estimate of
 * the maximum lag (number of items submitted but not yet
 * consumed) among all current subscribers. This value is at least
 * one (accounting for this submitted item) if there are any
 * subscribers, else zero.
 *
 * <p>If the Executor for this publisher throws a
 * RejectedExecutionException (or any other RuntimeException or
 * Error) when attempting to asynchronously notify subscribers, or
 * the drop handler throws an exception when processing a dropped
 * item, then this exception is rethrown.
 *
 * @param item the (non-null) item to publish
 * @param timeout how long to wait for resources for any subscriber
 * before giving up, in units of {@code unit}
 * @param unit a {@code TimeUnit} determining how to interpret the
 * {@code timeout} parameter
 * @param onDrop if non-null, the handler invoked upon a drop to a
 * subscriber, with arguments of the subscriber and item; if it
 * returns true, an offer is re-attempted (once)
 * @return if negative, the (negative) number of drops; otherwise
 * an estimate of maximum lag
 * @throws java.lang.IllegalStateException if closed
 * @throws java.lang.NullPointerException if item is null
 * @throws java.util.concurrent.RejectedExecutionException if thrown by Executor
 */

public int offer(T item, long timeout, java.util.concurrent.TimeUnit unit, java.util.function.BiPredicate<java.util.concurrent.Flow.Subscriber<? super T>,? super T> onDrop) { throw new RuntimeException("Stub!"); }

/**
 * Publishes the given item to each current subscriber by
 * asynchronously invoking its {@link java.util.concurrent.Flow.Subscriber#onNext(java.lang.Object)  onNext} method, blocking uninterruptibly while resources for any
 * subscriber are unavailable. This method returns an estimate of
 * the maximum lag (number of items submitted but not yet consumed)
 * among all current subscribers. This value is at least one
 * (accounting for this submitted item) if there are any
 * subscribers, else zero.
 *
 * <p>If the Executor for this publisher throws a
 * RejectedExecutionException (or any other RuntimeException or
 * Error) when attempting to asynchronously notify subscribers,
 * then this exception is rethrown, in which case not all
 * subscribers will have been issued this item.
 *
 * @param item the (non-null) item to publish
 * @return the estimated maximum lag among subscribers
 * @throws java.lang.IllegalStateException if closed
 * @throws java.lang.NullPointerException if item is null
 * @throws java.util.concurrent.RejectedExecutionException if thrown by Executor
 */

public int submit(T item) { throw new RuntimeException("Stub!"); }

/**
 * Adds the given Subscriber unless already subscribed.  If already
 * subscribed, the Subscriber's {@link java.util.concurrent.Flow.Subscriber#onError(java.lang.Throwable) onError} method is invoked on
 * the existing subscription with an {@link java.lang.IllegalStateException IllegalStateException}.
 * Otherwise, upon success, the Subscriber's {@link java.util.concurrent.Flow.Subscriber#onSubscribe onSubscribe} method is invoked
 * asynchronously with a new {@link java.util.concurrent.Flow.Subscription Flow.Subscription}.  If {@link java.util.concurrent.Flow.Subscriber#onSubscribe onSubscribe} throws an exception, the
 * subscription is cancelled. Otherwise, if this SubmissionPublisher
 * was closed exceptionally, then the subscriber's {@link java.util.concurrent.Flow.Subscriber#onError onError} method is invoked with the
 * corresponding exception, or if closed without exception, the
 * subscriber's {@link java.util.concurrent.Flow.Subscriber#onComplete() onComplete}
 * method is invoked.  Subscribers may enable receiving items by
 * invoking the {@link java.util.concurrent.Flow.Subscription#request(long) request}
 * method of the new Subscription, and may unsubscribe by invoking
 * its {@link java.util.concurrent.Flow.Subscription#cancel() cancel} method.
 *
 * @param subscriber the subscriber
 * @throws java.lang.NullPointerException if subscriber is null
 */

public void subscribe(java.util.concurrent.Flow.Subscriber<? super T> subscriber) { throw new RuntimeException("Stub!"); }
}

