#include "base/exception.hpp"
#include "base/objectlock.hpp"
#include "base/utility.hpp"
#include "base/convert.hpp"
#include "base/debug.hpp"
#include "base/dependencygraph.hpp"
#include "base/logger.hpp"
#include "base/function.hpp"
#include "base/configobject.hpp"
#include "base/configtype.hpp"
#ifdef _MSC_VER
#pragma warning( push )
#pragma warning( disable : 4244 )
#pragma warning( disable : 4800 )
#endif /* _MSC_VER */

namespace icinga
{

TypeImpl<ConfigObjectBase>::TypeImpl()
{ }

TypeImpl<ConfigObjectBase>::~TypeImpl()
{ }

String TypeImpl<ConfigObjectBase>::GetName() const
{
	return "ConfigObjectBase";
}

int TypeImpl<ConfigObjectBase>::GetAttributes() const
{
	return 1;
}

Type::Ptr TypeImpl<ConfigObjectBase>::GetBaseType() const
{
	return Object::TypeInstance;
}

int TypeImpl<ConfigObjectBase>::GetFieldId(const String& name) const
{

	return -1;
}

Field TypeImpl<ConfigObjectBase>::GetFieldInfo(int id) const
{
	throw std::runtime_error("Invalid field ID.");
}

int TypeImpl<ConfigObjectBase>::GetFieldCount() const
{
	return 0;
}

ObjectFactory TypeImpl<ConfigObjectBase>::GetFactory() const
{
	return TypeHelper<ConfigObjectBase, false>::GetFactory();
}

int TypeImpl<ConfigObjectBase>::GetActivationPriority() const
{
	return 0;
}

void TypeImpl<ConfigObjectBase>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	throw std::runtime_error("Invalid field ID.");
}

void ObjectImpl<ConfigObjectBase>::Validate(int types, const ValidationUtils& utils)
{
}

ObjectImpl<ConfigObjectBase>::ObjectImpl()
{
}

ObjectImpl<ConfigObjectBase>::~ObjectImpl()
{ }

TypeImpl<ConfigObject>::TypeImpl()
{ }

TypeImpl<ConfigObject>::~TypeImpl()
{ }

String TypeImpl<ConfigObject>::GetName() const
{
	return "ConfigObject";
}

int TypeImpl<ConfigObject>::GetAttributes() const
{
	return 1;
}

Type::Ptr TypeImpl<ConfigObject>::GetBaseType() const
{
	return ConfigObjectBase::TypeInstance;
}

int TypeImpl<ConfigObject>::GetFieldId(const String& name) const
{
	int offset = ConfigObjectBase::TypeInstance->GetFieldCount();

	switch (static_cast<int>(Utility::SDBM(name, 2))) {
		case 6232000:
			if (name == "__name")
				return offset + 4;

			break;
		case 6363202:
			if (name == "active")
				return offset + 16;

			break;
		case 6625619:
			if (name == "extensions")
				return offset + 8;

			break;
		case 6822393:
			if (name == "ha_mode")
				return offset + 10;

			break;
		case 6887994:
			if (name == "icingadb_identifier")
				return offset + 1;

			break;
		case 7215987:
			if (name == "name")
				return offset + 0;

			break;
		case 7281603:
			if (name == "original_attributes")
				return offset + 7;

			break;
		case 7347185:
			if (name == "package")
				return offset + 2;
			if (name == "paused")
				return offset + 11;
			if (name == "pause_called")
				return offset + 14;

			break;
		case 7478387:
			if (name == "resume_called")
				return offset + 15;

			break;
		case 7543996:
			if (name == "source_location")
				return offset + 9;

			break;
		case 7544001:
			if (name == "start_called")
				return offset + 12;
			if (name == "stop_called")
				return offset + 13;
			if (name == "state_loaded")
				return offset + 17;

			break;
		case 7609585:
			if (name == "templates")
				return offset + 6;

			break;
		case 7740783:
			if (name == "version")
				return offset + 5;

			break;
		case 8003189:
			if (name == "zone")
				return offset + 3;

			break;
	}

	return ConfigObjectBase::TypeInstance->GetFieldId(name);
}

Field TypeImpl<ConfigObject>::GetFieldInfo(int id) const
{
	int real_id = id - ConfigObjectBase::TypeInstance->GetFieldCount();
	if (real_id < 0) { return ConfigObjectBase::TypeInstance->GetFieldInfo(id); }
	switch (real_id) {
		case 0:
			return {0, "String", "name", "name", nullptr, 1282, 0};
		case 1:
			return {1, "String", "icingadb_identifier", "icingadb_identifier", nullptr, 3073, 0};
		case 2:
			return {2, "String", "package", "package", nullptr, 1026, 0};
		case 3:
			return {3, "String", "zone", "zone", "Zone", 1026, 0};
		case 4:
			return {4, "String", "__name", "__name", nullptr, 1026, 0};
		case 5:
			return {5, "Number", "version", "version", nullptr, 1028, 0};
		case 6:
			return {6, "Array", "templates", "templates", nullptr, 1042, 0};
		case 7:
			return {7, "Dictionary", "original_attributes", "original_attributes", nullptr, 1025, 0};
		case 8:
			return {8, "Dictionary", "extensions", "extensions", nullptr, 3121, 0};
		case 9:
			return {9, "Dictionary", "source_location", "source_location", nullptr, 1090, 0};
		case 10:
			return {10, "Number", "ha_mode", "ha_mode", nullptr, 9, 0};
		case 11:
			return {11, "Boolean", "paused", "paused", nullptr, 1041, 0};
		case 12:
			return {12, "Boolean", "start_called", "start_called", nullptr, 3089, 0};
		case 13:
			return {13, "Boolean", "stop_called", "stop_called", nullptr, 3089, 0};
		case 14:
			return {14, "Boolean", "pause_called", "pause_called", nullptr, 3089, 0};
		case 15:
			return {15, "Boolean", "resume_called", "resume_called", nullptr, 3089, 0};
		case 16:
			return {16, "Boolean", "active", "active", nullptr, 1041, 0};
		case 17:
			return {17, "Boolean", "state_loaded", "state_loaded", nullptr, 3121, 0};
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

int TypeImpl<ConfigObject>::GetFieldCount() const
{
	return 18 + ConfigObjectBase::TypeInstance->GetFieldCount();
}

ObjectFactory TypeImpl<ConfigObject>::GetFactory() const
{
	return TypeHelper<ConfigObject, false>::GetFactory();
}

int TypeImpl<ConfigObject>::GetActivationPriority() const
{
	return 0;
}

void TypeImpl<ConfigObject>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	int real_id = fieldId - ConfigObjectBase::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObjectBase::TypeInstance->RegisterAttributeHandler(fieldId, callback); return; }
	switch (real_id) {
		case 0:
			ObjectImpl<ConfigObject>::OnShortNameChanged.connect(callback);
			break;
		case 1:
			ObjectImpl<ConfigObject>::OnIcingadbIdentifierChanged.connect(callback);
			break;
		case 2:
			ObjectImpl<ConfigObject>::OnPackageChanged.connect(callback);
			break;
		case 3:
			ObjectImpl<ConfigObject>::OnZoneNameChanged.connect(callback);
			break;
		case 4:
			ObjectImpl<ConfigObject>::OnNameChanged.connect(callback);
			break;
		case 5:
			ObjectImpl<ConfigObject>::OnVersionChanged.connect(callback);
			break;
		case 6:
			ObjectImpl<ConfigObject>::OnTemplatesChanged.connect(callback);
			break;
		case 7:
			ObjectImpl<ConfigObject>::OnOriginalAttributesChanged.connect(callback);
			break;
		case 8:
			ObjectImpl<ConfigObject>::OnExtensionsChanged.connect(callback);
			break;
		case 9:
			ObjectImpl<ConfigObject>::OnSourceLocationChanged.connect(callback);
			break;
		case 10:
			ObjectImpl<ConfigObject>::OnHAModeChanged.connect(callback);
			break;
		case 11:
			ObjectImpl<ConfigObject>::OnPausedChanged.connect(callback);
			break;
		case 12:
			ObjectImpl<ConfigObject>::OnStartCalledChanged.connect(callback);
			break;
		case 13:
			ObjectImpl<ConfigObject>::OnStopCalledChanged.connect(callback);
			break;
		case 14:
			ObjectImpl<ConfigObject>::OnPauseCalledChanged.connect(callback);
			break;
		case 15:
			ObjectImpl<ConfigObject>::OnResumeCalledChanged.connect(callback);
			break;
		case 16:
			ObjectImpl<ConfigObject>::OnActiveChanged.connect(callback);
			break;
		case 17:
			ObjectImpl<ConfigObject>::OnStateLoadedChanged.connect(callback);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<ConfigObject>::Validate(int types, const ValidationUtils& utils)
{
	ConfigObjectBase::Validate(types, utils);

	if (2 & types)
		ValidateShortName(Lazy<String>([this]() { return GetShortName(); }), utils);
	if (1 & types)
		ValidateIcingadbIdentifier(Lazy<String>([this]() { return GetIcingadbIdentifier(); }), utils);
	if (2 & types)
		ValidatePackage(Lazy<String>([this]() { return GetPackage(); }), utils);
	if (2 & types)
		ValidateZoneName(Lazy<String>([this]() { return GetZoneName(); }), utils);
	if (2 & types)
		ValidateName(Lazy<String>([this]() { return GetName(); }), utils);
	if (4 & types)
		ValidateVersion(Lazy<double>([this]() { return GetVersion(); }), utils);
	if (2 & types)
		ValidateTemplates(Lazy<Array::Ptr>([this]() { return GetTemplates(); }), utils);
	if (1 & types)
		ValidateOriginalAttributes(Lazy<Dictionary::Ptr>([this]() { return GetOriginalAttributes(); }), utils);
	if (1 & types)
		ValidateExtensions(Lazy<Dictionary::Ptr>([this]() { return GetExtensions(); }), utils);
	if (2 & types)
		ValidateSourceLocation(Lazy<Dictionary::Ptr>([this]() { return GetSourceLocation(); }), utils);
	if (1 & types)
		ValidateHAMode(Lazy<HAMode>([this]() { return GetHAMode(); }), utils);
	if (1 & types)
		ValidatePaused(Lazy<bool>([this]() { return GetPaused(); }), utils);
	if (1 & types)
		ValidateStartCalled(Lazy<bool>([this]() { return GetStartCalled(); }), utils);
	if (1 & types)
		ValidateStopCalled(Lazy<bool>([this]() { return GetStopCalled(); }), utils);
	if (1 & types)
		ValidatePauseCalled(Lazy<bool>([this]() { return GetPauseCalled(); }), utils);
	if (1 & types)
		ValidateResumeCalled(Lazy<bool>([this]() { return GetResumeCalled(); }), utils);
	if (1 & types)
		ValidateActive(Lazy<bool>([this]() { return GetActive(); }), utils);
	if (1 & types)
		ValidateStateLoaded(Lazy<bool>([this]() { return GetStateLoaded(); }), utils);
}

void ObjectImpl<ConfigObject>::SimpleValidateShortName(const Lazy<String>& value, const ValidationUtils& utils)
{
	if (value().IsEmpty())
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "name" }, "Attribute must not be empty."));

}

void ObjectImpl<ConfigObject>::SimpleValidateIcingadbIdentifier(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidatePackage(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateZoneName(const Lazy<String>& value, const ValidationUtils& utils)
{
			if (!value().IsEmpty() && !utils.ValidateName("Zone", value()))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "zone" }, "Object '" + value() + "' of type 'Zone' does not exist."));
}

void ObjectImpl<ConfigObject>::SimpleValidateName(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateVersion(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateTemplates(const Lazy<Array::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateOriginalAttributes(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateExtensions(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateSourceLocation(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateHAMode(const Lazy<HAMode>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidatePaused(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateStartCalled(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateStopCalled(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidatePauseCalled(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateResumeCalled(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateActive(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<ConfigObject>::SimpleValidateStateLoaded(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

ObjectImpl<ConfigObject>::ObjectImpl()
{
	SetShortName(GetDefaultShortName(), true);
	SetIcingadbIdentifier(GetDefaultIcingadbIdentifier(), true);
	SetPackage(GetDefaultPackage(), true);
	SetZoneName(GetDefaultZoneName(), true);
	SetName(GetDefaultName(), true);
	SetVersion(GetDefaultVersion(), true);
	SetTemplates(GetDefaultTemplates(), true);
	SetOriginalAttributes(GetDefaultOriginalAttributes(), true);
	SetExtensions(GetDefaultExtensions(), true);
	SetSourceLocation(GetDefaultSourceLocation(), true);
	SetHAMode(GetDefaultHAMode(), true);
	SetPaused(GetDefaultPaused(), true);
	SetStartCalled(GetDefaultStartCalled(), true);
	SetStopCalled(GetDefaultStopCalled(), true);
	SetPauseCalled(GetDefaultPauseCalled(), true);
	SetResumeCalled(GetDefaultResumeCalled(), true);
	SetActive(GetDefaultActive(), true);
	SetStateLoaded(GetDefaultStateLoaded(), true);
}

ObjectImpl<ConfigObject>::~ObjectImpl()
{ }

void ObjectImpl<ConfigObject>::SetField(int id, const Value& value, bool suppress_events, const Value& cookie)
{
	int real_id = id - ConfigObjectBase::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObjectBase::SetField(id, value, suppress_events, cookie); return; }
	switch (real_id) {
		case 0:
			SetShortName(value, suppress_events, cookie);
			break;
		case 1:
			SetIcingadbIdentifier(value, suppress_events, cookie);
			break;
		case 2:
			SetPackage(value, suppress_events, cookie);
			break;
		case 3:
			SetZoneName(value, suppress_events, cookie);
			break;
		case 4:
			SetName(value, suppress_events, cookie);
			break;
		case 5:
			SetVersion(value, suppress_events, cookie);
			break;
		case 6:
			SetTemplates(value, suppress_events, cookie);
			break;
		case 7:
			SetOriginalAttributes(value, suppress_events, cookie);
			break;
		case 8:
			SetExtensions(value, suppress_events, cookie);
			break;
		case 9:
			SetSourceLocation(value, suppress_events, cookie);
			break;
		case 10:
			SetHAMode(static_cast<HAMode>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 11:
			SetPaused(value, suppress_events, cookie);
			break;
		case 12:
			SetStartCalled(value, suppress_events, cookie);
			break;
		case 13:
			SetStopCalled(value, suppress_events, cookie);
			break;
		case 14:
			SetPauseCalled(value, suppress_events, cookie);
			break;
		case 15:
			SetResumeCalled(value, suppress_events, cookie);
			break;
		case 16:
			SetActive(value, suppress_events, cookie);
			break;
		case 17:
			SetStateLoaded(value, suppress_events, cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<ConfigObject>::GetField(int id) const
{
	int real_id = id - ConfigObjectBase::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObjectBase::GetField(id); }
	switch (real_id) {
		case 0:
			return GetShortName();
		case 1:
			return GetIcingadbIdentifier();
		case 2:
			return GetPackage();
		case 3:
			return GetZoneName();
		case 4:
			return GetName();
		case 5:
			return GetVersion();
		case 6:
			return GetTemplates();
		case 7:
			return GetOriginalAttributes();
		case 8:
			return GetExtensions();
		case 9:
			return GetSourceLocation();
		case 10:
			return GetHAMode();
		case 11:
			return GetPaused();
		case 12:
			return GetStartCalled();
		case 13:
			return GetStopCalled();
		case 14:
			return GetPauseCalled();
		case 15:
			return GetResumeCalled();
		case 16:
			return GetActive();
		case 17:
			return GetStateLoaded();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<ConfigObject>::ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	int real_id = id - ConfigObjectBase::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObjectBase::ValidateField(id, lvalue, utils); return; }
	switch (real_id) {
		case 0:
			ValidateShortName(lvalue, utils);
			break;
		case 1:
			ValidateIcingadbIdentifier(lvalue, utils);
			break;
		case 2:
			ValidatePackage(lvalue, utils);
			break;
		case 3:
			ValidateZoneName(lvalue, utils);
			break;
		case 4:
			ValidateName(lvalue, utils);
			break;
		case 5:
			ValidateVersion(lvalue, utils);
			break;
		case 6:
			ValidateTemplates(lvalue, utils);
			break;
		case 7:
			ValidateOriginalAttributes(lvalue, utils);
			break;
		case 8:
			ValidateExtensions(lvalue, utils);
			break;
		case 9:
			ValidateSourceLocation(lvalue, utils);
			break;
		case 10:
			ValidateHAMode(static_cast<Lazy<HAMode> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 11:
			ValidatePaused(lvalue, utils);
			break;
		case 12:
			ValidateStartCalled(lvalue, utils);
			break;
		case 13:
			ValidateStopCalled(lvalue, utils);
			break;
		case 14:
			ValidatePauseCalled(lvalue, utils);
			break;
		case 15:
			ValidateResumeCalled(lvalue, utils);
			break;
		case 16:
			ValidateActive(lvalue, utils);
			break;
		case 17:
			ValidateStateLoaded(lvalue, utils);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<ConfigObject>::NotifyField(int id, const Value& cookie)
{
	int real_id = id - ConfigObjectBase::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObjectBase::NotifyField(id, cookie); return; }
	switch (real_id) {
		case 0:
			NotifyShortName(cookie);
			break;
		case 1:
			NotifyIcingadbIdentifier(cookie);
			break;
		case 2:
			NotifyPackage(cookie);
			break;
		case 3:
			NotifyZoneName(cookie);
			break;
		case 4:
			NotifyName(cookie);
			break;
		case 5:
			NotifyVersion(cookie);
			break;
		case 6:
			NotifyTemplates(cookie);
			break;
		case 7:
			NotifyOriginalAttributes(cookie);
			break;
		case 8:
			NotifyExtensions(cookie);
			break;
		case 9:
			NotifySourceLocation(cookie);
			break;
		case 10:
			NotifyHAMode(cookie);
			break;
		case 11:
			NotifyPaused(cookie);
			break;
		case 12:
			NotifyStartCalled(cookie);
			break;
		case 13:
			NotifyStopCalled(cookie);
			break;
		case 14:
			NotifyPauseCalled(cookie);
			break;
		case 15:
			NotifyResumeCalled(cookie);
			break;
		case 16:
			NotifyActive(cookie);
			break;
		case 17:
			NotifyStateLoaded(cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Object::Ptr ObjectImpl<ConfigObject>::NavigateField(int id) const
{
	int real_id = id - ConfigObjectBase::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObjectBase::NavigateField(id); }
	throw std::runtime_error("Invalid field ID.");
}

String ObjectImpl<ConfigObject>::GetShortName() const
{

			String shortName = m_ShortName.load();
			if (shortName.IsEmpty())
				return GetName();
			else
				return shortName;
		
}

String ObjectImpl<ConfigObject>::GetIcingadbIdentifier() const
{
	return m_IcingadbIdentifier.load();
}

String ObjectImpl<ConfigObject>::GetPackage() const
{
	return m_Package.load();
}

String ObjectImpl<ConfigObject>::GetZoneName() const
{
	return m_ZoneName.load();
}

String ObjectImpl<ConfigObject>::GetName() const
{
	return m_Name.load();
}

double ObjectImpl<ConfigObject>::GetVersion() const
{
	return m_Version.load();
}

Array::Ptr ObjectImpl<ConfigObject>::GetTemplates() const
{
	return m_Templates.load();
}

Dictionary::Ptr ObjectImpl<ConfigObject>::GetOriginalAttributes() const
{
	return m_OriginalAttributes.load();
}

Dictionary::Ptr ObjectImpl<ConfigObject>::GetExtensions() const
{
	return m_Extensions.load();
}

HAMode ObjectImpl<ConfigObject>::GetHAMode() const
{
	return m_HAMode.load();
}

bool ObjectImpl<ConfigObject>::GetPaused() const
{
	return m_Paused.load();
}

bool ObjectImpl<ConfigObject>::GetStartCalled() const
{
	return m_StartCalled.load();
}

bool ObjectImpl<ConfigObject>::GetStopCalled() const
{
	return m_StopCalled.load();
}

bool ObjectImpl<ConfigObject>::GetPauseCalled() const
{
	return m_PauseCalled.load();
}

bool ObjectImpl<ConfigObject>::GetResumeCalled() const
{
	return m_ResumeCalled.load();
}

bool ObjectImpl<ConfigObject>::GetActive() const
{
	return m_Active.load();
}

bool ObjectImpl<ConfigObject>::GetStateLoaded() const
{
	return m_StateLoaded.load();
}

void ObjectImpl<ConfigObject>::SetShortName(const String& value, bool suppress_events, const Value& cookie)
{
	m_ShortName.store(value);
	if (!suppress_events) {
		NotifyShortName(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetIcingadbIdentifier(const String& value, bool suppress_events, const Value& cookie)
{
	m_IcingadbIdentifier.store(value);
	if (!suppress_events) {
		NotifyIcingadbIdentifier(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetPackage(const String& value, bool suppress_events, const Value& cookie)
{
	m_Package.store(value);
	if (!suppress_events) {
		NotifyPackage(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetZoneName(const String& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetZoneName();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_ZoneName.store(value);
	if (!dobj || dobj->IsActive())
		TrackZoneName(oldValue, value);
	if (!suppress_events) {
		NotifyZoneName(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetName(const String& value, bool suppress_events, const Value& cookie)
{
	m_Name.store(value);
	if (!suppress_events) {
		NotifyName(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetVersion(double value, bool suppress_events, const Value& cookie)
{
	m_Version.store(value);
	if (!suppress_events) {
		NotifyVersion(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetTemplates(const Array::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_Templates.store(value);
	if (!suppress_events) {
		NotifyTemplates(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetOriginalAttributes(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_OriginalAttributes.store(value);
	if (!suppress_events) {
		NotifyOriginalAttributes(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetExtensions(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_Extensions.store(value);
	if (!suppress_events) {
		NotifyExtensions(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetSourceLocation(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifySourceLocation(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetHAMode(const HAMode& value, bool suppress_events, const Value& cookie)
{
	m_HAMode.store(value);
	if (!suppress_events) {
		NotifyHAMode(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetPaused(bool value, bool suppress_events, const Value& cookie)
{
	m_Paused.store(value);
	if (!suppress_events) {
		NotifyPaused(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetStartCalled(bool value, bool suppress_events, const Value& cookie)
{
	m_StartCalled.store(value);
	if (!suppress_events) {
		NotifyStartCalled(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetStopCalled(bool value, bool suppress_events, const Value& cookie)
{
	m_StopCalled.store(value);
	if (!suppress_events) {
		NotifyStopCalled(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetPauseCalled(bool value, bool suppress_events, const Value& cookie)
{
	m_PauseCalled.store(value);
	if (!suppress_events) {
		NotifyPauseCalled(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetResumeCalled(bool value, bool suppress_events, const Value& cookie)
{
	m_ResumeCalled.store(value);
	if (!suppress_events) {
		NotifyResumeCalled(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetActive(bool value, bool suppress_events, const Value& cookie)
{
	m_Active.store(value);
	if (!suppress_events) {
		NotifyActive(cookie);
	}

}

void ObjectImpl<ConfigObject>::SetStateLoaded(bool value, bool suppress_events, const Value& cookie)
{
	m_StateLoaded.store(value);
	if (!suppress_events) {
		NotifyStateLoaded(cookie);
	}

}

void ObjectImpl<ConfigObject>::TrackZoneName(const String& oldValue, const String& newValue)
{
	if (!oldValue.IsEmpty())
		DependencyGraph::RemoveDependency(dynamic_cast<ConfigObject*>(this), ConfigObject::GetObject("Zone", oldValue).get());
	if (!newValue.IsEmpty())
		DependencyGraph::AddDependency(dynamic_cast<ConfigObject*>(this), ConfigObject::GetObject("Zone", newValue).get());
}

void ObjectImpl<ConfigObject>::Start(bool runtimeCreated)
{
	ConfigObjectBase::Start(runtimeCreated);

	TrackZoneName(Empty, GetZoneName());
}

void ObjectImpl<ConfigObject>::Stop(bool runtimeRemoved)
{
	ConfigObjectBase::Stop(runtimeRemoved);

	TrackZoneName(GetZoneName(), Empty);
}

void ObjectImpl<ConfigObject>::NotifyShortName(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnShortNameChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyIcingadbIdentifier(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnIcingadbIdentifierChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyPackage(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnPackageChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyZoneName(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnZoneNameChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyName(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnNameChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyVersion(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnVersionChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyTemplates(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnTemplatesChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyOriginalAttributes(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnOriginalAttributesChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyExtensions(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnExtensionsChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifySourceLocation(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnSourceLocationChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyHAMode(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnHAModeChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyPaused(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnPausedChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyStartCalled(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStartCalledChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyStopCalled(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStopCalledChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyPauseCalled(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnPauseCalledChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyResumeCalled(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnResumeCalledChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyActive(const Value& cookie)
{
	OnActiveChanged(static_cast<ConfigObject *>(this), cookie);
}

void ObjectImpl<ConfigObject>::NotifyStateLoaded(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStateLoadedChanged(static_cast<ConfigObject *>(this), cookie);
}

String ObjectImpl<ConfigObject>::GetDefaultShortName() const
{
	return String();
}

String ObjectImpl<ConfigObject>::GetDefaultIcingadbIdentifier() const
{
	return String();
}

String ObjectImpl<ConfigObject>::GetDefaultPackage() const
{
	return String();
}

String ObjectImpl<ConfigObject>::GetDefaultZoneName() const
{
	return String();
}

String ObjectImpl<ConfigObject>::GetDefaultName() const
{
	return String();
}

double ObjectImpl<ConfigObject>::GetDefaultVersion() const
{
	 return 0; 
}

Array::Ptr ObjectImpl<ConfigObject>::GetDefaultTemplates() const
{
	return Array::Ptr();
}

Dictionary::Ptr ObjectImpl<ConfigObject>::GetDefaultOriginalAttributes() const
{
	return Dictionary::Ptr();
}

Dictionary::Ptr ObjectImpl<ConfigObject>::GetDefaultExtensions() const
{
	return Dictionary::Ptr();
}

Dictionary::Ptr ObjectImpl<ConfigObject>::GetDefaultSourceLocation() const
{
	return Dictionary::Ptr();
}

HAMode ObjectImpl<ConfigObject>::GetDefaultHAMode() const
{
	return HAMode();
}

bool ObjectImpl<ConfigObject>::GetDefaultPaused() const
{
	 return true; 
}

bool ObjectImpl<ConfigObject>::GetDefaultStartCalled() const
{
	return bool();
}

bool ObjectImpl<ConfigObject>::GetDefaultStopCalled() const
{
	return bool();
}

bool ObjectImpl<ConfigObject>::GetDefaultPauseCalled() const
{
	return bool();
}

bool ObjectImpl<ConfigObject>::GetDefaultResumeCalled() const
{
	return bool();
}

bool ObjectImpl<ConfigObject>::GetDefaultActive() const
{
	return bool();
}

bool ObjectImpl<ConfigObject>::GetDefaultStateLoaded() const
{
	return bool();
}


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnShortNameChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnIcingadbIdentifierChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnPackageChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnZoneNameChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnNameChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnVersionChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnTemplatesChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnOriginalAttributesChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnExtensionsChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnSourceLocationChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnHAModeChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnPausedChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnStartCalledChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnStopCalledChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnPauseCalledChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnResumeCalledChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnActiveChanged;


boost::signals2::signal<void (const intrusive_ptr<ConfigObject>&, const Value&)> ObjectImpl<ConfigObject>::OnStateLoadedChanged;

void ObjectImpl<ConfigObject>::ValidateActive(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateActive(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateExtensions(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateExtensions(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateHAMode(const Lazy<HAMode>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateHAMode(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateIcingadbIdentifier(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateIcingadbIdentifier(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateName(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateName(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateOriginalAttributes(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateOriginalAttributes(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidatePackage(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidatePackage(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidatePauseCalled(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidatePauseCalled(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidatePaused(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidatePaused(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateResumeCalled(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateResumeCalled(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateShortName(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateShortName(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateSourceLocation(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateSourceLocation(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateStartCalled(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateStartCalled(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateStateLoaded(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateStateLoaded(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateStopCalled(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateStopCalled(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateTemplates(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateTemplates(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateVersion(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateVersion(lvalue, utils);
}

void ObjectImpl<ConfigObject>::ValidateZoneName(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateZoneName(lvalue, utils);
}

}
#ifdef _MSC_VER
#pragma warning ( pop )
#endif /* _MSC_VER */
