#ifndef FUNCTION_TI
#define FUNCTION_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class Function;

template<>
struct TypeHelper<Function, false>
{
	static ObjectFactory GetFactory()
	{
		return nullptr;
	}
};

template<>
class TypeImpl<Function> : public Type
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<Function>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<Function> : public Object
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<Function>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateName(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateArguments(const Lazy<Array::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateSideEffectFree(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateDeprecated(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<Function>();
public:
	~ObjectImpl<Function>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetName() const;
public:
	Array::Ptr GetArguments() const;
public:
	bool GetSideEffectFree() const;
public:
	bool GetDeprecated() const;
public:
	void SetName(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetArguments(const Array::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSideEffectFree(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetDeprecated(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyName(const Value& cookie = Empty);
public:
	virtual void NotifyArguments(const Value& cookie = Empty);
public:
	virtual void NotifySideEffectFree(const Value& cookie = Empty);
public:
	virtual void NotifyDeprecated(const Value& cookie = Empty);
private:
	inline String GetDefaultName() const;
private:
	inline Array::Ptr GetDefaultArguments() const;
private:
	inline bool GetDefaultSideEffectFree() const;
private:
	inline bool GetDefaultDeprecated() const;
protected:
	virtual void ValidateName(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateArguments(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSideEffectFree(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateDeprecated(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Name;
	AtomicOrLocked<Array::Ptr> m_Arguments;
	AtomicOrLocked<bool> m_SideEffectFree;
	AtomicOrLocked<bool> m_Deprecated;
public:
	static boost::signals2::signal<void (const intrusive_ptr<Function>&, const Value&)> OnNameChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Function>&, const Value&)> OnArgumentsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Function>&, const Value&)> OnSideEffectFreeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Function>&, const Value&)> OnDeprecatedChanged;
};

}
#endif /* FUNCTION_TI */
