#ifndef LOGGER_TI
#define LOGGER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class Logger;

template<>
struct TypeHelper<Logger, false>
{
	static ObjectFactory GetFactory()
	{
		return nullptr;
	}
};

template<>
class TypeImpl<Logger> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<Logger>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<Logger> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<Logger>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateSeverity(const Lazy<String>& value, const ValidationUtils& utils);
public:
	ObjectImpl<Logger>();
public:
	~ObjectImpl<Logger>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetSeverity() const;
public:
	virtual void SetSeverity(const String& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifySeverity(const Value& cookie = Empty);
private:
	inline String GetDefaultSeverity() const;
protected:
	virtual void ValidateSeverity(const Lazy<String>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Severity;
public:
	static boost::signals2::signal<void (const intrusive_ptr<Logger>&, const Value&)> OnSeverityChanged;
};

}
#endif /* LOGGER_TI */
