#ifndef CHECKERCOMPONENT_TI
#define CHECKERCOMPONENT_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class CheckerComponent;

template<>
class TypeImpl<CheckerComponent> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<CheckerComponent>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<CheckerComponent> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<CheckerComponent>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateConcurrentChecks(const Lazy<int>& value, const ValidationUtils& utils);
public:
	ObjectImpl<CheckerComponent>();
public:
	~ObjectImpl<CheckerComponent>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	int GetConcurrentChecks() const;
public:
	void SetConcurrentChecks(int value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyConcurrentChecks(const Value& cookie = Empty);
private:
	inline int GetDefaultConcurrentChecks() const;
protected:
	virtual void ValidateConcurrentChecks(const Lazy<int>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<int> m_ConcurrentChecks;
public:
	static boost::signals2::signal<void (const intrusive_ptr<CheckerComponent>&, const Value&)> OnConcurrentChecksChanged;
};

}
#endif /* CHECKERCOMPONENT_TI */
