#include "base/exception.hpp"
#include "base/objectlock.hpp"
#include "base/utility.hpp"
#include "base/convert.hpp"
#include "base/debug.hpp"
#include "base/dependencygraph.hpp"
#include "base/logger.hpp"
#include "base/function.hpp"
#include "base/configobject.hpp"
#include "base/configtype.hpp"
#ifdef _MSC_VER
#pragma warning( push )
#pragma warning( disable : 4244 )
#pragma warning( disable : 4800 )
#endif /* _MSC_VER */

namespace icinga
{

TypeImpl<CompatLogger>::TypeImpl()
{ }

TypeImpl<CompatLogger>::~TypeImpl()
{ }

String TypeImpl<CompatLogger>::GetName() const
{
	return "CompatLogger";
}

int TypeImpl<CompatLogger>::GetAttributes() const
{
	return 0;
}

Type::Ptr TypeImpl<CompatLogger>::GetBaseType() const
{
	return ConfigObject::TypeInstance;
}

int TypeImpl<CompatLogger>::GetFieldId(const String& name) const
{
	int offset = ConfigObject::TypeInstance->GetFieldCount();

	switch (static_cast<int>(Utility::SDBM(name, 1))) {
		case 108:
			if (name == "log_dir")
				return offset + 0;

			break;
		case 114:
			if (name == "rotation_method")
				return offset + 1;

			break;
	}

	return ConfigObject::TypeInstance->GetFieldId(name);
}

Field TypeImpl<CompatLogger>::GetFieldInfo(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount();
	if (real_id < 0) { return ConfigObject::TypeInstance->GetFieldInfo(id); }
	switch (real_id) {
		case 0:
			return {0, "String", "log_dir", "log_dir", nullptr, 2, 0};
		case 1:
			return {1, "String", "rotation_method", "rotation_method", nullptr, 2, 0};
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

int TypeImpl<CompatLogger>::GetFieldCount() const
{
	return 2 + ConfigObject::TypeInstance->GetFieldCount();
}

ObjectFactory TypeImpl<CompatLogger>::GetFactory() const
{
	return TypeHelper<CompatLogger, false>::GetFactory();
}

int TypeImpl<CompatLogger>::GetActivationPriority() const
{
	return 100;
}

void TypeImpl<CompatLogger>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	int real_id = fieldId - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::TypeInstance->RegisterAttributeHandler(fieldId, callback); return; }
	switch (real_id) {
		case 0:
			ObjectImpl<CompatLogger>::OnLogDirChanged.connect(callback);
			break;
		case 1:
			ObjectImpl<CompatLogger>::OnRotationMethodChanged.connect(callback);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<CompatLogger>::Validate(int types, const ValidationUtils& utils)
{
	ConfigObject::Validate(types, utils);

	if (2 & types)
		ValidateLogDir(Lazy<String>([this]() { return GetLogDir(); }), utils);
	if (2 & types)
		ValidateRotationMethod(Lazy<String>([this]() { return GetRotationMethod(); }), utils);
}

void ObjectImpl<CompatLogger>::SimpleValidateLogDir(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<CompatLogger>::SimpleValidateRotationMethod(const Lazy<String>& value, const ValidationUtils& utils)
{
}

ObjectImpl<CompatLogger>::ObjectImpl()
{
	SetLogDir(GetDefaultLogDir(), true);
	SetRotationMethod(GetDefaultRotationMethod(), true);
}

ObjectImpl<CompatLogger>::~ObjectImpl()
{ }

void ObjectImpl<CompatLogger>::SetField(int id, const Value& value, bool suppress_events, const Value& cookie)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::SetField(id, value, suppress_events, cookie); return; }
	switch (real_id) {
		case 0:
			SetLogDir(value, suppress_events, cookie);
			break;
		case 1:
			SetRotationMethod(value, suppress_events, cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<CompatLogger>::GetField(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObject::GetField(id); }
	switch (real_id) {
		case 0:
			return GetLogDir();
		case 1:
			return GetRotationMethod();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<CompatLogger>::ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::ValidateField(id, lvalue, utils); return; }
	switch (real_id) {
		case 0:
			ValidateLogDir(lvalue, utils);
			break;
		case 1:
			ValidateRotationMethod(lvalue, utils);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<CompatLogger>::NotifyField(int id, const Value& cookie)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::NotifyField(id, cookie); return; }
	switch (real_id) {
		case 0:
			NotifyLogDir(cookie);
			break;
		case 1:
			NotifyRotationMethod(cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Object::Ptr ObjectImpl<CompatLogger>::NavigateField(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObject::NavigateField(id); }
	throw std::runtime_error("Invalid field ID.");
}

String ObjectImpl<CompatLogger>::GetLogDir() const
{
	return m_LogDir.load();
}

String ObjectImpl<CompatLogger>::GetRotationMethod() const
{
	return m_RotationMethod.load();
}

void ObjectImpl<CompatLogger>::SetLogDir(const String& value, bool suppress_events, const Value& cookie)
{
	m_LogDir.store(value);
	if (!suppress_events) {
		NotifyLogDir(cookie);
	}

}

void ObjectImpl<CompatLogger>::SetRotationMethod(const String& value, bool suppress_events, const Value& cookie)
{
	m_RotationMethod.store(value);
	if (!suppress_events) {
		NotifyRotationMethod(cookie);
	}

}

void ObjectImpl<CompatLogger>::NotifyLogDir(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLogDirChanged(static_cast<CompatLogger *>(this), cookie);
}

void ObjectImpl<CompatLogger>::NotifyRotationMethod(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnRotationMethodChanged(static_cast<CompatLogger *>(this), cookie);
}

String ObjectImpl<CompatLogger>::GetDefaultLogDir() const
{
	 return Configuration::LogDir + "/compat"; 
}

String ObjectImpl<CompatLogger>::GetDefaultRotationMethod() const
{
	 return "HOURLY"; 
}


boost::signals2::signal<void (const intrusive_ptr<CompatLogger>&, const Value&)> ObjectImpl<CompatLogger>::OnLogDirChanged;


boost::signals2::signal<void (const intrusive_ptr<CompatLogger>&, const Value&)> ObjectImpl<CompatLogger>::OnRotationMethodChanged;

void ObjectImpl<CompatLogger>::ValidateLogDir(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLogDir(lvalue, utils);
}

void ObjectImpl<CompatLogger>::ValidateRotationMethod(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateRotationMethod(lvalue, utils);
}

}
#ifdef _MSC_VER
#pragma warning ( pop )
#endif /* _MSC_VER */
