#ifndef COMPATLOGGER_TI
#define COMPATLOGGER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

#include "base/application.hpp"

namespace icinga
{

class CompatLogger;

template<>
class TypeImpl<CompatLogger> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<CompatLogger>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<CompatLogger> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<CompatLogger>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateLogDir(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateRotationMethod(const Lazy<String>& value, const ValidationUtils& utils);
public:
	ObjectImpl<CompatLogger>();
public:
	~ObjectImpl<CompatLogger>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetLogDir() const;
public:
	String GetRotationMethod() const;
public:
	void SetLogDir(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetRotationMethod(const String& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyLogDir(const Value& cookie = Empty);
public:
	virtual void NotifyRotationMethod(const Value& cookie = Empty);
private:
	inline String GetDefaultLogDir() const;
private:
	inline String GetDefaultRotationMethod() const;
protected:
	virtual void ValidateLogDir(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateRotationMethod(const Lazy<String>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_LogDir;
	AtomicOrLocked<String> m_RotationMethod;
public:
	static boost::signals2::signal<void (const intrusive_ptr<CompatLogger>&, const Value&)> OnLogDirChanged;
	static boost::signals2::signal<void (const intrusive_ptr<CompatLogger>&, const Value&)> OnRotationMethodChanged;
};

}
#endif /* COMPATLOGGER_TI */
