#include "base/exception.hpp"
#include "base/objectlock.hpp"
#include "base/utility.hpp"
#include "base/convert.hpp"
#include "base/debug.hpp"
#include "base/dependencygraph.hpp"
#include "base/logger.hpp"
#include "base/function.hpp"
#include "base/configobject.hpp"
#include "base/configtype.hpp"
#ifdef _MSC_VER
#pragma warning( push )
#pragma warning( disable : 4244 )
#pragma warning( disable : 4800 )
#endif /* _MSC_VER */

namespace icinga
{

TypeImpl<DbConnection>::TypeImpl()
{ }

TypeImpl<DbConnection>::~TypeImpl()
{ }

String TypeImpl<DbConnection>::GetName() const
{
	return "DbConnection";
}

int TypeImpl<DbConnection>::GetAttributes() const
{
	return 1;
}

Type::Ptr TypeImpl<DbConnection>::GetBaseType() const
{
	return ConfigObject::TypeInstance;
}

int TypeImpl<DbConnection>::GetFieldId(const String& name) const
{
	int offset = ConfigObject::TypeInstance->GetFieldCount();

	switch (static_cast<int>(Utility::SDBM(name, 1))) {
		case 99:
			if (name == "categories")
				return offset + 4;
			if (name == "cleanup")
				return offset + 5;
			if (name == "categories_filter_real")
				return offset + 6;
			if (name == "connected")
				return offset + 8;

			break;
		case 101:
			if (name == "enable_ha")
				return offset + 7;

			break;
		case 102:
			if (name == "failover_timeout")
				return offset + 2;

			break;
		case 108:
			if (name == "last_failover")
				return offset + 3;

			break;
		case 115:
			if (name == "schema_version")
				return offset + 1;
			if (name == "should_connect")
				return offset + 9;

			break;
		case 116:
			if (name == "table_prefix")
				return offset + 0;

			break;
	}

	return ConfigObject::TypeInstance->GetFieldId(name);
}

Field TypeImpl<DbConnection>::GetFieldInfo(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount();
	if (real_id < 0) { return ConfigObject::TypeInstance->GetFieldInfo(id); }
	switch (real_id) {
		case 0:
			return {0, "String", "table_prefix", "table_prefix", nullptr, 2, 0};
		case 1:
			return {1, "String", "schema_version", "schema_version", nullptr, 1025, 0};
		case 2:
			return {2, "Number", "failover_timeout", "failover_timeout", nullptr, 2, 0};
		case 3:
			return {3, "Number", "last_failover", "last_failover", nullptr, 1028, 0};
		case 4:
			return {4, "Array", "categories", "categories", nullptr, 2, 0};
		case 5:
			return {5, "Dictionary", "cleanup", "cleanup", nullptr, 258, 0};
		case 6:
			return {6, "Number", "categories_filter_real", "categories_filter_real", nullptr, 3073, 0};
		case 7:
			return {7, "Boolean", "enable_ha", "enable_ha", nullptr, 2, 0};
		case 8:
			return {8, "Boolean", "connected", "connected", nullptr, 1025, 0};
		case 9:
			return {9, "Boolean", "should_connect", "should_connect", nullptr, 1025, 0};
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

int TypeImpl<DbConnection>::GetFieldCount() const
{
	return 10 + ConfigObject::TypeInstance->GetFieldCount();
}

ObjectFactory TypeImpl<DbConnection>::GetFactory() const
{
	return TypeHelper<DbConnection, false>::GetFactory();
}

int TypeImpl<DbConnection>::GetActivationPriority() const
{
	return 0;
}

void TypeImpl<DbConnection>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	int real_id = fieldId - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::TypeInstance->RegisterAttributeHandler(fieldId, callback); return; }
	switch (real_id) {
		case 0:
			ObjectImpl<DbConnection>::OnTablePrefixChanged.connect(callback);
			break;
		case 1:
			ObjectImpl<DbConnection>::OnSchemaVersionChanged.connect(callback);
			break;
		case 2:
			ObjectImpl<DbConnection>::OnFailoverTimeoutChanged.connect(callback);
			break;
		case 3:
			ObjectImpl<DbConnection>::OnLastFailoverChanged.connect(callback);
			break;
		case 4:
			ObjectImpl<DbConnection>::OnCategoriesChanged.connect(callback);
			break;
		case 5:
			ObjectImpl<DbConnection>::OnCleanupChanged.connect(callback);
			break;
		case 6:
			ObjectImpl<DbConnection>::OnCategoryFilterChanged.connect(callback);
			break;
		case 7:
			ObjectImpl<DbConnection>::OnEnableHaChanged.connect(callback);
			break;
		case 8:
			ObjectImpl<DbConnection>::OnConnectedChanged.connect(callback);
			break;
		case 9:
			ObjectImpl<DbConnection>::OnShouldConnectChanged.connect(callback);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<DbConnection>::Validate(int types, const ValidationUtils& utils)
{
	ConfigObject::Validate(types, utils);

	if (2 & types)
		ValidateTablePrefix(Lazy<String>([this]() { return GetTablePrefix(); }), utils);
	if (1 & types)
		ValidateSchemaVersion(Lazy<String>([this]() { return GetSchemaVersion(); }), utils);
	if (2 & types)
		ValidateFailoverTimeout(Lazy<double>([this]() { return GetFailoverTimeout(); }), utils);
	if (4 & types)
		ValidateLastFailover(Lazy<double>([this]() { return GetLastFailover(); }), utils);
	if (2 & types)
		ValidateCategories(Lazy<Array::Ptr>([this]() { return GetCategories(); }), utils);
	if (2 & types)
		ValidateCleanup(Lazy<Dictionary::Ptr>([this]() { return GetCleanup(); }), utils);
	if (1 & types)
		ValidateCategoryFilter(Lazy<int>([this]() { return GetCategoryFilter(); }), utils);
	if (2 & types)
		ValidateEnableHa(Lazy<bool>([this]() { return GetEnableHa(); }), utils);
	if (1 & types)
		ValidateConnected(Lazy<bool>([this]() { return GetConnected(); }), utils);
	if (1 & types)
		ValidateShouldConnect(Lazy<bool>([this]() { return GetShouldConnect(); }), utils);
}

void ObjectImpl<DbConnection>::SimpleValidateTablePrefix(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateSchemaVersion(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateFailoverTimeout(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateLastFailover(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateCategories(const Lazy<Array::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateCleanup(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
	if (!value())
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "cleanup" }, "Attribute must not be empty."));

}

void ObjectImpl<DbConnection>::SimpleValidateCategoryFilter(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateEnableHa(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateConnected(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<DbConnection>::SimpleValidateShouldConnect(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

ObjectImpl<DbConnection>::ObjectImpl()
{
	SetTablePrefix(GetDefaultTablePrefix(), true);
	SetSchemaVersion(GetDefaultSchemaVersion(), true);
	SetFailoverTimeout(GetDefaultFailoverTimeout(), true);
	SetLastFailover(GetDefaultLastFailover(), true);
	SetCategories(GetDefaultCategories(), true);
	SetCleanup(GetDefaultCleanup(), true);
	SetCategoryFilter(GetDefaultCategoryFilter(), true);
	SetEnableHa(GetDefaultEnableHa(), true);
	SetConnected(GetDefaultConnected(), true);
	SetShouldConnect(GetDefaultShouldConnect(), true);
}

ObjectImpl<DbConnection>::~ObjectImpl()
{ }

void ObjectImpl<DbConnection>::SetField(int id, const Value& value, bool suppress_events, const Value& cookie)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::SetField(id, value, suppress_events, cookie); return; }
	switch (real_id) {
		case 0:
			SetTablePrefix(value, suppress_events, cookie);
			break;
		case 1:
			SetSchemaVersion(value, suppress_events, cookie);
			break;
		case 2:
			SetFailoverTimeout(value, suppress_events, cookie);
			break;
		case 3:
			SetLastFailover(value, suppress_events, cookie);
			break;
		case 4:
			SetCategories(value, suppress_events, cookie);
			break;
		case 5:
			SetCleanup(value, suppress_events, cookie);
			break;
		case 6:
			SetCategoryFilter(value, suppress_events, cookie);
			break;
		case 7:
			SetEnableHa(value, suppress_events, cookie);
			break;
		case 8:
			SetConnected(value, suppress_events, cookie);
			break;
		case 9:
			SetShouldConnect(value, suppress_events, cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<DbConnection>::GetField(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObject::GetField(id); }
	switch (real_id) {
		case 0:
			return GetTablePrefix();
		case 1:
			return GetSchemaVersion();
		case 2:
			return GetFailoverTimeout();
		case 3:
			return GetLastFailover();
		case 4:
			return GetCategories();
		case 5:
			return GetCleanup();
		case 6:
			return GetCategoryFilter();
		case 7:
			return GetEnableHa();
		case 8:
			return GetConnected();
		case 9:
			return GetShouldConnect();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<DbConnection>::ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::ValidateField(id, lvalue, utils); return; }
	switch (real_id) {
		case 0:
			ValidateTablePrefix(lvalue, utils);
			break;
		case 1:
			ValidateSchemaVersion(lvalue, utils);
			break;
		case 2:
			ValidateFailoverTimeout(lvalue, utils);
			break;
		case 3:
			ValidateLastFailover(lvalue, utils);
			break;
		case 4:
			ValidateCategories(lvalue, utils);
			break;
		case 5:
			ValidateCleanup(lvalue, utils);
			break;
		case 6:
			ValidateCategoryFilter(lvalue, utils);
			break;
		case 7:
			ValidateEnableHa(lvalue, utils);
			break;
		case 8:
			ValidateConnected(lvalue, utils);
			break;
		case 9:
			ValidateShouldConnect(lvalue, utils);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<DbConnection>::NotifyField(int id, const Value& cookie)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::NotifyField(id, cookie); return; }
	switch (real_id) {
		case 0:
			NotifyTablePrefix(cookie);
			break;
		case 1:
			NotifySchemaVersion(cookie);
			break;
		case 2:
			NotifyFailoverTimeout(cookie);
			break;
		case 3:
			NotifyLastFailover(cookie);
			break;
		case 4:
			NotifyCategories(cookie);
			break;
		case 5:
			NotifyCleanup(cookie);
			break;
		case 6:
			NotifyCategoryFilter(cookie);
			break;
		case 7:
			NotifyEnableHa(cookie);
			break;
		case 8:
			NotifyConnected(cookie);
			break;
		case 9:
			NotifyShouldConnect(cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Object::Ptr ObjectImpl<DbConnection>::NavigateField(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObject::NavigateField(id); }
	throw std::runtime_error("Invalid field ID.");
}

String ObjectImpl<DbConnection>::GetTablePrefix() const
{
	return m_TablePrefix.load();
}

String ObjectImpl<DbConnection>::GetSchemaVersion() const
{
	return m_SchemaVersion.load();
}

double ObjectImpl<DbConnection>::GetFailoverTimeout() const
{
	return m_FailoverTimeout.load();
}

double ObjectImpl<DbConnection>::GetLastFailover() const
{
	return m_LastFailover.load();
}

Array::Ptr ObjectImpl<DbConnection>::GetCategories() const
{
	return m_Categories.load();
}

Dictionary::Ptr ObjectImpl<DbConnection>::GetCleanup() const
{
	return m_Cleanup.load();
}

int ObjectImpl<DbConnection>::GetCategoryFilter() const
{
	return m_CategoryFilter.load();
}

bool ObjectImpl<DbConnection>::GetEnableHa() const
{
	return m_EnableHa.load();
}

bool ObjectImpl<DbConnection>::GetConnected() const
{
	return m_Connected.load();
}

bool ObjectImpl<DbConnection>::GetShouldConnect() const
{
	return m_ShouldConnect.load();
}

void ObjectImpl<DbConnection>::SetTablePrefix(const String& value, bool suppress_events, const Value& cookie)
{
	m_TablePrefix.store(value);
	if (!suppress_events) {
		NotifyTablePrefix(cookie);
	}

}

void ObjectImpl<DbConnection>::SetSchemaVersion(const String& value, bool suppress_events, const Value& cookie)
{
	m_SchemaVersion.store(value);
	if (!suppress_events) {
		NotifySchemaVersion(cookie);
	}

}

void ObjectImpl<DbConnection>::SetFailoverTimeout(double value, bool suppress_events, const Value& cookie)
{
	m_FailoverTimeout.store(value);
	if (!suppress_events) {
		NotifyFailoverTimeout(cookie);
	}

}

void ObjectImpl<DbConnection>::SetLastFailover(double value, bool suppress_events, const Value& cookie)
{
	m_LastFailover.store(value);
	if (!suppress_events) {
		NotifyLastFailover(cookie);
	}

}

void ObjectImpl<DbConnection>::SetCategories(const Array::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_Categories.store(value);
	if (!suppress_events) {
		NotifyCategories(cookie);
	}

}

void ObjectImpl<DbConnection>::SetCleanup(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_Cleanup.store(value);
	if (!suppress_events) {
		NotifyCleanup(cookie);
	}

}

void ObjectImpl<DbConnection>::SetCategoryFilter(int value, bool suppress_events, const Value& cookie)
{
	m_CategoryFilter.store(value);
	if (!suppress_events) {
		NotifyCategoryFilter(cookie);
	}

}

void ObjectImpl<DbConnection>::SetEnableHa(bool value, bool suppress_events, const Value& cookie)
{
	m_EnableHa.store(value);
	if (!suppress_events) {
		NotifyEnableHa(cookie);
	}

}

void ObjectImpl<DbConnection>::SetConnected(bool value, bool suppress_events, const Value& cookie)
{
	m_Connected.store(value);
	if (!suppress_events) {
		NotifyConnected(cookie);
	}

}

void ObjectImpl<DbConnection>::SetShouldConnect(bool value, bool suppress_events, const Value& cookie)
{
	m_ShouldConnect.store(value);
	if (!suppress_events) {
		NotifyShouldConnect(cookie);
	}

}

void ObjectImpl<DbConnection>::NotifyTablePrefix(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnTablePrefixChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifySchemaVersion(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnSchemaVersionChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyFailoverTimeout(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFailoverTimeoutChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyLastFailover(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastFailoverChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyCategories(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCategoriesChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyCleanup(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCleanupChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyCategoryFilter(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCategoryFilterChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyEnableHa(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnableHaChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyConnected(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnConnectedChanged(static_cast<DbConnection *>(this), cookie);
}

void ObjectImpl<DbConnection>::NotifyShouldConnect(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnShouldConnectChanged(static_cast<DbConnection *>(this), cookie);
}

String ObjectImpl<DbConnection>::GetDefaultTablePrefix() const
{
	 return "icinga_"; 
}

String ObjectImpl<DbConnection>::GetDefaultSchemaVersion() const
{
	return String();
}

double ObjectImpl<DbConnection>::GetDefaultFailoverTimeout() const
{
	 return 30; 
}

double ObjectImpl<DbConnection>::GetDefaultLastFailover() const
{
	return double();
}

Array::Ptr ObjectImpl<DbConnection>::GetDefaultCategories() const
{
	
			return new Array({
				"DbCatConfig",
				"DbCatState",
				"DbCatAcknowledgement",
				"DbCatComment",
				"DbCatDowntime",
				"DbCatEventHandler",
				"DbCatFlapping",
				"DbCatNotification",
				"DbCatProgramStatus",
				"DbCatRetention",
				"DbCatStateHistory"
			});
		
}

Dictionary::Ptr ObjectImpl<DbConnection>::GetDefaultCleanup() const
{
	 return new Dictionary(); 
}

int ObjectImpl<DbConnection>::GetDefaultCategoryFilter() const
{
	return int();
}

bool ObjectImpl<DbConnection>::GetDefaultEnableHa() const
{
	 return true; 
}

bool ObjectImpl<DbConnection>::GetDefaultConnected() const
{
	return bool();
}

bool ObjectImpl<DbConnection>::GetDefaultShouldConnect() const
{
	 return true; 
}


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnTablePrefixChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnSchemaVersionChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnFailoverTimeoutChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnLastFailoverChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnCategoriesChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnCleanupChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnCategoryFilterChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnEnableHaChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnConnectedChanged;


boost::signals2::signal<void (const intrusive_ptr<DbConnection>&, const Value&)> ObjectImpl<DbConnection>::OnShouldConnectChanged;

static void TIValidateDbConnection_1(const intrusive_ptr<ObjectImpl<DbConnection> >& object, const String& key, const Value& value, std::vector<String>& location, const ValidationUtils& utils)
{
	bool known_attribute = false;
	do {
		if (key != "acknowledgements_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "commenthistory_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "contactnotifications_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "contactnotificationmethods_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "downtimehistory_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "eventhandlers_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "externalcommands_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "flappinghistory_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "hostchecks_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "logentries_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "notifications_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "processevents_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "statehistory_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "servicechecks_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	do {
		if (key != "systemcommands_age")
			break;
		known_attribute = true;
		try {
			Convert::ToDouble(value);
			return;
		} catch (...) { }
	} while (0);

	if (!known_attribute)
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_pointer_cast<ConfigObject>(object), location, "Invalid attribute: " + key));
	else
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_pointer_cast<ConfigObject>(object), location, "Invalid type."));
}

static void TIValidateDbConnection_2(const intrusive_ptr<ObjectImpl<DbConnection> >& object, const String& key, const Value& value, std::vector<String>& location, const ValidationUtils& utils)
{
	bool known_attribute = false;
	do {
		known_attribute = true;
		if (value.IsEmpty() || value.IsScalar())
			return;
	} while (0);

	if (!known_attribute)
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_pointer_cast<ConfigObject>(object), location, "Invalid attribute: " + key));
	else
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_pointer_cast<ConfigObject>(object), location, "Invalid type."));
}

static void TIValidateDbConnectionCategories(const intrusive_ptr<ObjectImpl<DbConnection> >& object, const Array::Ptr& value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (!value)
		return;

	do {
		const Array::Ptr& arr = value;
		Array::SizeType anum = 0;
		{
			ObjectLock olock(arr);
			for (const Value& avalue : arr) {
				String akey = Convert::ToString(anum);
				location.emplace_back(akey);
				TIValidateDbConnection_2(object, akey, avalue, location, utils);
				location.pop_back();
				anum++;
			}
		}
		return;
	} while (0);

}

static void TIValidateDbConnectionCategoryFilter(const intrusive_ptr<ObjectImpl<DbConnection> >& object, int value, std::vector<String>& location, const ValidationUtils& utils)
{
}

static void TIValidateDbConnectionCleanup(const intrusive_ptr<ObjectImpl<DbConnection> >& object, const Dictionary::Ptr& value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (!value)
		return;

	do {
		const Dictionary::Ptr& dict = value;
		{
			ObjectLock olock(dict);
			for (const Dictionary::Pair& kv : dict) {
				const String& akey = kv.first;
				const Value& avalue = kv.second;
				location.emplace_back(akey);
				TIValidateDbConnection_1(object, akey, avalue, location, utils);
				location.pop_back();
			}
		}
		return;
	} while (0);

}

static void TIValidateDbConnectionConnected(const intrusive_ptr<ObjectImpl<DbConnection> >& object, bool value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (!value)
		return;

}

static void TIValidateDbConnectionEnableHa(const intrusive_ptr<ObjectImpl<DbConnection> >& object, bool value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (!value)
		return;

}

static void TIValidateDbConnectionFailoverTimeout(const intrusive_ptr<ObjectImpl<DbConnection> >& object, double value, std::vector<String>& location, const ValidationUtils& utils)
{
}

static void TIValidateDbConnectionLastFailover(const intrusive_ptr<ObjectImpl<DbConnection> >& object, double value, std::vector<String>& location, const ValidationUtils& utils)
{
}

static void TIValidateDbConnectionSchemaVersion(const intrusive_ptr<ObjectImpl<DbConnection> >& object, const String& value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (value.IsEmpty())
		return;

}

static void TIValidateDbConnectionShouldConnect(const intrusive_ptr<ObjectImpl<DbConnection> >& object, bool value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (!value)
		return;

}

static void TIValidateDbConnectionTablePrefix(const intrusive_ptr<ObjectImpl<DbConnection> >& object, const String& value, std::vector<String>& location, const ValidationUtils& utils)
{
	if (value.IsEmpty())
		return;

}

void ObjectImpl<DbConnection>::ValidateCategories(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCategories(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("categories");
	TIValidateDbConnectionCategories(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateCategoryFilter(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCategoryFilter(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("categories_filter_real");
	TIValidateDbConnectionCategoryFilter(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateCleanup(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCleanup(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("cleanup");
	TIValidateDbConnectionCleanup(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateConnected(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateConnected(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("connected");
	TIValidateDbConnectionConnected(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateEnableHa(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnableHa(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("enable_ha");
	TIValidateDbConnectionEnableHa(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateFailoverTimeout(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFailoverTimeout(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("failover_timeout");
	TIValidateDbConnectionFailoverTimeout(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateLastFailover(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastFailover(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("last_failover");
	TIValidateDbConnectionLastFailover(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateSchemaVersion(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateSchemaVersion(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("schema_version");
	TIValidateDbConnectionSchemaVersion(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateShouldConnect(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateShouldConnect(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("should_connect");
	TIValidateDbConnectionShouldConnect(this, lvalue(), location, utils);
	location.pop_back();
}

void ObjectImpl<DbConnection>::ValidateTablePrefix(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateTablePrefix(lvalue, utils);
	std::vector<String> location;
	location.emplace_back("table_prefix");
	TIValidateDbConnectionTablePrefix(this, lvalue(), location, utils);
	location.pop_back();
}

}
#ifdef _MSC_VER
#pragma warning ( pop )
#endif /* _MSC_VER */
