#ifndef IDOMYSQLCONNECTION_TI
#define IDOMYSQLCONNECTION_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "db_ido/dbconnection.hpp"

namespace icinga
{

class IdoMysqlConnection;

template<>
class TypeImpl<IdoMysqlConnection> : public TypeImpl<DbConnection>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<IdoMysqlConnection>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<IdoMysqlConnection> : public DbConnection
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<IdoMysqlConnection>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSocketPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateUser(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePassword(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateDatabase(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslKey(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslCert(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslCa(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslCapath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslCipher(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateInstanceName(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateInstanceDescription(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePort(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateEnableSsl(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<IdoMysqlConnection>();
public:
	~ObjectImpl<IdoMysqlConnection>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHost() const;
public:
	String GetSocketPath() const;
public:
	String GetUser() const;
public:
	String GetPassword() const;
public:
	String GetDatabase() const;
public:
	String GetSslKey() const;
public:
	String GetSslCert() const;
public:
	String GetSslCa() const;
public:
	String GetSslCapath() const;
public:
	String GetSslCipher() const;
public:
	String GetInstanceName() const;
public:
	String GetInstanceDescription() const;
public:
	int GetPort() const;
public:
	bool GetEnableSsl() const;
public:
	void SetHost(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSocketPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetUser(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPassword(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetDatabase(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslKey(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslCert(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslCa(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslCapath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslCipher(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetInstanceName(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetInstanceDescription(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPort(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableSsl(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHost(const Value& cookie = Empty);
public:
	virtual void NotifySocketPath(const Value& cookie = Empty);
public:
	virtual void NotifyUser(const Value& cookie = Empty);
public:
	virtual void NotifyPassword(const Value& cookie = Empty);
public:
	virtual void NotifyDatabase(const Value& cookie = Empty);
public:
	virtual void NotifySslKey(const Value& cookie = Empty);
public:
	virtual void NotifySslCert(const Value& cookie = Empty);
public:
	virtual void NotifySslCa(const Value& cookie = Empty);
public:
	virtual void NotifySslCapath(const Value& cookie = Empty);
public:
	virtual void NotifySslCipher(const Value& cookie = Empty);
public:
	virtual void NotifyInstanceName(const Value& cookie = Empty);
public:
	virtual void NotifyInstanceDescription(const Value& cookie = Empty);
public:
	virtual void NotifyPort(const Value& cookie = Empty);
public:
	virtual void NotifyEnableSsl(const Value& cookie = Empty);
private:
	inline String GetDefaultHost() const;
private:
	inline String GetDefaultSocketPath() const;
private:
	inline String GetDefaultUser() const;
private:
	inline String GetDefaultPassword() const;
private:
	inline String GetDefaultDatabase() const;
private:
	inline String GetDefaultSslKey() const;
private:
	inline String GetDefaultSslCert() const;
private:
	inline String GetDefaultSslCa() const;
private:
	inline String GetDefaultSslCapath() const;
private:
	inline String GetDefaultSslCipher() const;
private:
	inline String GetDefaultInstanceName() const;
private:
	inline String GetDefaultInstanceDescription() const;
private:
	inline int GetDefaultPort() const;
private:
	inline bool GetDefaultEnableSsl() const;
protected:
	virtual void ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSocketPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateUser(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePassword(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateDatabase(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslKey(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslCert(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslCa(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslCapath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslCipher(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateInstanceName(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateInstanceDescription(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePort(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableSsl(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Host;
	AtomicOrLocked<String> m_SocketPath;
	AtomicOrLocked<String> m_User;
	AtomicOrLocked<String> m_Password;
	AtomicOrLocked<String> m_Database;
	AtomicOrLocked<String> m_SslKey;
	AtomicOrLocked<String> m_SslCert;
	AtomicOrLocked<String> m_SslCa;
	AtomicOrLocked<String> m_SslCapath;
	AtomicOrLocked<String> m_SslCipher;
	AtomicOrLocked<String> m_InstanceName;
	AtomicOrLocked<String> m_InstanceDescription;
	AtomicOrLocked<int> m_Port;
	AtomicOrLocked<bool> m_EnableSsl;
public:
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnHostChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnSocketPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnUserChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnPasswordChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnDatabaseChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnSslKeyChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnSslCertChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnSslCaChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnSslCapathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnSslCipherChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnInstanceNameChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnInstanceDescriptionChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnPortChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IdoMysqlConnection>&, const Value&)> OnEnableSslChanged;
};

}
#endif /* IDOMYSQLCONNECTION_TI */
