#include "base/exception.hpp"
#include "base/objectlock.hpp"
#include "base/utility.hpp"
#include "base/convert.hpp"
#include "base/debug.hpp"
#include "base/dependencygraph.hpp"
#include "base/logger.hpp"
#include "base/function.hpp"
#include "base/configobject.hpp"
#include "base/configtype.hpp"
#ifdef _MSC_VER
#pragma warning( push )
#pragma warning( disable : 4244 )
#pragma warning( disable : 4800 )
#endif /* _MSC_VER */

#include "icinga/checkcommand.hpp"

#include "icinga/eventcommand.hpp"

namespace icinga
{

TypeImpl<Checkable>::TypeImpl()
{ }

TypeImpl<Checkable>::~TypeImpl()
{ }

String TypeImpl<Checkable>::GetName() const
{
	return "Checkable";
}

int TypeImpl<Checkable>::GetAttributes() const
{
	return 1;
}

Type::Ptr TypeImpl<Checkable>::GetBaseType() const
{
	return CustomVarObject::TypeInstance;
}

int TypeImpl<Checkable>::GetFieldId(const String& name) const
{
	int offset = CustomVarObject::TypeInstance->GetFieldCount();

	switch (static_cast<int>(Utility::SDBM(name, 8))) {
		case -2076410069:
			if (name == "force_next_check")
				return offset + 48;
			if (name == "force_next_notification")
				return offset + 51;

			break;
		case -1793514404:
			if (name == "retry_interval")
				return offset + 11;

			break;
		case -1786153727:
			if (name == "notes")
				return offset + 9;

			break;
		case -1766450875:
			if (name == "enable_active_checks")
				return offset + 58;

			break;
		case -1766450871:
			if (name == "enable_event_handler")
				return offset + 52;

			break;
		case -1766450870:
			if (name == "enable_flapping")
				return offset + 55;

			break;
		case -1766450862:
			if (name == "enable_notifications")
				return offset + 53;

			break;
		case -1766450860:
			if (name == "enable_passive_checks")
				return offset + 56;
			if (name == "enable_perfdata")
				return offset + 57;

			break;
		case -1328894435:
			if (name == "severity")
				return offset + 26;

			break;
		case -1229597302:
			if (name == "check_attempt")
				return offset + 27;

			break;
		case -1229466109:
			if (name == "check_command")
				return offset + 2;

			break;
		case -1229072516:
			if (name == "check_interval")
				return offset + 10;

			break;
		case -1228613332:
			if (name == "check_period")
				return offset + 3;

			break;
		case -1228350932:
			if (name == "check_timeout")
				return offset + 0;

			break;
		case -597558821:
			if (name == "notes_url")
				return offset + 8;

			break;
		case -413408948:
			if (name == "action_url")
				return offset + 7;

			break;
		case -103543305:
			if (name == "icon_image_alt")
				return offset + 5;
			if (name == "icon_image")
				return offset + 6;

			break;
		case -99631620:
			if (name == "handled")
				return offset + 49;

			break;
		case 8496327:
			if (name == "flapping_threshold")
				return offset + 12;
			if (name == "flapping_current")
				return offset + 13;
			if (name == "flapping_threshold_low")
				return offset + 14;
			if (name == "flapping_threshold_high")
				return offset + 15;
			if (name == "flapping_ignore_states")
				return offset + 16;
			if (name == "flapping_index")
				return offset + 22;
			if (name == "flapping_buffer")
				return offset + 23;
			if (name == "flapping_ignore_states_filter_real")
				return offset + 28;
			if (name == "flapping_last_state")
				return offset + 30;
			if (name == "flapping_last_change")
				return offset + 46;
			if (name == "flapping")
				return offset + 47;

			break;
		case 78447953:
			if (name == "state_before_suppression")
				return offset + 31;

			break;
		case 79497533:
			if (name == "state_raw")
				return offset + 29;

			break;
		case 79628755:
			if (name == "state_type")
				return offset + 35;

			break;
		case 188245183:
			if (name == "problem")
				return offset + 50;

			break;
		case 297506263:
			if (name == "previous_state_change")
				return offset + 39;

			break;
		case 547544276:
			if (name == "command_endpoint")
				return offset + 1;

			break;
		case 603301623:
			if (name == "last_check_result")
				return offset + 17;
			if (name == "last_check_started")
				return offset + 43;
			if (name == "last_check")
				return offset + 45;

			break;
		case 644149968:
			if (name == "last_hard_state_raw")
				return offset + 32;
			if (name == "last_hard_state_change")
				return offset + 36;
			if (name == "last_hard_states_raw")
				return offset + 61;

			break;
		case 727027397:
			if (name == "last_reachable")
				return offset + 54;

			break;
		case 735944897:
			if (name == "last_soft_states_raw")
				return offset + 60;

			break;
		case 736272887:
			if (name == "last_state_raw")
				return offset + 33;
			if (name == "last_state_type")
				return offset + 34;
			if (name == "last_state_change")
				return offset + 37;
			if (name == "last_state_unreachable")
				return offset + 38;

			break;
		case 832744116:
			if (name == "next_check")
				return offset + 44;

			break;
		case 837952360:
			if (name == "pending_executions")
				return offset + 19;

			break;
		case 981975997:
			if (name == "next_update")
				return offset + 40;

			break;
		case 992710678:
			if (name == "executions")
				return offset + 18;

			break;
		case 1092202577:
			if (name == "event_command")
				return offset + 4;

			break;
		case 1458629871:
			if (name == "downtime_depth")
				return offset + 24;

			break;
		case 1611662238:
			if (name == "max_check_attempts")
				return offset + 21;

			break;
		case 1732602716:
			if (name == "volatile")
				return offset + 59;

			break;
		case 2006852981:
			if (name == "suppressed_notifications")
				return offset + 20;

			break;
		case 2011553734:
			if (name == "acknowledgement")
				return offset + 25;
			if (name == "acknowledgement_expiry")
				return offset + 41;
			if (name == "acknowledgement_last_change")
				return offset + 42;

			break;
	}

	return CustomVarObject::TypeInstance->GetFieldId(name);
}

Field TypeImpl<Checkable>::GetFieldInfo(int id) const
{
	int real_id = id - CustomVarObject::TypeInstance->GetFieldCount();
	if (real_id < 0) { return CustomVarObject::TypeInstance->GetFieldInfo(id); }
	switch (real_id) {
		case 0:
			return {0, "Value", "check_timeout", "check_timeout", nullptr, 2, 0};
		case 1:
			return {1, "String", "command_endpoint", "command_endpoint", "Endpoint", 514, 0};
		case 2:
			return {2, "String", "check_command", "check_command", "CheckCommand", 770, 0};
		case 3:
			return {3, "String", "check_period", "check_period", "TimePeriod", 514, 0};
		case 4:
			return {4, "String", "event_command", "event_command", "EventCommand", 514, 0};
		case 5:
			return {5, "String", "icon_image_alt", "icon_image_alt", nullptr, 2, 0};
		case 6:
			return {6, "String", "icon_image", "icon_image", nullptr, 2, 0};
		case 7:
			return {7, "String", "action_url", "action_url", nullptr, 2, 0};
		case 8:
			return {8, "String", "notes_url", "notes_url", nullptr, 2, 0};
		case 9:
			return {9, "String", "notes", "notes", nullptr, 2, 0};
		case 10:
			return {10, "Number", "check_interval", "check_interval", nullptr, 2, 0};
		case 11:
			return {11, "Number", "retry_interval", "retry_interval", nullptr, 2, 0};
		case 12:
			return {12, "Number", "flapping_threshold", "flapping_threshold", nullptr, 4098, 0};
		case 13:
			return {13, "Number", "flapping_current", "flapping_current", nullptr, 4, 0};
		case 14:
			return {14, "Number", "flapping_threshold_low", "flapping_threshold_low", nullptr, 2, 0};
		case 15:
			return {15, "Number", "flapping_threshold_high", "flapping_threshold_high", nullptr, 2, 0};
		case 16:
			return {16, "Array", "flapping_ignore_states", "flapping_ignore_states", nullptr, 2, 1};
		case 17:
			return {17, "CheckResult", "last_check_result", "last_check_result", nullptr, 4, 0};
		case 18:
			return {18, "Dictionary", "executions", "executions", nullptr, 1028, 0};
		case 19:
			return {19, "Dictionary", "pending_executions", "pending_executions", nullptr, 3076, 0};
		case 20:
			return {20, "Number", "suppressed_notifications", "suppressed_notifications", nullptr, 3076, 0};
		case 21:
			return {21, "Number", "max_check_attempts", "max_check_attempts", nullptr, 2, 0};
		case 22:
			return {22, "Number", "flapping_index", "flapping_index", nullptr, 3076, 0};
		case 23:
			return {23, "Number", "flapping_buffer", "flapping_buffer", nullptr, 3076, 0};
		case 24:
			return {24, "Number", "downtime_depth", "downtime_depth", nullptr, 65, 0};
		case 25:
			return {25, "Number", "acknowledgement", "acknowledgement", nullptr, 4, 0};
		case 26:
			return {26, "Number", "severity", "severity", nullptr, 65, 0};
		case 27:
			return {27, "Number", "check_attempt", "check_attempt", nullptr, 4, 0};
		case 28:
			return {28, "Number", "flapping_ignore_states_filter_real", "flapping_ignore_states_filter_real", nullptr, 3073, 0};
		case 29:
			return {29, "Number", "state_raw", "state_raw", nullptr, 3084, 0};
		case 30:
			return {30, "Number", "flapping_last_state", "flapping_last_state", nullptr, 3084, 0};
		case 31:
			return {31, "Number", "state_before_suppression", "state_before_suppression", nullptr, 3084, 0};
		case 32:
			return {32, "Number", "last_hard_state_raw", "last_hard_state_raw", nullptr, 3084, 0};
		case 33:
			return {33, "Number", "last_state_raw", "last_state_raw", nullptr, 3084, 0};
		case 34:
			return {34, "Number", "last_state_type", "last_state_type", nullptr, 12, 0};
		case 35:
			return {35, "Number", "state_type", "state_type", nullptr, 12, 0};
		case 36:
			return {36, "Timestamp", "last_hard_state_change", "last_hard_state_change", nullptr, 4, 0};
		case 37:
			return {37, "Timestamp", "last_state_change", "last_state_change", nullptr, 4, 0};
		case 38:
			return {38, "Timestamp", "last_state_unreachable", "last_state_unreachable", nullptr, 4, 0};
		case 39:
			return {39, "Timestamp", "previous_state_change", "previous_state_change", nullptr, 4, 0};
		case 40:
			return {40, "Timestamp", "next_update", "next_update", nullptr, 65, 0};
		case 41:
			return {41, "Timestamp", "acknowledgement_expiry", "acknowledgement_expiry", nullptr, 4, 0};
		case 42:
			return {42, "Timestamp", "acknowledgement_last_change", "acknowledgement_last_change", nullptr, 4, 0};
		case 43:
			return {43, "Timestamp", "last_check_started", "last_check_started", nullptr, 3076, 0};
		case 44:
			return {44, "Timestamp", "next_check", "next_check", nullptr, 4, 0};
		case 45:
			return {45, "Timestamp", "last_check", "last_check", nullptr, 65, 0};
		case 46:
			return {46, "Timestamp", "flapping_last_change", "flapping_last_change", nullptr, 4, 0};
		case 47:
			return {47, "Boolean", "flapping", "flapping", nullptr, 52, 0};
		case 48:
			return {48, "Boolean", "force_next_check", "force_next_check", nullptr, 4, 0};
		case 49:
			return {49, "Boolean", "handled", "handled", nullptr, 65, 0};
		case 50:
			return {50, "Boolean", "problem", "problem", nullptr, 65, 0};
		case 51:
			return {51, "Boolean", "force_next_notification", "force_next_notification", nullptr, 4, 0};
		case 52:
			return {52, "Boolean", "enable_event_handler", "enable_event_handler", nullptr, 2, 0};
		case 53:
			return {53, "Boolean", "enable_notifications", "enable_notifications", nullptr, 2, 0};
		case 54:
			return {54, "Boolean", "last_reachable", "last_reachable", nullptr, 4, 0};
		case 55:
			return {55, "Boolean", "enable_flapping", "enable_flapping", nullptr, 2, 0};
		case 56:
			return {56, "Boolean", "enable_passive_checks", "enable_passive_checks", nullptr, 2, 0};
		case 57:
			return {57, "Boolean", "enable_perfdata", "enable_perfdata", nullptr, 2, 0};
		case 58:
			return {58, "Boolean", "enable_active_checks", "enable_active_checks", nullptr, 2, 0};
		case 59:
			return {59, "Boolean", "volatile", "volatile", nullptr, 2, 0};
		case 60:
			return {60, "unsigned short", "last_soft_states_raw", "last_soft_states_raw", nullptr, 3076, 0};
		case 61:
			return {61, "unsigned short", "last_hard_states_raw", "last_hard_states_raw", nullptr, 3076, 0};
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

int TypeImpl<Checkable>::GetFieldCount() const
{
	return 62 + CustomVarObject::TypeInstance->GetFieldCount();
}

ObjectFactory TypeImpl<Checkable>::GetFactory() const
{
	return TypeHelper<Checkable, false>::GetFactory();
}

int TypeImpl<Checkable>::GetActivationPriority() const
{
	return 0;
}

void TypeImpl<Checkable>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	int real_id = fieldId - CustomVarObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { CustomVarObject::TypeInstance->RegisterAttributeHandler(fieldId, callback); return; }
	switch (real_id) {
		case 0:
			ObjectImpl<Checkable>::OnCheckTimeoutChanged.connect(callback);
			break;
		case 1:
			ObjectImpl<Checkable>::OnCommandEndpointRawChanged.connect(callback);
			break;
		case 2:
			ObjectImpl<Checkable>::OnCheckCommandRawChanged.connect(callback);
			break;
		case 3:
			ObjectImpl<Checkable>::OnCheckPeriodRawChanged.connect(callback);
			break;
		case 4:
			ObjectImpl<Checkable>::OnEventCommandRawChanged.connect(callback);
			break;
		case 5:
			ObjectImpl<Checkable>::OnIconImageAltChanged.connect(callback);
			break;
		case 6:
			ObjectImpl<Checkable>::OnIconImageChanged.connect(callback);
			break;
		case 7:
			ObjectImpl<Checkable>::OnActionUrlChanged.connect(callback);
			break;
		case 8:
			ObjectImpl<Checkable>::OnNotesUrlChanged.connect(callback);
			break;
		case 9:
			ObjectImpl<Checkable>::OnNotesChanged.connect(callback);
			break;
		case 10:
			ObjectImpl<Checkable>::OnCheckIntervalChanged.connect(callback);
			break;
		case 11:
			ObjectImpl<Checkable>::OnRetryIntervalChanged.connect(callback);
			break;
		case 12:
			ObjectImpl<Checkable>::OnFlappingThresholdChanged.connect(callback);
			break;
		case 13:
			ObjectImpl<Checkable>::OnFlappingCurrentChanged.connect(callback);
			break;
		case 14:
			ObjectImpl<Checkable>::OnFlappingThresholdLowChanged.connect(callback);
			break;
		case 15:
			ObjectImpl<Checkable>::OnFlappingThresholdHighChanged.connect(callback);
			break;
		case 16:
			ObjectImpl<Checkable>::OnFlappingIgnoreStatesChanged.connect(callback);
			break;
		case 17:
			ObjectImpl<Checkable>::OnLastCheckResultChanged.connect(callback);
			break;
		case 18:
			ObjectImpl<Checkable>::OnExecutionsChanged.connect(callback);
			break;
		case 19:
			ObjectImpl<Checkable>::OnPendingExecutionsChanged.connect(callback);
			break;
		case 20:
			ObjectImpl<Checkable>::OnSuppressedNotificationsChanged.connect(callback);
			break;
		case 21:
			ObjectImpl<Checkable>::OnMaxCheckAttemptsChanged.connect(callback);
			break;
		case 22:
			ObjectImpl<Checkable>::OnFlappingIndexChanged.connect(callback);
			break;
		case 23:
			ObjectImpl<Checkable>::OnFlappingBufferChanged.connect(callback);
			break;
		case 24:
			ObjectImpl<Checkable>::OnDowntimeDepthChanged.connect(callback);
			break;
		case 25:
			ObjectImpl<Checkable>::OnAcknowledgementRawChanged.connect(callback);
			break;
		case 26:
			ObjectImpl<Checkable>::OnSeverityChanged.connect(callback);
			break;
		case 27:
			ObjectImpl<Checkable>::OnCheckAttemptChanged.connect(callback);
			break;
		case 28:
			ObjectImpl<Checkable>::OnFlappingIgnoreStatesFilterChanged.connect(callback);
			break;
		case 29:
			ObjectImpl<Checkable>::OnStateRawChanged.connect(callback);
			break;
		case 30:
			ObjectImpl<Checkable>::OnFlappingLastStateChanged.connect(callback);
			break;
		case 31:
			ObjectImpl<Checkable>::OnStateBeforeSuppressionChanged.connect(callback);
			break;
		case 32:
			ObjectImpl<Checkable>::OnLastHardStateRawChanged.connect(callback);
			break;
		case 33:
			ObjectImpl<Checkable>::OnLastStateRawChanged.connect(callback);
			break;
		case 34:
			ObjectImpl<Checkable>::OnLastStateTypeChanged.connect(callback);
			break;
		case 35:
			ObjectImpl<Checkable>::OnStateTypeChanged.connect(callback);
			break;
		case 36:
			ObjectImpl<Checkable>::OnLastHardStateChangeChanged.connect(callback);
			break;
		case 37:
			ObjectImpl<Checkable>::OnLastStateChangeChanged.connect(callback);
			break;
		case 38:
			ObjectImpl<Checkable>::OnLastStateUnreachableChanged.connect(callback);
			break;
		case 39:
			ObjectImpl<Checkable>::OnPreviousStateChangeChanged.connect(callback);
			break;
		case 40:
			ObjectImpl<Checkable>::OnNextUpdateChanged.connect(callback);
			break;
		case 41:
			ObjectImpl<Checkable>::OnAcknowledgementExpiryChanged.connect(callback);
			break;
		case 42:
			ObjectImpl<Checkable>::OnAcknowledgementLastChangeChanged.connect(callback);
			break;
		case 43:
			ObjectImpl<Checkable>::OnLastCheckStartedChanged.connect(callback);
			break;
		case 44:
			ObjectImpl<Checkable>::OnNextCheckChanged.connect(callback);
			break;
		case 45:
			ObjectImpl<Checkable>::OnLastCheckChanged.connect(callback);
			break;
		case 46:
			ObjectImpl<Checkable>::OnFlappingLastChangeChanged.connect(callback);
			break;
		case 47:
			ObjectImpl<Checkable>::OnFlappingChanged.connect(callback);
			break;
		case 48:
			ObjectImpl<Checkable>::OnForceNextCheckChanged.connect(callback);
			break;
		case 49:
			ObjectImpl<Checkable>::OnHandledChanged.connect(callback);
			break;
		case 50:
			ObjectImpl<Checkable>::OnProblemChanged.connect(callback);
			break;
		case 51:
			ObjectImpl<Checkable>::OnForceNextNotificationChanged.connect(callback);
			break;
		case 52:
			ObjectImpl<Checkable>::OnEnableEventHandlerChanged.connect(callback);
			break;
		case 53:
			ObjectImpl<Checkable>::OnEnableNotificationsChanged.connect(callback);
			break;
		case 54:
			ObjectImpl<Checkable>::OnLastReachableChanged.connect(callback);
			break;
		case 55:
			ObjectImpl<Checkable>::OnEnableFlappingChanged.connect(callback);
			break;
		case 56:
			ObjectImpl<Checkable>::OnEnablePassiveChecksChanged.connect(callback);
			break;
		case 57:
			ObjectImpl<Checkable>::OnEnablePerfdataChanged.connect(callback);
			break;
		case 58:
			ObjectImpl<Checkable>::OnEnableActiveChecksChanged.connect(callback);
			break;
		case 59:
			ObjectImpl<Checkable>::OnVolatileChanged.connect(callback);
			break;
		case 60:
			ObjectImpl<Checkable>::OnLastSoftStatesRawChanged.connect(callback);
			break;
		case 61:
			ObjectImpl<Checkable>::OnLastHardStatesRawChanged.connect(callback);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Checkable>::Validate(int types, const ValidationUtils& utils)
{
	CustomVarObject::Validate(types, utils);

	if (2 & types)
		ValidateCheckTimeout(Lazy<Value>([this]() { return GetCheckTimeout(); }), utils);
	if (2 & types)
		ValidateCommandEndpointRaw(Lazy<String>([this]() { return GetCommandEndpointRaw(); }), utils);
	if (2 & types)
		ValidateCheckCommandRaw(Lazy<String>([this]() { return GetCheckCommandRaw(); }), utils);
	if (2 & types)
		ValidateCheckPeriodRaw(Lazy<String>([this]() { return GetCheckPeriodRaw(); }), utils);
	if (2 & types)
		ValidateEventCommandRaw(Lazy<String>([this]() { return GetEventCommandRaw(); }), utils);
	if (2 & types)
		ValidateIconImageAlt(Lazy<String>([this]() { return GetIconImageAlt(); }), utils);
	if (2 & types)
		ValidateIconImage(Lazy<String>([this]() { return GetIconImage(); }), utils);
	if (2 & types)
		ValidateActionUrl(Lazy<String>([this]() { return GetActionUrl(); }), utils);
	if (2 & types)
		ValidateNotesUrl(Lazy<String>([this]() { return GetNotesUrl(); }), utils);
	if (2 & types)
		ValidateNotes(Lazy<String>([this]() { return GetNotes(); }), utils);
	if (2 & types)
		ValidateCheckInterval(Lazy<double>([this]() { return GetCheckInterval(); }), utils);
	if (2 & types)
		ValidateRetryInterval(Lazy<double>([this]() { return GetRetryInterval(); }), utils);
	if (2 & types)
		ValidateFlappingThreshold(Lazy<double>([this]() { return GetFlappingThreshold(); }), utils);
	if (4 & types)
		ValidateFlappingCurrent(Lazy<double>([this]() { return GetFlappingCurrent(); }), utils);
	if (2 & types)
		ValidateFlappingThresholdLow(Lazy<double>([this]() { return GetFlappingThresholdLow(); }), utils);
	if (2 & types)
		ValidateFlappingThresholdHigh(Lazy<double>([this]() { return GetFlappingThresholdHigh(); }), utils);
	if (2 & types)
		ValidateFlappingIgnoreStates(Lazy<Array::Ptr>([this]() { return GetFlappingIgnoreStates(); }), utils);
	if (4 & types)
		ValidateLastCheckResult(Lazy<CheckResult::Ptr>([this]() { return GetLastCheckResult(); }), utils);
	if (4 & types)
		ValidateExecutions(Lazy<Dictionary::Ptr>([this]() { return GetExecutions(); }), utils);
	if (4 & types)
		ValidatePendingExecutions(Lazy<Dictionary::Ptr>([this]() { return GetPendingExecutions(); }), utils);
	if (4 & types)
		ValidateSuppressedNotifications(Lazy<int>([this]() { return GetSuppressedNotifications(); }), utils);
	if (2 & types)
		ValidateMaxCheckAttempts(Lazy<int>([this]() { return GetMaxCheckAttempts(); }), utils);
	if (4 & types)
		ValidateFlappingIndex(Lazy<int>([this]() { return GetFlappingIndex(); }), utils);
	if (4 & types)
		ValidateFlappingBuffer(Lazy<int>([this]() { return GetFlappingBuffer(); }), utils);
	if (1 & types)
		ValidateDowntimeDepth(Lazy<int>([this]() { return GetDowntimeDepth(); }), utils);
	if (4 & types)
		ValidateAcknowledgementRaw(Lazy<int>([this]() { return GetAcknowledgementRaw(); }), utils);
	if (1 & types)
		ValidateSeverity(Lazy<int>([this]() { return GetSeverity(); }), utils);
	if (4 & types)
		ValidateCheckAttempt(Lazy<int>([this]() { return GetCheckAttempt(); }), utils);
	if (1 & types)
		ValidateFlappingIgnoreStatesFilter(Lazy<int>([this]() { return GetFlappingIgnoreStatesFilter(); }), utils);
	if (4 & types)
		ValidateStateRaw(Lazy<ServiceState>([this]() { return GetStateRaw(); }), utils);
	if (4 & types)
		ValidateFlappingLastState(Lazy<ServiceState>([this]() { return GetFlappingLastState(); }), utils);
	if (4 & types)
		ValidateStateBeforeSuppression(Lazy<ServiceState>([this]() { return GetStateBeforeSuppression(); }), utils);
	if (4 & types)
		ValidateLastHardStateRaw(Lazy<ServiceState>([this]() { return GetLastHardStateRaw(); }), utils);
	if (4 & types)
		ValidateLastStateRaw(Lazy<ServiceState>([this]() { return GetLastStateRaw(); }), utils);
	if (4 & types)
		ValidateLastStateType(Lazy<StateType>([this]() { return GetLastStateType(); }), utils);
	if (4 & types)
		ValidateStateType(Lazy<StateType>([this]() { return GetStateType(); }), utils);
	if (4 & types)
		ValidateLastHardStateChange(Lazy<Timestamp>([this]() { return GetLastHardStateChange(); }), utils);
	if (4 & types)
		ValidateLastStateChange(Lazy<Timestamp>([this]() { return GetLastStateChange(); }), utils);
	if (4 & types)
		ValidateLastStateUnreachable(Lazy<Timestamp>([this]() { return GetLastStateUnreachable(); }), utils);
	if (4 & types)
		ValidatePreviousStateChange(Lazy<Timestamp>([this]() { return GetPreviousStateChange(); }), utils);
	if (1 & types)
		ValidateNextUpdate(Lazy<Timestamp>([this]() { return GetNextUpdate(); }), utils);
	if (4 & types)
		ValidateAcknowledgementExpiry(Lazy<Timestamp>([this]() { return GetAcknowledgementExpiry(); }), utils);
	if (4 & types)
		ValidateAcknowledgementLastChange(Lazy<Timestamp>([this]() { return GetAcknowledgementLastChange(); }), utils);
	if (4 & types)
		ValidateLastCheckStarted(Lazy<Timestamp>([this]() { return GetLastCheckStarted(); }), utils);
	if (4 & types)
		ValidateNextCheck(Lazy<Timestamp>([this]() { return GetNextCheck(); }), utils);
	if (1 & types)
		ValidateLastCheck(Lazy<Timestamp>([this]() { return GetLastCheck(); }), utils);
	if (4 & types)
		ValidateFlappingLastChange(Lazy<Timestamp>([this]() { return GetFlappingLastChange(); }), utils);
	if (4 & types)
		ValidateFlapping(Lazy<bool>([this]() { return GetFlapping(); }), utils);
	if (4 & types)
		ValidateForceNextCheck(Lazy<bool>([this]() { return GetForceNextCheck(); }), utils);
	if (1 & types)
		ValidateHandled(Lazy<bool>([this]() { return GetHandled(); }), utils);
	if (1 & types)
		ValidateProblem(Lazy<bool>([this]() { return GetProblem(); }), utils);
	if (4 & types)
		ValidateForceNextNotification(Lazy<bool>([this]() { return GetForceNextNotification(); }), utils);
	if (2 & types)
		ValidateEnableEventHandler(Lazy<bool>([this]() { return GetEnableEventHandler(); }), utils);
	if (2 & types)
		ValidateEnableNotifications(Lazy<bool>([this]() { return GetEnableNotifications(); }), utils);
	if (4 & types)
		ValidateLastReachable(Lazy<bool>([this]() { return GetLastReachable(); }), utils);
	if (2 & types)
		ValidateEnableFlapping(Lazy<bool>([this]() { return GetEnableFlapping(); }), utils);
	if (2 & types)
		ValidateEnablePassiveChecks(Lazy<bool>([this]() { return GetEnablePassiveChecks(); }), utils);
	if (2 & types)
		ValidateEnablePerfdata(Lazy<bool>([this]() { return GetEnablePerfdata(); }), utils);
	if (2 & types)
		ValidateEnableActiveChecks(Lazy<bool>([this]() { return GetEnableActiveChecks(); }), utils);
	if (2 & types)
		ValidateVolatile(Lazy<bool>([this]() { return GetVolatile(); }), utils);
	if (4 & types)
		ValidateLastSoftStatesRaw(Lazy<unsigned short>([this]() { return GetLastSoftStatesRaw(); }), utils);
	if (4 & types)
		ValidateLastHardStatesRaw(Lazy<unsigned short>([this]() { return GetLastHardStatesRaw(); }), utils);
}

void ObjectImpl<Checkable>::SimpleValidateCheckTimeout(const Lazy<Value>& value, const ValidationUtils& utils)
{
	if (value().IsObjectType<Function>()) {
		Function::Ptr func = value();
		if (func->IsDeprecated())
			Log(LogWarning, "Checkable") << "Attribute 'check_timeout' for object '" << dynamic_cast<ConfigObject *>(this)->GetName() << "' of type '" << dynamic_cast<ConfigObject *>(this)->GetReflectionType()->GetName() << "' is set to a deprecated function: " << func->GetName();
	}

}

void ObjectImpl<Checkable>::SimpleValidateCommandEndpointRaw(const Lazy<String>& value, const ValidationUtils& utils)
{
			if (!value().IsEmpty() && !utils.ValidateName("Endpoint", value()))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "command_endpoint" }, "Object '" + value() + "' of type 'Endpoint' does not exist."));
}

void ObjectImpl<Checkable>::SimpleValidateCheckCommandRaw(const Lazy<String>& value, const ValidationUtils& utils)
{
	if (value().IsEmpty())
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "check_command" }, "Attribute must not be empty."));

			if (!value().IsEmpty() && !utils.ValidateName("CheckCommand", value()))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "check_command" }, "Object '" + value() + "' of type 'CheckCommand' does not exist."));
}

void ObjectImpl<Checkable>::SimpleValidateCheckPeriodRaw(const Lazy<String>& value, const ValidationUtils& utils)
{
			if (!value().IsEmpty() && !utils.ValidateName("TimePeriod", value()))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "check_period" }, "Object '" + value() + "' of type 'TimePeriod' does not exist."));
}

void ObjectImpl<Checkable>::SimpleValidateEventCommandRaw(const Lazy<String>& value, const ValidationUtils& utils)
{
			if (!value().IsEmpty() && !utils.ValidateName("EventCommand", value()))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "event_command" }, "Object '" + value() + "' of type 'EventCommand' does not exist."));
}

void ObjectImpl<Checkable>::SimpleValidateIconImageAlt(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateIconImage(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateActionUrl(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateNotesUrl(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateNotes(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateCheckInterval(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateRetryInterval(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingThreshold(const Lazy<double>& value, const ValidationUtils& utils)
{
	if (value() != GetDefaultFlappingThreshold())
		Log(LogWarning, "Checkable") << "Attribute 'flapping_threshold' for object '" << dynamic_cast<ConfigObject *>(this)->GetName() << "' of type '" << dynamic_cast<ConfigObject *>(this)->GetReflectionType()->GetName() << "' is deprecated and should not be used.";
}

void ObjectImpl<Checkable>::SimpleValidateFlappingCurrent(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingThresholdLow(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingThresholdHigh(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingIgnoreStates(const Lazy<Array::Ptr>& avalue, const ValidationUtils& utils)
{
	if (avalue()) {
		ObjectLock olock(avalue());
		for (const Value& value : avalue()) {
		}
	}
}

void ObjectImpl<Checkable>::SimpleValidateLastCheckResult(const Lazy<CheckResult::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateExecutions(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidatePendingExecutions(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateSuppressedNotifications(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateMaxCheckAttempts(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingIndex(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingBuffer(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateDowntimeDepth(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateAcknowledgementRaw(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateSeverity(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateCheckAttempt(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingIgnoreStatesFilter(const Lazy<int>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateStateRaw(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingLastState(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateStateBeforeSuppression(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastHardStateRaw(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastStateRaw(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastStateType(const Lazy<StateType>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateStateType(const Lazy<StateType>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastHardStateChange(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastStateChange(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastStateUnreachable(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidatePreviousStateChange(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateNextUpdate(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateAcknowledgementExpiry(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateAcknowledgementLastChange(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastCheckStarted(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateNextCheck(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastCheck(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlappingLastChange(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateFlapping(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateForceNextCheck(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateHandled(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateProblem(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateForceNextNotification(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateEnableEventHandler(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateEnableNotifications(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastReachable(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateEnableFlapping(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateEnablePassiveChecks(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateEnablePerfdata(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateEnableActiveChecks(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateVolatile(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastSoftStatesRaw(const Lazy<unsigned short>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Checkable>::SimpleValidateLastHardStatesRaw(const Lazy<unsigned short>& value, const ValidationUtils& utils)
{
}

ObjectImpl<Checkable>::ObjectImpl()
{
	SetCheckTimeout(GetDefaultCheckTimeout(), true);
	SetCommandEndpointRaw(GetDefaultCommandEndpointRaw(), true);
	SetCheckCommandRaw(GetDefaultCheckCommandRaw(), true);
	SetCheckPeriodRaw(GetDefaultCheckPeriodRaw(), true);
	SetEventCommandRaw(GetDefaultEventCommandRaw(), true);
	SetIconImageAlt(GetDefaultIconImageAlt(), true);
	SetIconImage(GetDefaultIconImage(), true);
	SetActionUrl(GetDefaultActionUrl(), true);
	SetNotesUrl(GetDefaultNotesUrl(), true);
	SetNotes(GetDefaultNotes(), true);
	SetCheckInterval(GetDefaultCheckInterval(), true);
	SetRetryInterval(GetDefaultRetryInterval(), true);
	SetFlappingThreshold(GetDefaultFlappingThreshold(), true);
	SetFlappingCurrent(GetDefaultFlappingCurrent(), true);
	SetFlappingThresholdLow(GetDefaultFlappingThresholdLow(), true);
	SetFlappingThresholdHigh(GetDefaultFlappingThresholdHigh(), true);
	SetFlappingIgnoreStates(GetDefaultFlappingIgnoreStates(), true);
	SetLastCheckResult(GetDefaultLastCheckResult(), true);
	SetExecutions(GetDefaultExecutions(), true);
	SetPendingExecutions(GetDefaultPendingExecutions(), true);
	SetSuppressedNotifications(GetDefaultSuppressedNotifications(), true);
	SetMaxCheckAttempts(GetDefaultMaxCheckAttempts(), true);
	SetFlappingIndex(GetDefaultFlappingIndex(), true);
	SetFlappingBuffer(GetDefaultFlappingBuffer(), true);
	SetDowntimeDepth(GetDefaultDowntimeDepth(), true);
	SetAcknowledgementRaw(GetDefaultAcknowledgementRaw(), true);
	SetSeverity(GetDefaultSeverity(), true);
	SetCheckAttempt(GetDefaultCheckAttempt(), true);
	SetFlappingIgnoreStatesFilter(GetDefaultFlappingIgnoreStatesFilter(), true);
	SetStateRaw(GetDefaultStateRaw(), true);
	SetFlappingLastState(GetDefaultFlappingLastState(), true);
	SetStateBeforeSuppression(GetDefaultStateBeforeSuppression(), true);
	SetLastHardStateRaw(GetDefaultLastHardStateRaw(), true);
	SetLastStateRaw(GetDefaultLastStateRaw(), true);
	SetLastStateType(GetDefaultLastStateType(), true);
	SetStateType(GetDefaultStateType(), true);
	SetLastHardStateChange(GetDefaultLastHardStateChange(), true);
	SetLastStateChange(GetDefaultLastStateChange(), true);
	SetLastStateUnreachable(GetDefaultLastStateUnreachable(), true);
	SetPreviousStateChange(GetDefaultPreviousStateChange(), true);
	SetNextUpdate(GetDefaultNextUpdate(), true);
	SetAcknowledgementExpiry(GetDefaultAcknowledgementExpiry(), true);
	SetAcknowledgementLastChange(GetDefaultAcknowledgementLastChange(), true);
	SetLastCheckStarted(GetDefaultLastCheckStarted(), true);
	SetNextCheck(GetDefaultNextCheck(), true);
	SetLastCheck(GetDefaultLastCheck(), true);
	SetFlappingLastChange(GetDefaultFlappingLastChange(), true);
	SetFlapping(GetDefaultFlapping(), true);
	SetForceNextCheck(GetDefaultForceNextCheck(), true);
	SetHandled(GetDefaultHandled(), true);
	SetProblem(GetDefaultProblem(), true);
	SetForceNextNotification(GetDefaultForceNextNotification(), true);
	SetEnableEventHandler(GetDefaultEnableEventHandler(), true);
	SetEnableNotifications(GetDefaultEnableNotifications(), true);
	SetLastReachable(GetDefaultLastReachable(), true);
	SetEnableFlapping(GetDefaultEnableFlapping(), true);
	SetEnablePassiveChecks(GetDefaultEnablePassiveChecks(), true);
	SetEnablePerfdata(GetDefaultEnablePerfdata(), true);
	SetEnableActiveChecks(GetDefaultEnableActiveChecks(), true);
	SetVolatile(GetDefaultVolatile(), true);
	SetLastSoftStatesRaw(GetDefaultLastSoftStatesRaw(), true);
	SetLastHardStatesRaw(GetDefaultLastHardStatesRaw(), true);
}

ObjectImpl<Checkable>::~ObjectImpl()
{ }

void ObjectImpl<Checkable>::SetField(int id, const Value& value, bool suppress_events, const Value& cookie)
{
	int real_id = id - CustomVarObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { CustomVarObject::SetField(id, value, suppress_events, cookie); return; }
	switch (real_id) {
		case 0:
			SetCheckTimeout(value, suppress_events, cookie);
			break;
		case 1:
			SetCommandEndpointRaw(value, suppress_events, cookie);
			break;
		case 2:
			SetCheckCommandRaw(value, suppress_events, cookie);
			break;
		case 3:
			SetCheckPeriodRaw(value, suppress_events, cookie);
			break;
		case 4:
			SetEventCommandRaw(value, suppress_events, cookie);
			break;
		case 5:
			SetIconImageAlt(value, suppress_events, cookie);
			break;
		case 6:
			SetIconImage(value, suppress_events, cookie);
			break;
		case 7:
			SetActionUrl(value, suppress_events, cookie);
			break;
		case 8:
			SetNotesUrl(value, suppress_events, cookie);
			break;
		case 9:
			SetNotes(value, suppress_events, cookie);
			break;
		case 10:
			SetCheckInterval(value, suppress_events, cookie);
			break;
		case 11:
			SetRetryInterval(value, suppress_events, cookie);
			break;
		case 12:
			SetFlappingThreshold(value, suppress_events, cookie);
			break;
		case 13:
			SetFlappingCurrent(value, suppress_events, cookie);
			break;
		case 14:
			SetFlappingThresholdLow(value, suppress_events, cookie);
			break;
		case 15:
			SetFlappingThresholdHigh(value, suppress_events, cookie);
			break;
		case 16:
			SetFlappingIgnoreStates(value, suppress_events, cookie);
			break;
		case 17:
			SetLastCheckResult(value, suppress_events, cookie);
			break;
		case 18:
			SetExecutions(value, suppress_events, cookie);
			break;
		case 19:
			SetPendingExecutions(value, suppress_events, cookie);
			break;
		case 20:
			SetSuppressedNotifications(value, suppress_events, cookie);
			break;
		case 21:
			SetMaxCheckAttempts(value, suppress_events, cookie);
			break;
		case 22:
			SetFlappingIndex(value, suppress_events, cookie);
			break;
		case 23:
			SetFlappingBuffer(value, suppress_events, cookie);
			break;
		case 24:
			SetDowntimeDepth(value, suppress_events, cookie);
			break;
		case 25:
			SetAcknowledgementRaw(value, suppress_events, cookie);
			break;
		case 26:
			SetSeverity(value, suppress_events, cookie);
			break;
		case 27:
			SetCheckAttempt(value, suppress_events, cookie);
			break;
		case 28:
			SetFlappingIgnoreStatesFilter(value, suppress_events, cookie);
			break;
		case 29:
			SetStateRaw(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 30:
			SetFlappingLastState(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 31:
			SetStateBeforeSuppression(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 32:
			SetLastHardStateRaw(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 33:
			SetLastStateRaw(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 34:
			SetLastStateType(static_cast<StateType>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 35:
			SetStateType(static_cast<StateType>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 36:
			SetLastHardStateChange(value, suppress_events, cookie);
			break;
		case 37:
			SetLastStateChange(value, suppress_events, cookie);
			break;
		case 38:
			SetLastStateUnreachable(value, suppress_events, cookie);
			break;
		case 39:
			SetPreviousStateChange(value, suppress_events, cookie);
			break;
		case 40:
			SetNextUpdate(value, suppress_events, cookie);
			break;
		case 41:
			SetAcknowledgementExpiry(value, suppress_events, cookie);
			break;
		case 42:
			SetAcknowledgementLastChange(value, suppress_events, cookie);
			break;
		case 43:
			SetLastCheckStarted(value, suppress_events, cookie);
			break;
		case 44:
			SetNextCheck(value, suppress_events, cookie);
			break;
		case 45:
			SetLastCheck(value, suppress_events, cookie);
			break;
		case 46:
			SetFlappingLastChange(value, suppress_events, cookie);
			break;
		case 47:
			SetFlapping(value, suppress_events, cookie);
			break;
		case 48:
			SetForceNextCheck(value, suppress_events, cookie);
			break;
		case 49:
			SetHandled(value, suppress_events, cookie);
			break;
		case 50:
			SetProblem(value, suppress_events, cookie);
			break;
		case 51:
			SetForceNextNotification(value, suppress_events, cookie);
			break;
		case 52:
			SetEnableEventHandler(value, suppress_events, cookie);
			break;
		case 53:
			SetEnableNotifications(value, suppress_events, cookie);
			break;
		case 54:
			SetLastReachable(value, suppress_events, cookie);
			break;
		case 55:
			SetEnableFlapping(value, suppress_events, cookie);
			break;
		case 56:
			SetEnablePassiveChecks(value, suppress_events, cookie);
			break;
		case 57:
			SetEnablePerfdata(value, suppress_events, cookie);
			break;
		case 58:
			SetEnableActiveChecks(value, suppress_events, cookie);
			break;
		case 59:
			SetVolatile(value, suppress_events, cookie);
			break;
		case 60:
			SetLastSoftStatesRaw(value, suppress_events, cookie);
			break;
		case 61:
			SetLastHardStatesRaw(value, suppress_events, cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<Checkable>::GetField(int id) const
{
	int real_id = id - CustomVarObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return CustomVarObject::GetField(id); }
	switch (real_id) {
		case 0:
			return GetCheckTimeout();
		case 1:
			return GetCommandEndpointRaw();
		case 2:
			return GetCheckCommandRaw();
		case 3:
			return GetCheckPeriodRaw();
		case 4:
			return GetEventCommandRaw();
		case 5:
			return GetIconImageAlt();
		case 6:
			return GetIconImage();
		case 7:
			return GetActionUrl();
		case 8:
			return GetNotesUrl();
		case 9:
			return GetNotes();
		case 10:
			return GetCheckInterval();
		case 11:
			return GetRetryInterval();
		case 12:
			return GetFlappingThreshold();
		case 13:
			return GetFlappingCurrent();
		case 14:
			return GetFlappingThresholdLow();
		case 15:
			return GetFlappingThresholdHigh();
		case 16:
			return GetFlappingIgnoreStates();
		case 17:
			return GetLastCheckResult();
		case 18:
			return GetExecutions();
		case 19:
			return GetPendingExecutions();
		case 20:
			return GetSuppressedNotifications();
		case 21:
			return GetMaxCheckAttempts();
		case 22:
			return GetFlappingIndex();
		case 23:
			return GetFlappingBuffer();
		case 24:
			return GetDowntimeDepth();
		case 25:
			return GetAcknowledgementRaw();
		case 26:
			return GetSeverity();
		case 27:
			return GetCheckAttempt();
		case 28:
			return GetFlappingIgnoreStatesFilter();
		case 29:
			return GetStateRaw();
		case 30:
			return GetFlappingLastState();
		case 31:
			return GetStateBeforeSuppression();
		case 32:
			return GetLastHardStateRaw();
		case 33:
			return GetLastStateRaw();
		case 34:
			return GetLastStateType();
		case 35:
			return GetStateType();
		case 36:
			return GetLastHardStateChange();
		case 37:
			return GetLastStateChange();
		case 38:
			return GetLastStateUnreachable();
		case 39:
			return GetPreviousStateChange();
		case 40:
			return GetNextUpdate();
		case 41:
			return GetAcknowledgementExpiry();
		case 42:
			return GetAcknowledgementLastChange();
		case 43:
			return GetLastCheckStarted();
		case 44:
			return GetNextCheck();
		case 45:
			return GetLastCheck();
		case 46:
			return GetFlappingLastChange();
		case 47:
			return GetFlapping();
		case 48:
			return GetForceNextCheck();
		case 49:
			return GetHandled();
		case 50:
			return GetProblem();
		case 51:
			return GetForceNextNotification();
		case 52:
			return GetEnableEventHandler();
		case 53:
			return GetEnableNotifications();
		case 54:
			return GetLastReachable();
		case 55:
			return GetEnableFlapping();
		case 56:
			return GetEnablePassiveChecks();
		case 57:
			return GetEnablePerfdata();
		case 58:
			return GetEnableActiveChecks();
		case 59:
			return GetVolatile();
		case 60:
			return GetLastSoftStatesRaw();
		case 61:
			return GetLastHardStatesRaw();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Checkable>::ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	int real_id = id - CustomVarObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { CustomVarObject::ValidateField(id, lvalue, utils); return; }
	switch (real_id) {
		case 0:
			ValidateCheckTimeout(lvalue, utils);
			break;
		case 1:
			ValidateCommandEndpointRaw(lvalue, utils);
			break;
		case 2:
			ValidateCheckCommandRaw(lvalue, utils);
			break;
		case 3:
			ValidateCheckPeriodRaw(lvalue, utils);
			break;
		case 4:
			ValidateEventCommandRaw(lvalue, utils);
			break;
		case 5:
			ValidateIconImageAlt(lvalue, utils);
			break;
		case 6:
			ValidateIconImage(lvalue, utils);
			break;
		case 7:
			ValidateActionUrl(lvalue, utils);
			break;
		case 8:
			ValidateNotesUrl(lvalue, utils);
			break;
		case 9:
			ValidateNotes(lvalue, utils);
			break;
		case 10:
			ValidateCheckInterval(lvalue, utils);
			break;
		case 11:
			ValidateRetryInterval(lvalue, utils);
			break;
		case 12:
			ValidateFlappingThreshold(lvalue, utils);
			break;
		case 13:
			ValidateFlappingCurrent(lvalue, utils);
			break;
		case 14:
			ValidateFlappingThresholdLow(lvalue, utils);
			break;
		case 15:
			ValidateFlappingThresholdHigh(lvalue, utils);
			break;
		case 16:
			ValidateFlappingIgnoreStates(lvalue, utils);
			break;
		case 17:
			ValidateLastCheckResult(lvalue, utils);
			break;
		case 18:
			ValidateExecutions(lvalue, utils);
			break;
		case 19:
			ValidatePendingExecutions(lvalue, utils);
			break;
		case 20:
			ValidateSuppressedNotifications(lvalue, utils);
			break;
		case 21:
			ValidateMaxCheckAttempts(lvalue, utils);
			break;
		case 22:
			ValidateFlappingIndex(lvalue, utils);
			break;
		case 23:
			ValidateFlappingBuffer(lvalue, utils);
			break;
		case 24:
			ValidateDowntimeDepth(lvalue, utils);
			break;
		case 25:
			ValidateAcknowledgementRaw(lvalue, utils);
			break;
		case 26:
			ValidateSeverity(lvalue, utils);
			break;
		case 27:
			ValidateCheckAttempt(lvalue, utils);
			break;
		case 28:
			ValidateFlappingIgnoreStatesFilter(lvalue, utils);
			break;
		case 29:
			ValidateStateRaw(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 30:
			ValidateFlappingLastState(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 31:
			ValidateStateBeforeSuppression(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 32:
			ValidateLastHardStateRaw(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 33:
			ValidateLastStateRaw(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 34:
			ValidateLastStateType(static_cast<Lazy<StateType> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 35:
			ValidateStateType(static_cast<Lazy<StateType> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 36:
			ValidateLastHardStateChange(lvalue, utils);
			break;
		case 37:
			ValidateLastStateChange(lvalue, utils);
			break;
		case 38:
			ValidateLastStateUnreachable(lvalue, utils);
			break;
		case 39:
			ValidatePreviousStateChange(lvalue, utils);
			break;
		case 40:
			ValidateNextUpdate(lvalue, utils);
			break;
		case 41:
			ValidateAcknowledgementExpiry(lvalue, utils);
			break;
		case 42:
			ValidateAcknowledgementLastChange(lvalue, utils);
			break;
		case 43:
			ValidateLastCheckStarted(lvalue, utils);
			break;
		case 44:
			ValidateNextCheck(lvalue, utils);
			break;
		case 45:
			ValidateLastCheck(lvalue, utils);
			break;
		case 46:
			ValidateFlappingLastChange(lvalue, utils);
			break;
		case 47:
			ValidateFlapping(lvalue, utils);
			break;
		case 48:
			ValidateForceNextCheck(lvalue, utils);
			break;
		case 49:
			ValidateHandled(lvalue, utils);
			break;
		case 50:
			ValidateProblem(lvalue, utils);
			break;
		case 51:
			ValidateForceNextNotification(lvalue, utils);
			break;
		case 52:
			ValidateEnableEventHandler(lvalue, utils);
			break;
		case 53:
			ValidateEnableNotifications(lvalue, utils);
			break;
		case 54:
			ValidateLastReachable(lvalue, utils);
			break;
		case 55:
			ValidateEnableFlapping(lvalue, utils);
			break;
		case 56:
			ValidateEnablePassiveChecks(lvalue, utils);
			break;
		case 57:
			ValidateEnablePerfdata(lvalue, utils);
			break;
		case 58:
			ValidateEnableActiveChecks(lvalue, utils);
			break;
		case 59:
			ValidateVolatile(lvalue, utils);
			break;
		case 60:
			ValidateLastSoftStatesRaw(lvalue, utils);
			break;
		case 61:
			ValidateLastHardStatesRaw(lvalue, utils);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Checkable>::NotifyField(int id, const Value& cookie)
{
	int real_id = id - CustomVarObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { CustomVarObject::NotifyField(id, cookie); return; }
	switch (real_id) {
		case 0:
			NotifyCheckTimeout(cookie);
			break;
		case 1:
			NotifyCommandEndpointRaw(cookie);
			break;
		case 2:
			NotifyCheckCommandRaw(cookie);
			break;
		case 3:
			NotifyCheckPeriodRaw(cookie);
			break;
		case 4:
			NotifyEventCommandRaw(cookie);
			break;
		case 5:
			NotifyIconImageAlt(cookie);
			break;
		case 6:
			NotifyIconImage(cookie);
			break;
		case 7:
			NotifyActionUrl(cookie);
			break;
		case 8:
			NotifyNotesUrl(cookie);
			break;
		case 9:
			NotifyNotes(cookie);
			break;
		case 10:
			NotifyCheckInterval(cookie);
			break;
		case 11:
			NotifyRetryInterval(cookie);
			break;
		case 12:
			NotifyFlappingThreshold(cookie);
			break;
		case 13:
			NotifyFlappingCurrent(cookie);
			break;
		case 14:
			NotifyFlappingThresholdLow(cookie);
			break;
		case 15:
			NotifyFlappingThresholdHigh(cookie);
			break;
		case 16:
			NotifyFlappingIgnoreStates(cookie);
			break;
		case 17:
			NotifyLastCheckResult(cookie);
			break;
		case 18:
			NotifyExecutions(cookie);
			break;
		case 19:
			NotifyPendingExecutions(cookie);
			break;
		case 20:
			NotifySuppressedNotifications(cookie);
			break;
		case 21:
			NotifyMaxCheckAttempts(cookie);
			break;
		case 22:
			NotifyFlappingIndex(cookie);
			break;
		case 23:
			NotifyFlappingBuffer(cookie);
			break;
		case 24:
			NotifyDowntimeDepth(cookie);
			break;
		case 25:
			NotifyAcknowledgementRaw(cookie);
			break;
		case 26:
			NotifySeverity(cookie);
			break;
		case 27:
			NotifyCheckAttempt(cookie);
			break;
		case 28:
			NotifyFlappingIgnoreStatesFilter(cookie);
			break;
		case 29:
			NotifyStateRaw(cookie);
			break;
		case 30:
			NotifyFlappingLastState(cookie);
			break;
		case 31:
			NotifyStateBeforeSuppression(cookie);
			break;
		case 32:
			NotifyLastHardStateRaw(cookie);
			break;
		case 33:
			NotifyLastStateRaw(cookie);
			break;
		case 34:
			NotifyLastStateType(cookie);
			break;
		case 35:
			NotifyStateType(cookie);
			break;
		case 36:
			NotifyLastHardStateChange(cookie);
			break;
		case 37:
			NotifyLastStateChange(cookie);
			break;
		case 38:
			NotifyLastStateUnreachable(cookie);
			break;
		case 39:
			NotifyPreviousStateChange(cookie);
			break;
		case 40:
			NotifyNextUpdate(cookie);
			break;
		case 41:
			NotifyAcknowledgementExpiry(cookie);
			break;
		case 42:
			NotifyAcknowledgementLastChange(cookie);
			break;
		case 43:
			NotifyLastCheckStarted(cookie);
			break;
		case 44:
			NotifyNextCheck(cookie);
			break;
		case 45:
			NotifyLastCheck(cookie);
			break;
		case 46:
			NotifyFlappingLastChange(cookie);
			break;
		case 47:
			NotifyFlapping(cookie);
			break;
		case 48:
			NotifyForceNextCheck(cookie);
			break;
		case 49:
			NotifyHandled(cookie);
			break;
		case 50:
			NotifyProblem(cookie);
			break;
		case 51:
			NotifyForceNextNotification(cookie);
			break;
		case 52:
			NotifyEnableEventHandler(cookie);
			break;
		case 53:
			NotifyEnableNotifications(cookie);
			break;
		case 54:
			NotifyLastReachable(cookie);
			break;
		case 55:
			NotifyEnableFlapping(cookie);
			break;
		case 56:
			NotifyEnablePassiveChecks(cookie);
			break;
		case 57:
			NotifyEnablePerfdata(cookie);
			break;
		case 58:
			NotifyEnableActiveChecks(cookie);
			break;
		case 59:
			NotifyVolatile(cookie);
			break;
		case 60:
			NotifyLastSoftStatesRaw(cookie);
			break;
		case 61:
			NotifyLastHardStatesRaw(cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Object::Ptr ObjectImpl<Checkable>::NavigateField(int id) const
{
	int real_id = id - CustomVarObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return CustomVarObject::NavigateField(id); }
	switch (real_id) {
		case 1:
			return NavigateCommandEndpointRaw();
		case 2:
			return NavigateCheckCommandRaw();
		case 3:
			return NavigateCheckPeriodRaw();
		case 4:
			return NavigateEventCommandRaw();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<Checkable>::GetCheckTimeout() const
{
	return m_CheckTimeout.load();
}

String ObjectImpl<Checkable>::GetCommandEndpointRaw() const
{
	return m_CommandEndpointRaw.load();
}

String ObjectImpl<Checkable>::GetCheckCommandRaw() const
{
	return m_CheckCommandRaw.load();
}

String ObjectImpl<Checkable>::GetCheckPeriodRaw() const
{
	return m_CheckPeriodRaw.load();
}

String ObjectImpl<Checkable>::GetEventCommandRaw() const
{
	return m_EventCommandRaw.load();
}

String ObjectImpl<Checkable>::GetIconImageAlt() const
{
	return m_IconImageAlt.load();
}

String ObjectImpl<Checkable>::GetIconImage() const
{
	return m_IconImage.load();
}

String ObjectImpl<Checkable>::GetActionUrl() const
{
	return m_ActionUrl.load();
}

String ObjectImpl<Checkable>::GetNotesUrl() const
{
	return m_NotesUrl.load();
}

String ObjectImpl<Checkable>::GetNotes() const
{
	return m_Notes.load();
}

double ObjectImpl<Checkable>::GetCheckInterval() const
{
	return m_CheckInterval.load();
}

double ObjectImpl<Checkable>::GetRetryInterval() const
{
	return m_RetryInterval.load();
}

double ObjectImpl<Checkable>::GetFlappingThreshold() const
{
	return m_FlappingThreshold.load();
}

double ObjectImpl<Checkable>::GetFlappingCurrent() const
{
	return m_FlappingCurrent.load();
}

double ObjectImpl<Checkable>::GetFlappingThresholdLow() const
{
	return m_FlappingThresholdLow.load();
}

double ObjectImpl<Checkable>::GetFlappingThresholdHigh() const
{
	return m_FlappingThresholdHigh.load();
}

Array::Ptr ObjectImpl<Checkable>::GetFlappingIgnoreStates() const
{
	return m_FlappingIgnoreStates.load();
}

CheckResult::Ptr ObjectImpl<Checkable>::GetLastCheckResult() const
{
	return m_LastCheckResult.load();
}

Dictionary::Ptr ObjectImpl<Checkable>::GetExecutions() const
{
	return m_Executions.load();
}

Dictionary::Ptr ObjectImpl<Checkable>::GetPendingExecutions() const
{
	return m_PendingExecutions.load();
}

int ObjectImpl<Checkable>::GetSuppressedNotifications() const
{
	return m_SuppressedNotifications.load();
}

int ObjectImpl<Checkable>::GetMaxCheckAttempts() const
{
	return m_MaxCheckAttempts.load();
}

int ObjectImpl<Checkable>::GetFlappingIndex() const
{
	return m_FlappingIndex.load();
}

int ObjectImpl<Checkable>::GetFlappingBuffer() const
{
	return m_FlappingBuffer.load();
}

int ObjectImpl<Checkable>::GetAcknowledgementRaw() const
{
	return m_AcknowledgementRaw.load();
}

int ObjectImpl<Checkable>::GetCheckAttempt() const
{
	return m_CheckAttempt.load();
}

int ObjectImpl<Checkable>::GetFlappingIgnoreStatesFilter() const
{
	return m_FlappingIgnoreStatesFilter.load();
}

ServiceState ObjectImpl<Checkable>::GetStateRaw() const
{
	return m_StateRaw.load();
}

ServiceState ObjectImpl<Checkable>::GetFlappingLastState() const
{
	return m_FlappingLastState.load();
}

ServiceState ObjectImpl<Checkable>::GetStateBeforeSuppression() const
{
	return m_StateBeforeSuppression.load();
}

ServiceState ObjectImpl<Checkable>::GetLastHardStateRaw() const
{
	return m_LastHardStateRaw.load();
}

ServiceState ObjectImpl<Checkable>::GetLastStateRaw() const
{
	return m_LastStateRaw.load();
}

StateType ObjectImpl<Checkable>::GetLastStateType() const
{
	return m_LastStateType.load();
}

StateType ObjectImpl<Checkable>::GetStateType() const
{
	return m_StateType.load();
}

Timestamp ObjectImpl<Checkable>::GetLastHardStateChange() const
{
	return m_LastHardStateChange.load();
}

Timestamp ObjectImpl<Checkable>::GetLastStateChange() const
{
	return m_LastStateChange.load();
}

Timestamp ObjectImpl<Checkable>::GetLastStateUnreachable() const
{
	return m_LastStateUnreachable.load();
}

Timestamp ObjectImpl<Checkable>::GetPreviousStateChange() const
{
	return m_PreviousStateChange.load();
}

Timestamp ObjectImpl<Checkable>::GetAcknowledgementExpiry() const
{
	return m_AcknowledgementExpiry.load();
}

Timestamp ObjectImpl<Checkable>::GetAcknowledgementLastChange() const
{
	return m_AcknowledgementLastChange.load();
}

Timestamp ObjectImpl<Checkable>::GetLastCheckStarted() const
{
	return m_LastCheckStarted.load();
}

Timestamp ObjectImpl<Checkable>::GetNextCheck() const
{
	return m_NextCheck.load();
}

Timestamp ObjectImpl<Checkable>::GetFlappingLastChange() const
{
	return m_FlappingLastChange.load();
}

bool ObjectImpl<Checkable>::GetFlapping() const
{
	return m_Flapping.load();
}

bool ObjectImpl<Checkable>::GetForceNextCheck() const
{
	return m_ForceNextCheck.load();
}

bool ObjectImpl<Checkable>::GetForceNextNotification() const
{
	return m_ForceNextNotification.load();
}

bool ObjectImpl<Checkable>::GetEnableEventHandler() const
{
	return m_EnableEventHandler.load();
}

bool ObjectImpl<Checkable>::GetEnableNotifications() const
{
	return m_EnableNotifications.load();
}

bool ObjectImpl<Checkable>::GetLastReachable() const
{
	return m_LastReachable.load();
}

bool ObjectImpl<Checkable>::GetEnableFlapping() const
{
	return m_EnableFlapping.load();
}

bool ObjectImpl<Checkable>::GetEnablePassiveChecks() const
{
	return m_EnablePassiveChecks.load();
}

bool ObjectImpl<Checkable>::GetEnablePerfdata() const
{
	return m_EnablePerfdata.load();
}

bool ObjectImpl<Checkable>::GetEnableActiveChecks() const
{
	return m_EnableActiveChecks.load();
}

bool ObjectImpl<Checkable>::GetVolatile() const
{
	return m_Volatile.load();
}

unsigned short ObjectImpl<Checkable>::GetLastSoftStatesRaw() const
{
	return m_LastSoftStatesRaw.load();
}

unsigned short ObjectImpl<Checkable>::GetLastHardStatesRaw() const
{
	return m_LastHardStatesRaw.load();
}

void ObjectImpl<Checkable>::SetCheckTimeout(const Value& value, bool suppress_events, const Value& cookie)
{
	m_CheckTimeout.store(value);
	if (!suppress_events) {
		NotifyCheckTimeout(cookie);
	}

}

void ObjectImpl<Checkable>::SetCommandEndpointRaw(const String& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetCommandEndpointRaw();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_CommandEndpointRaw.store(value);
	if (!dobj || dobj->IsActive())
		TrackCommandEndpointRaw(oldValue, value);
	if (!suppress_events) {
		NotifyCommandEndpointRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetCheckCommandRaw(const String& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetCheckCommandRaw();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_CheckCommandRaw.store(value);
	if (!dobj || dobj->IsActive())
		TrackCheckCommandRaw(oldValue, value);
	if (!suppress_events) {
		NotifyCheckCommandRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetCheckPeriodRaw(const String& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetCheckPeriodRaw();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_CheckPeriodRaw.store(value);
	if (!dobj || dobj->IsActive())
		TrackCheckPeriodRaw(oldValue, value);
	if (!suppress_events) {
		NotifyCheckPeriodRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetEventCommandRaw(const String& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetEventCommandRaw();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_EventCommandRaw.store(value);
	if (!dobj || dobj->IsActive())
		TrackEventCommandRaw(oldValue, value);
	if (!suppress_events) {
		NotifyEventCommandRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetIconImageAlt(const String& value, bool suppress_events, const Value& cookie)
{
	m_IconImageAlt.store(value);
	if (!suppress_events) {
		NotifyIconImageAlt(cookie);
	}

}

void ObjectImpl<Checkable>::SetIconImage(const String& value, bool suppress_events, const Value& cookie)
{
	m_IconImage.store(value);
	if (!suppress_events) {
		NotifyIconImage(cookie);
	}

}

void ObjectImpl<Checkable>::SetActionUrl(const String& value, bool suppress_events, const Value& cookie)
{
	m_ActionUrl.store(value);
	if (!suppress_events) {
		NotifyActionUrl(cookie);
	}

}

void ObjectImpl<Checkable>::SetNotesUrl(const String& value, bool suppress_events, const Value& cookie)
{
	m_NotesUrl.store(value);
	if (!suppress_events) {
		NotifyNotesUrl(cookie);
	}

}

void ObjectImpl<Checkable>::SetNotes(const String& value, bool suppress_events, const Value& cookie)
{
	m_Notes.store(value);
	if (!suppress_events) {
		NotifyNotes(cookie);
	}

}

void ObjectImpl<Checkable>::SetCheckInterval(double value, bool suppress_events, const Value& cookie)
{
	m_CheckInterval.store(value);
	if (!suppress_events) {
		NotifyCheckInterval(cookie);
	}

}

void ObjectImpl<Checkable>::SetRetryInterval(double value, bool suppress_events, const Value& cookie)
{
	m_RetryInterval.store(value);
	if (!suppress_events) {
		NotifyRetryInterval(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingThreshold(double value, bool suppress_events, const Value& cookie)
{
	m_FlappingThreshold.store(value);
	if (!suppress_events) {
		NotifyFlappingThreshold(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingCurrent(double value, bool suppress_events, const Value& cookie)
{
	m_FlappingCurrent.store(value);
	if (!suppress_events) {
		NotifyFlappingCurrent(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingThresholdLow(double value, bool suppress_events, const Value& cookie)
{
	m_FlappingThresholdLow.store(value);
	if (!suppress_events) {
		NotifyFlappingThresholdLow(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingThresholdHigh(double value, bool suppress_events, const Value& cookie)
{
	m_FlappingThresholdHigh.store(value);
	if (!suppress_events) {
		NotifyFlappingThresholdHigh(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingIgnoreStates(const Array::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_FlappingIgnoreStates.store(value);
	if (!suppress_events) {
		NotifyFlappingIgnoreStates(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastCheckResult(const CheckResult::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_LastCheckResult.store(value);
	if (!suppress_events) {
		NotifyLastCheckResult(cookie);
	}

}

void ObjectImpl<Checkable>::SetExecutions(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_Executions.store(value);
	if (!suppress_events) {
		NotifyExecutions(cookie);
	}

}

void ObjectImpl<Checkable>::SetPendingExecutions(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{
	m_PendingExecutions.store(value);
	if (!suppress_events) {
		NotifyPendingExecutions(cookie);
	}

}

void ObjectImpl<Checkable>::SetSuppressedNotifications(int value, bool suppress_events, const Value& cookie)
{
	m_SuppressedNotifications.store(value);
	if (!suppress_events) {
		NotifySuppressedNotifications(cookie);
	}

}

void ObjectImpl<Checkable>::SetMaxCheckAttempts(int value, bool suppress_events, const Value& cookie)
{
	m_MaxCheckAttempts.store(value);
	if (!suppress_events) {
		NotifyMaxCheckAttempts(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingIndex(int value, bool suppress_events, const Value& cookie)
{
	m_FlappingIndex.store(value);
	if (!suppress_events) {
		NotifyFlappingIndex(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingBuffer(int value, bool suppress_events, const Value& cookie)
{
	m_FlappingBuffer.store(value);
	if (!suppress_events) {
		NotifyFlappingBuffer(cookie);
	}

}

void ObjectImpl<Checkable>::SetDowntimeDepth(int value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyDowntimeDepth(cookie);
	}

}

void ObjectImpl<Checkable>::SetAcknowledgementRaw(int value, bool suppress_events, const Value& cookie)
{
	m_AcknowledgementRaw.store(value);
	if (!suppress_events) {
		NotifyAcknowledgementRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetSeverity(int value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifySeverity(cookie);
	}

}

void ObjectImpl<Checkable>::SetCheckAttempt(int value, bool suppress_events, const Value& cookie)
{
	m_CheckAttempt.store(value);
	if (!suppress_events) {
		NotifyCheckAttempt(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingIgnoreStatesFilter(int value, bool suppress_events, const Value& cookie)
{
	m_FlappingIgnoreStatesFilter.store(value);
	if (!suppress_events) {
		NotifyFlappingIgnoreStatesFilter(cookie);
	}

}

void ObjectImpl<Checkable>::SetStateRaw(const ServiceState& value, bool suppress_events, const Value& cookie)
{
	m_StateRaw.store(value);
	if (!suppress_events) {
		NotifyStateRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingLastState(const ServiceState& value, bool suppress_events, const Value& cookie)
{
	m_FlappingLastState.store(value);
	if (!suppress_events) {
		NotifyFlappingLastState(cookie);
	}

}

void ObjectImpl<Checkable>::SetStateBeforeSuppression(const ServiceState& value, bool suppress_events, const Value& cookie)
{
	m_StateBeforeSuppression.store(value);
	if (!suppress_events) {
		NotifyStateBeforeSuppression(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastHardStateRaw(const ServiceState& value, bool suppress_events, const Value& cookie)
{
	m_LastHardStateRaw.store(value);
	if (!suppress_events) {
		NotifyLastHardStateRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastStateRaw(const ServiceState& value, bool suppress_events, const Value& cookie)
{
	m_LastStateRaw.store(value);
	if (!suppress_events) {
		NotifyLastStateRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastStateType(const StateType& value, bool suppress_events, const Value& cookie)
{
	m_LastStateType.store(value);
	if (!suppress_events) {
		NotifyLastStateType(cookie);
	}

}

void ObjectImpl<Checkable>::SetStateType(const StateType& value, bool suppress_events, const Value& cookie)
{
	m_StateType.store(value);
	if (!suppress_events) {
		NotifyStateType(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastHardStateChange(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastHardStateChange.store(value);
	if (!suppress_events) {
		NotifyLastHardStateChange(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastStateChange(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastStateChange.store(value);
	if (!suppress_events) {
		NotifyLastStateChange(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastStateUnreachable(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastStateUnreachable.store(value);
	if (!suppress_events) {
		NotifyLastStateUnreachable(cookie);
	}

}

void ObjectImpl<Checkable>::SetPreviousStateChange(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_PreviousStateChange.store(value);
	if (!suppress_events) {
		NotifyPreviousStateChange(cookie);
	}

}

void ObjectImpl<Checkable>::SetNextUpdate(const Timestamp& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyNextUpdate(cookie);
	}

}

void ObjectImpl<Checkable>::SetAcknowledgementExpiry(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_AcknowledgementExpiry.store(value);
	if (!suppress_events) {
		NotifyAcknowledgementExpiry(cookie);
	}

}

void ObjectImpl<Checkable>::SetAcknowledgementLastChange(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_AcknowledgementLastChange.store(value);
	if (!suppress_events) {
		NotifyAcknowledgementLastChange(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastCheckStarted(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastCheckStarted.store(value);
	if (!suppress_events) {
		NotifyLastCheckStarted(cookie);
	}

}

void ObjectImpl<Checkable>::SetNextCheck(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_NextCheck.store(value);
	if (!suppress_events) {
		NotifyNextCheck(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastCheck(const Timestamp& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyLastCheck(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlappingLastChange(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_FlappingLastChange.store(value);
	if (!suppress_events) {
		NotifyFlappingLastChange(cookie);
	}

}

void ObjectImpl<Checkable>::SetFlapping(bool value, bool suppress_events, const Value& cookie)
{
	m_Flapping.store(value);
	if (!suppress_events) {
		NotifyFlapping(cookie);
	}

}

void ObjectImpl<Checkable>::SetForceNextCheck(bool value, bool suppress_events, const Value& cookie)
{
	m_ForceNextCheck.store(value);
	if (!suppress_events) {
		NotifyForceNextCheck(cookie);
	}

}

void ObjectImpl<Checkable>::SetHandled(bool value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyHandled(cookie);
	}

}

void ObjectImpl<Checkable>::SetProblem(bool value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyProblem(cookie);
	}

}

void ObjectImpl<Checkable>::SetForceNextNotification(bool value, bool suppress_events, const Value& cookie)
{
	m_ForceNextNotification.store(value);
	if (!suppress_events) {
		NotifyForceNextNotification(cookie);
	}

}

void ObjectImpl<Checkable>::SetEnableEventHandler(bool value, bool suppress_events, const Value& cookie)
{
	m_EnableEventHandler.store(value);
	if (!suppress_events) {
		NotifyEnableEventHandler(cookie);
	}

}

void ObjectImpl<Checkable>::SetEnableNotifications(bool value, bool suppress_events, const Value& cookie)
{
	m_EnableNotifications.store(value);
	if (!suppress_events) {
		NotifyEnableNotifications(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastReachable(bool value, bool suppress_events, const Value& cookie)
{
	m_LastReachable.store(value);
	if (!suppress_events) {
		NotifyLastReachable(cookie);
	}

}

void ObjectImpl<Checkable>::SetEnableFlapping(bool value, bool suppress_events, const Value& cookie)
{
	m_EnableFlapping.store(value);
	if (!suppress_events) {
		NotifyEnableFlapping(cookie);
	}

}

void ObjectImpl<Checkable>::SetEnablePassiveChecks(bool value, bool suppress_events, const Value& cookie)
{
	m_EnablePassiveChecks.store(value);
	if (!suppress_events) {
		NotifyEnablePassiveChecks(cookie);
	}

}

void ObjectImpl<Checkable>::SetEnablePerfdata(bool value, bool suppress_events, const Value& cookie)
{
	m_EnablePerfdata.store(value);
	if (!suppress_events) {
		NotifyEnablePerfdata(cookie);
	}

}

void ObjectImpl<Checkable>::SetEnableActiveChecks(bool value, bool suppress_events, const Value& cookie)
{
	m_EnableActiveChecks.store(value);
	if (!suppress_events) {
		NotifyEnableActiveChecks(cookie);
	}

}

void ObjectImpl<Checkable>::SetVolatile(bool value, bool suppress_events, const Value& cookie)
{
	m_Volatile.store(value);
	if (!suppress_events) {
		NotifyVolatile(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastSoftStatesRaw(const unsigned short& value, bool suppress_events, const Value& cookie)
{
	m_LastSoftStatesRaw.store(value);
	if (!suppress_events) {
		NotifyLastSoftStatesRaw(cookie);
	}

}

void ObjectImpl<Checkable>::SetLastHardStatesRaw(const unsigned short& value, bool suppress_events, const Value& cookie)
{
	m_LastHardStatesRaw.store(value);
	if (!suppress_events) {
		NotifyLastHardStatesRaw(cookie);
	}

}

void ObjectImpl<Checkable>::TrackCommandEndpointRaw(const String& oldValue, const String& newValue)
{
	if (!oldValue.IsEmpty())
		DependencyGraph::RemoveDependency(this, ConfigObject::GetObject<Endpoint>(oldValue).get());
	if (!newValue.IsEmpty())
		DependencyGraph::AddDependency(this, ConfigObject::GetObject<Endpoint>(newValue).get());
}

void ObjectImpl<Checkable>::TrackCheckCommandRaw(const String& oldValue, const String& newValue)
{
	if (!oldValue.IsEmpty())
		DependencyGraph::RemoveDependency(this, ConfigObject::GetObject<CheckCommand>(oldValue).get());
	if (!newValue.IsEmpty())
		DependencyGraph::AddDependency(this, ConfigObject::GetObject<CheckCommand>(newValue).get());
}

void ObjectImpl<Checkable>::TrackCheckPeriodRaw(const String& oldValue, const String& newValue)
{
	if (!oldValue.IsEmpty())
		DependencyGraph::RemoveDependency(this, ConfigObject::GetObject<TimePeriod>(oldValue).get());
	if (!newValue.IsEmpty())
		DependencyGraph::AddDependency(this, ConfigObject::GetObject<TimePeriod>(newValue).get());
}

void ObjectImpl<Checkable>::TrackEventCommandRaw(const String& oldValue, const String& newValue)
{
	if (!oldValue.IsEmpty())
		DependencyGraph::RemoveDependency(this, ConfigObject::GetObject<EventCommand>(oldValue).get());
	if (!newValue.IsEmpty())
		DependencyGraph::AddDependency(this, ConfigObject::GetObject<EventCommand>(newValue).get());
}

Object::Ptr ObjectImpl<Checkable>::NavigateCommandEndpointRaw() const
{
	
			return Endpoint::GetByName(GetCommandEndpointRaw());
		
}

Object::Ptr ObjectImpl<Checkable>::NavigateCheckCommandRaw() const
{
	
			return CheckCommand::GetByName(GetCheckCommandRaw());
		
}

Object::Ptr ObjectImpl<Checkable>::NavigateCheckPeriodRaw() const
{
	
			return TimePeriod::GetByName(GetCheckPeriodRaw());
		
}

Object::Ptr ObjectImpl<Checkable>::NavigateEventCommandRaw() const
{
	
			return EventCommand::GetByName(GetEventCommandRaw());
		
}

void ObjectImpl<Checkable>::Start(bool runtimeCreated)
{
	CustomVarObject::Start(runtimeCreated);

	TrackCommandEndpointRaw(Empty, GetCommandEndpointRaw());
	TrackCheckCommandRaw(Empty, GetCheckCommandRaw());
	TrackCheckPeriodRaw(Empty, GetCheckPeriodRaw());
	TrackEventCommandRaw(Empty, GetEventCommandRaw());
}

void ObjectImpl<Checkable>::Stop(bool runtimeRemoved)
{
	CustomVarObject::Stop(runtimeRemoved);

	TrackCommandEndpointRaw(GetCommandEndpointRaw(), Empty);
	TrackCheckCommandRaw(GetCheckCommandRaw(), Empty);
	TrackCheckPeriodRaw(GetCheckPeriodRaw(), Empty);
	TrackEventCommandRaw(GetEventCommandRaw(), Empty);
}

void ObjectImpl<Checkable>::NotifyCheckTimeout(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCheckTimeoutChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyCommandEndpointRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCommandEndpointRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyCheckCommandRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCheckCommandRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyCheckPeriodRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCheckPeriodRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEventCommandRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEventCommandRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyIconImageAlt(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnIconImageAltChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyIconImage(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnIconImageChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyActionUrl(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnActionUrlChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyNotesUrl(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnNotesUrlChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyNotes(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnNotesChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyCheckInterval(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCheckIntervalChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyRetryInterval(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnRetryIntervalChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingThreshold(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingThresholdChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingCurrent(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingCurrentChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingThresholdLow(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingThresholdLowChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingThresholdHigh(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingThresholdHighChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingIgnoreStates(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingIgnoreStatesChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastCheckResult(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastCheckResultChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyExecutions(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnExecutionsChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyPendingExecutions(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnPendingExecutionsChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifySuppressedNotifications(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnSuppressedNotificationsChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyMaxCheckAttempts(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnMaxCheckAttemptsChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingIndex(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingIndexChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingBuffer(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingBufferChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyDowntimeDepth(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnDowntimeDepthChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyAcknowledgementRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnAcknowledgementRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifySeverity(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnSeverityChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyCheckAttempt(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCheckAttemptChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingIgnoreStatesFilter(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingIgnoreStatesFilterChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyStateRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStateRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingLastState(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingLastStateChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyStateBeforeSuppression(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStateBeforeSuppressionChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastHardStateRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastHardStateRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastStateRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastStateType(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateTypeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyStateType(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStateTypeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastHardStateChange(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastHardStateChangeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastStateChange(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateChangeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastStateUnreachable(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateUnreachableChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyPreviousStateChange(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnPreviousStateChangeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyNextUpdate(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnNextUpdateChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyAcknowledgementExpiry(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnAcknowledgementExpiryChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyAcknowledgementLastChange(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnAcknowledgementLastChangeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastCheckStarted(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastCheckStartedChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyNextCheck(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnNextCheckChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastCheck(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastCheckChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlappingLastChange(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingLastChangeChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyFlapping(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnFlappingChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyForceNextCheck(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnForceNextCheckChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyHandled(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnHandledChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyProblem(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnProblemChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyForceNextNotification(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnForceNextNotificationChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEnableEventHandler(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnableEventHandlerChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEnableNotifications(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnableNotificationsChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastReachable(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastReachableChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEnableFlapping(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnableFlappingChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEnablePassiveChecks(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnablePassiveChecksChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEnablePerfdata(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnablePerfdataChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyEnableActiveChecks(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnEnableActiveChecksChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyVolatile(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnVolatileChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastSoftStatesRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastSoftStatesRawChanged(static_cast<Checkable *>(this), cookie);
}

void ObjectImpl<Checkable>::NotifyLastHardStatesRaw(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastHardStatesRawChanged(static_cast<Checkable *>(this), cookie);
}

Value ObjectImpl<Checkable>::GetDefaultCheckTimeout() const
{
	return Value();
}

String ObjectImpl<Checkable>::GetDefaultCommandEndpointRaw() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultCheckCommandRaw() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultCheckPeriodRaw() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultEventCommandRaw() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultIconImageAlt() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultIconImage() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultActionUrl() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultNotesUrl() const
{
	return String();
}

String ObjectImpl<Checkable>::GetDefaultNotes() const
{
	return String();
}

double ObjectImpl<Checkable>::GetDefaultCheckInterval() const
{
	 return 5 * 60; 
}

double ObjectImpl<Checkable>::GetDefaultRetryInterval() const
{
	 return 60; 
}

double ObjectImpl<Checkable>::GetDefaultFlappingThreshold() const
{
	return double();
}

double ObjectImpl<Checkable>::GetDefaultFlappingCurrent() const
{
	 return 0; 
}

double ObjectImpl<Checkable>::GetDefaultFlappingThresholdLow() const
{
	 return 25; 
}

double ObjectImpl<Checkable>::GetDefaultFlappingThresholdHigh() const
{
	 return 30; 
}

Array::Ptr ObjectImpl<Checkable>::GetDefaultFlappingIgnoreStates() const
{
	return Array::Ptr();
}

CheckResult::Ptr ObjectImpl<Checkable>::GetDefaultLastCheckResult() const
{
	return CheckResult::Ptr();
}

Dictionary::Ptr ObjectImpl<Checkable>::GetDefaultExecutions() const
{
	return Dictionary::Ptr();
}

Dictionary::Ptr ObjectImpl<Checkable>::GetDefaultPendingExecutions() const
{
	return Dictionary::Ptr();
}

int ObjectImpl<Checkable>::GetDefaultSuppressedNotifications() const
{
	 return 0; 
}

int ObjectImpl<Checkable>::GetDefaultMaxCheckAttempts() const
{
	 return 3; 
}

int ObjectImpl<Checkable>::GetDefaultFlappingIndex() const
{
	return int();
}

int ObjectImpl<Checkable>::GetDefaultFlappingBuffer() const
{
	return int();
}

int ObjectImpl<Checkable>::GetDefaultDowntimeDepth() const
{
	return int();
}

int ObjectImpl<Checkable>::GetDefaultAcknowledgementRaw() const
{
	 return AcknowledgementNone; 
}

int ObjectImpl<Checkable>::GetDefaultSeverity() const
{
	return int();
}

int ObjectImpl<Checkable>::GetDefaultCheckAttempt() const
{
	 return 1; 
}

int ObjectImpl<Checkable>::GetDefaultFlappingIgnoreStatesFilter() const
{
	return int();
}

ServiceState ObjectImpl<Checkable>::GetDefaultStateRaw() const
{
	 return ServiceUnknown; 
}

ServiceState ObjectImpl<Checkable>::GetDefaultFlappingLastState() const
{
	 return ServiceUnknown; 
}

ServiceState ObjectImpl<Checkable>::GetDefaultStateBeforeSuppression() const
{
	 return ServiceOK; 
}

ServiceState ObjectImpl<Checkable>::GetDefaultLastHardStateRaw() const
{
	 return ServiceUnknown; 
}

ServiceState ObjectImpl<Checkable>::GetDefaultLastStateRaw() const
{
	 return ServiceUnknown; 
}

StateType ObjectImpl<Checkable>::GetDefaultLastStateType() const
{
	 return StateTypeSoft; 
}

StateType ObjectImpl<Checkable>::GetDefaultStateType() const
{
	 return StateTypeSoft; 
}

Timestamp ObjectImpl<Checkable>::GetDefaultLastHardStateChange() const
{
	 return Application::GetStartTime(); 
}

Timestamp ObjectImpl<Checkable>::GetDefaultLastStateChange() const
{
	 return Application::GetStartTime(); 
}

Timestamp ObjectImpl<Checkable>::GetDefaultLastStateUnreachable() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultPreviousStateChange() const
{
	 return Application::GetStartTime(); 
}

Timestamp ObjectImpl<Checkable>::GetDefaultNextUpdate() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultAcknowledgementExpiry() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultAcknowledgementLastChange() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultLastCheckStarted() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultNextCheck() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultLastCheck() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Checkable>::GetDefaultFlappingLastChange() const
{
	return Timestamp();
}

bool ObjectImpl<Checkable>::GetDefaultFlapping() const
{
	return bool();
}

bool ObjectImpl<Checkable>::GetDefaultForceNextCheck() const
{
	return bool();
}

bool ObjectImpl<Checkable>::GetDefaultHandled() const
{
	return bool();
}

bool ObjectImpl<Checkable>::GetDefaultProblem() const
{
	return bool();
}

bool ObjectImpl<Checkable>::GetDefaultForceNextNotification() const
{
	return bool();
}

bool ObjectImpl<Checkable>::GetDefaultEnableEventHandler() const
{
	 return true; 
}

bool ObjectImpl<Checkable>::GetDefaultEnableNotifications() const
{
	 return true; 
}

bool ObjectImpl<Checkable>::GetDefaultLastReachable() const
{
	 return true; 
}

bool ObjectImpl<Checkable>::GetDefaultEnableFlapping() const
{
	 return false; 
}

bool ObjectImpl<Checkable>::GetDefaultEnablePassiveChecks() const
{
	 return true; 
}

bool ObjectImpl<Checkable>::GetDefaultEnablePerfdata() const
{
	 return true; 
}

bool ObjectImpl<Checkable>::GetDefaultEnableActiveChecks() const
{
	 return true; 
}

bool ObjectImpl<Checkable>::GetDefaultVolatile() const
{
	return bool();
}

unsigned short ObjectImpl<Checkable>::GetDefaultLastSoftStatesRaw() const
{
	 return /* current */ 99 * 100 + /* previous */ 99; 
}

unsigned short ObjectImpl<Checkable>::GetDefaultLastHardStatesRaw() const
{
	 return /* current */ 99 * 100 + /* previous */ 99; 
}


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnCheckTimeoutChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnCommandEndpointRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnCheckCommandRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnCheckPeriodRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEventCommandRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnIconImageAltChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnIconImageChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnActionUrlChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnNotesUrlChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnNotesChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnCheckIntervalChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnRetryIntervalChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingThresholdChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingCurrentChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingThresholdLowChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingThresholdHighChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingIgnoreStatesChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastCheckResultChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnExecutionsChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnPendingExecutionsChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnSuppressedNotificationsChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnMaxCheckAttemptsChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingIndexChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingBufferChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnDowntimeDepthChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnAcknowledgementRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnSeverityChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnCheckAttemptChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingIgnoreStatesFilterChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnStateRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingLastStateChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnStateBeforeSuppressionChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastHardStateRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastStateRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastStateTypeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnStateTypeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastHardStateChangeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastStateChangeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastStateUnreachableChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnPreviousStateChangeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnNextUpdateChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnAcknowledgementExpiryChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnAcknowledgementLastChangeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastCheckStartedChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnNextCheckChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastCheckChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingLastChangeChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnFlappingChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnForceNextCheckChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnHandledChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnProblemChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnForceNextNotificationChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEnableEventHandlerChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEnableNotificationsChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastReachableChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEnableFlappingChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEnablePassiveChecksChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEnablePerfdataChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnEnableActiveChecksChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnVolatileChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastSoftStatesRawChanged;


boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> ObjectImpl<Checkable>::OnLastHardStatesRawChanged;

void ObjectImpl<Checkable>::ValidateAcknowledgementExpiry(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateAcknowledgementExpiry(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateAcknowledgementLastChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateAcknowledgementLastChange(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateAcknowledgementRaw(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateAcknowledgementRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateActionUrl(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateActionUrl(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateCheckAttempt(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCheckAttempt(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateCheckCommandRaw(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCheckCommandRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateCheckInterval(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCheckInterval(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateCheckPeriodRaw(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCheckPeriodRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateCheckTimeout(const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCheckTimeout(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateCommandEndpointRaw(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCommandEndpointRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateDowntimeDepth(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateDowntimeDepth(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEnableActiveChecks(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnableActiveChecks(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEnableEventHandler(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnableEventHandler(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEnableFlapping(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnableFlapping(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEnableNotifications(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnableNotifications(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEnablePassiveChecks(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnablePassiveChecks(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEnablePerfdata(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEnablePerfdata(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateEventCommandRaw(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateEventCommandRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateExecutions(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateExecutions(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlapping(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlapping(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingBuffer(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingBuffer(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingCurrent(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingCurrent(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingIgnoreStates(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingIgnoreStates(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingIgnoreStatesFilter(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingIgnoreStatesFilter(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingIndex(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingIndex(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingLastChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingLastChange(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingLastState(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingLastState(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingThreshold(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingThreshold(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingThresholdHigh(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingThresholdHigh(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateFlappingThresholdLow(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateFlappingThresholdLow(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateForceNextCheck(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateForceNextCheck(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateForceNextNotification(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateForceNextNotification(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateHandled(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateHandled(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateIconImage(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateIconImage(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateIconImageAlt(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateIconImageAlt(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastCheck(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastCheck(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastCheckResult(const Lazy<CheckResult::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastCheckResult(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastCheckStarted(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastCheckStarted(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastHardStateChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastHardStateChange(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastHardStateRaw(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastHardStateRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastHardStatesRaw(const Lazy<unsigned short>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastHardStatesRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastReachable(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastReachable(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastSoftStatesRaw(const Lazy<unsigned short>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastSoftStatesRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastStateChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateChange(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastStateRaw(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastStateType(const Lazy<StateType>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateType(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateLastStateUnreachable(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateUnreachable(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateMaxCheckAttempts(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateMaxCheckAttempts(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateNextCheck(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateNextCheck(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateNextUpdate(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateNextUpdate(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateNotes(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateNotes(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateNotesUrl(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateNotesUrl(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidatePendingExecutions(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidatePendingExecutions(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidatePreviousStateChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidatePreviousStateChange(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateProblem(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateProblem(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateRetryInterval(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateRetryInterval(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateSeverity(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateSeverity(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateStateBeforeSuppression(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateStateBeforeSuppression(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateStateRaw(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateStateRaw(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateStateType(const Lazy<StateType>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateStateType(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateSuppressedNotifications(const Lazy<int>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateSuppressedNotifications(lvalue, utils);
}

void ObjectImpl<Checkable>::ValidateVolatile(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateVolatile(lvalue, utils);
}

}
#ifdef _MSC_VER
#pragma warning ( pop )
#endif /* _MSC_VER */
