#ifndef CHECKABLE_TI
#define CHECKABLE_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "icinga/icingaapplication.hpp"

#include "icinga/customvarobject.hpp"

#include "base/array.hpp"

namespace icinga
{


/**
 * The acknowledgement type of a service.
 *
 * @ingroup icinga
 */
enum AcknowledgementType
{
	AcknowledgementNone = 0,
	AcknowledgementNormal = 1,
	AcknowledgementSticky = 2
};

class Checkable;

template<>
struct TypeHelper<Checkable, false>
{
	static ObjectFactory GetFactory()
	{
		return nullptr;
	}
};

template<>
class TypeImpl<Checkable> : public TypeImpl<CustomVarObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<Checkable>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<Checkable> : public CustomVarObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<Checkable>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateCheckTimeout(const Lazy<Value>& value, const ValidationUtils& utils);
	void SimpleValidateCommandEndpointRaw(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCheckCommandRaw(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCheckPeriodRaw(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateEventCommandRaw(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateIconImageAlt(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateIconImage(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateActionUrl(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateNotesUrl(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateNotes(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCheckInterval(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateRetryInterval(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingThreshold(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingCurrent(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingThresholdLow(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingThresholdHigh(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingIgnoreStates(const Lazy<Array::Ptr>& avalue, const ValidationUtils& utils);
	void SimpleValidateLastCheckResult(const Lazy<CheckResult::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateExecutions(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidatePendingExecutions(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateSuppressedNotifications(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateMaxCheckAttempts(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingIndex(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingBuffer(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateDowntimeDepth(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateAcknowledgementRaw(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateSeverity(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateCheckAttempt(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingIgnoreStatesFilter(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateStateRaw(const Lazy<ServiceState>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingLastState(const Lazy<ServiceState>& value, const ValidationUtils& utils);
	void SimpleValidateStateBeforeSuppression(const Lazy<ServiceState>& value, const ValidationUtils& utils);
	void SimpleValidateLastHardStateRaw(const Lazy<ServiceState>& value, const ValidationUtils& utils);
	void SimpleValidateLastStateRaw(const Lazy<ServiceState>& value, const ValidationUtils& utils);
	void SimpleValidateLastStateType(const Lazy<StateType>& value, const ValidationUtils& utils);
	void SimpleValidateStateType(const Lazy<StateType>& value, const ValidationUtils& utils);
	void SimpleValidateLastHardStateChange(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateLastStateChange(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateLastStateUnreachable(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidatePreviousStateChange(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateNextUpdate(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateAcknowledgementExpiry(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateAcknowledgementLastChange(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateLastCheckStarted(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateNextCheck(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateLastCheck(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateFlappingLastChange(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateFlapping(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateForceNextCheck(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateHandled(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateProblem(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateForceNextNotification(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableEventHandler(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableNotifications(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateLastReachable(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableFlapping(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnablePassiveChecks(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnablePerfdata(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableActiveChecks(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateVolatile(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateLastSoftStatesRaw(const Lazy<unsigned short>& value, const ValidationUtils& utils);
	void SimpleValidateLastHardStatesRaw(const Lazy<unsigned short>& value, const ValidationUtils& utils);
public:
	ObjectImpl<Checkable>();
public:
	~ObjectImpl<Checkable>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	Value GetCheckTimeout() const;
public:
	String GetCommandEndpointRaw() const;
public:
	String GetCheckCommandRaw() const;
public:
	String GetCheckPeriodRaw() const;
public:
	String GetEventCommandRaw() const;
public:
	String GetIconImageAlt() const;
public:
	String GetIconImage() const;
public:
	String GetActionUrl() const;
public:
	String GetNotesUrl() const;
public:
	String GetNotes() const;
public:
	double GetCheckInterval() const;
public:
	double GetRetryInterval() const;
public:
	double GetFlappingThreshold() const;
public:
	double GetFlappingCurrent() const;
public:
	double GetFlappingThresholdLow() const;
public:
	double GetFlappingThresholdHigh() const;
public:
	Array::Ptr GetFlappingIgnoreStates() const;
public:
	CheckResult::Ptr GetLastCheckResult() const;
public:
	Dictionary::Ptr GetExecutions() const;
public:
	Dictionary::Ptr GetPendingExecutions() const;
public:
	int GetSuppressedNotifications() const;
public:
	int GetMaxCheckAttempts() const;
public:
	int GetFlappingIndex() const;
public:
	int GetFlappingBuffer() const;
public:
	virtual int GetDowntimeDepth() const = 0;
public:
	int GetAcknowledgementRaw() const;
public:
	virtual int GetSeverity() const = 0;
public:
	int GetCheckAttempt() const;
public:
	int GetFlappingIgnoreStatesFilter() const;
public:
	ServiceState GetStateRaw() const;
public:
	ServiceState GetFlappingLastState() const;
public:
	ServiceState GetStateBeforeSuppression() const;
public:
	ServiceState GetLastHardStateRaw() const;
public:
	ServiceState GetLastStateRaw() const;
public:
	StateType GetLastStateType() const;
public:
	StateType GetStateType() const;
public:
	Timestamp GetLastHardStateChange() const;
public:
	Timestamp GetLastStateChange() const;
public:
	Timestamp GetLastStateUnreachable() const;
public:
	Timestamp GetPreviousStateChange() const;
public:
	virtual Timestamp GetNextUpdate() const = 0;
public:
	Timestamp GetAcknowledgementExpiry() const;
public:
	Timestamp GetAcknowledgementLastChange() const;
public:
	Timestamp GetLastCheckStarted() const;
public:
	Timestamp GetNextCheck() const;
public:
	virtual Timestamp GetLastCheck() const = 0;
public:
	Timestamp GetFlappingLastChange() const;
protected:
	bool GetFlapping() const;
public:
	bool GetForceNextCheck() const;
public:
	virtual bool GetHandled() const = 0;
public:
	virtual bool GetProblem() const = 0;
public:
	bool GetForceNextNotification() const;
public:
	bool GetEnableEventHandler() const;
public:
	bool GetEnableNotifications() const;
public:
	bool GetLastReachable() const;
public:
	bool GetEnableFlapping() const;
public:
	bool GetEnablePassiveChecks() const;
public:
	bool GetEnablePerfdata() const;
public:
	bool GetEnableActiveChecks() const;
public:
	bool GetVolatile() const;
public:
	unsigned short GetLastSoftStatesRaw() const;
public:
	unsigned short GetLastHardStatesRaw() const;
public:
	void SetCheckTimeout(const Value& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCommandEndpointRaw(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCheckCommandRaw(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCheckPeriodRaw(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEventCommandRaw(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetIconImageAlt(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetIconImage(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetActionUrl(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetNotesUrl(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetNotes(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCheckInterval(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetRetryInterval(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingThreshold(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingCurrent(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingThresholdLow(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingThresholdHigh(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingIgnoreStates(const Array::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastCheckResult(const CheckResult::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetExecutions(const Dictionary::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPendingExecutions(const Dictionary::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSuppressedNotifications(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetMaxCheckAttempts(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingIndex(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingBuffer(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetDowntimeDepth(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetAcknowledgementRaw(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSeverity(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCheckAttempt(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingIgnoreStatesFilter(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetStateRaw(const ServiceState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingLastState(const ServiceState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetStateBeforeSuppression(const ServiceState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastHardStateRaw(const ServiceState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastStateRaw(const ServiceState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastStateType(const StateType& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetStateType(const StateType& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastHardStateChange(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastStateChange(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastStateUnreachable(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPreviousStateChange(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetNextUpdate(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetAcknowledgementExpiry(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetAcknowledgementLastChange(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastCheckStarted(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetNextCheck(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastCheck(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlappingLastChange(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void SetFlapping(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetForceNextCheck(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetHandled(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetProblem(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetForceNextNotification(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableEventHandler(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableNotifications(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastReachable(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableFlapping(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnablePassiveChecks(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnablePerfdata(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableActiveChecks(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetVolatile(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastSoftStatesRaw(const unsigned short& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastHardStatesRaw(const unsigned short& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void TrackCommandEndpointRaw(const String& oldValue, const String& newValue);
	void TrackCheckCommandRaw(const String& oldValue, const String& newValue);
	void TrackCheckPeriodRaw(const String& oldValue, const String& newValue);
	void TrackEventCommandRaw(const String& oldValue, const String& newValue);
public:
	Object::Ptr NavigateCommandEndpointRaw() const;
public:
	Object::Ptr NavigateCheckCommandRaw() const;
public:
	Object::Ptr NavigateCheckPeriodRaw() const;
public:
	Object::Ptr NavigateEventCommandRaw() const;
protected:
	void Start(bool runtimeCreated = false) override;
	void Stop(bool runtimeRemoved = false) override;
public:
	virtual void NotifyCheckTimeout(const Value& cookie = Empty);
public:
	virtual void NotifyCommandEndpointRaw(const Value& cookie = Empty);
public:
	virtual void NotifyCheckCommandRaw(const Value& cookie = Empty);
public:
	virtual void NotifyCheckPeriodRaw(const Value& cookie = Empty);
public:
	virtual void NotifyEventCommandRaw(const Value& cookie = Empty);
public:
	virtual void NotifyIconImageAlt(const Value& cookie = Empty);
public:
	virtual void NotifyIconImage(const Value& cookie = Empty);
public:
	virtual void NotifyActionUrl(const Value& cookie = Empty);
public:
	virtual void NotifyNotesUrl(const Value& cookie = Empty);
public:
	virtual void NotifyNotes(const Value& cookie = Empty);
public:
	virtual void NotifyCheckInterval(const Value& cookie = Empty);
public:
	virtual void NotifyRetryInterval(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingThreshold(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingCurrent(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingThresholdLow(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingThresholdHigh(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingIgnoreStates(const Value& cookie = Empty);
public:
	virtual void NotifyLastCheckResult(const Value& cookie = Empty);
public:
	virtual void NotifyExecutions(const Value& cookie = Empty);
public:
	virtual void NotifyPendingExecutions(const Value& cookie = Empty);
public:
	virtual void NotifySuppressedNotifications(const Value& cookie = Empty);
public:
	virtual void NotifyMaxCheckAttempts(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingIndex(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingBuffer(const Value& cookie = Empty);
public:
	virtual void NotifyDowntimeDepth(const Value& cookie = Empty);
public:
	virtual void NotifyAcknowledgementRaw(const Value& cookie = Empty);
public:
	virtual void NotifySeverity(const Value& cookie = Empty);
public:
	virtual void NotifyCheckAttempt(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingIgnoreStatesFilter(const Value& cookie = Empty);
public:
	virtual void NotifyStateRaw(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingLastState(const Value& cookie = Empty);
public:
	virtual void NotifyStateBeforeSuppression(const Value& cookie = Empty);
public:
	virtual void NotifyLastHardStateRaw(const Value& cookie = Empty);
public:
	virtual void NotifyLastStateRaw(const Value& cookie = Empty);
public:
	virtual void NotifyLastStateType(const Value& cookie = Empty);
public:
	virtual void NotifyStateType(const Value& cookie = Empty);
public:
	virtual void NotifyLastHardStateChange(const Value& cookie = Empty);
public:
	virtual void NotifyLastStateChange(const Value& cookie = Empty);
public:
	virtual void NotifyLastStateUnreachable(const Value& cookie = Empty);
public:
	virtual void NotifyPreviousStateChange(const Value& cookie = Empty);
public:
	virtual void NotifyNextUpdate(const Value& cookie = Empty);
public:
	virtual void NotifyAcknowledgementExpiry(const Value& cookie = Empty);
public:
	virtual void NotifyAcknowledgementLastChange(const Value& cookie = Empty);
public:
	virtual void NotifyLastCheckStarted(const Value& cookie = Empty);
public:
	virtual void NotifyNextCheck(const Value& cookie = Empty);
public:
	virtual void NotifyLastCheck(const Value& cookie = Empty);
public:
	virtual void NotifyFlappingLastChange(const Value& cookie = Empty);
protected:
	virtual void NotifyFlapping(const Value& cookie = Empty);
public:
	virtual void NotifyForceNextCheck(const Value& cookie = Empty);
public:
	virtual void NotifyHandled(const Value& cookie = Empty);
public:
	virtual void NotifyProblem(const Value& cookie = Empty);
public:
	virtual void NotifyForceNextNotification(const Value& cookie = Empty);
public:
	virtual void NotifyEnableEventHandler(const Value& cookie = Empty);
public:
	virtual void NotifyEnableNotifications(const Value& cookie = Empty);
public:
	virtual void NotifyLastReachable(const Value& cookie = Empty);
public:
	virtual void NotifyEnableFlapping(const Value& cookie = Empty);
public:
	virtual void NotifyEnablePassiveChecks(const Value& cookie = Empty);
public:
	virtual void NotifyEnablePerfdata(const Value& cookie = Empty);
public:
	virtual void NotifyEnableActiveChecks(const Value& cookie = Empty);
public:
	virtual void NotifyVolatile(const Value& cookie = Empty);
public:
	virtual void NotifyLastSoftStatesRaw(const Value& cookie = Empty);
public:
	virtual void NotifyLastHardStatesRaw(const Value& cookie = Empty);
private:
	inline Value GetDefaultCheckTimeout() const;
private:
	inline String GetDefaultCommandEndpointRaw() const;
private:
	inline String GetDefaultCheckCommandRaw() const;
private:
	inline String GetDefaultCheckPeriodRaw() const;
private:
	inline String GetDefaultEventCommandRaw() const;
private:
	inline String GetDefaultIconImageAlt() const;
private:
	inline String GetDefaultIconImage() const;
private:
	inline String GetDefaultActionUrl() const;
private:
	inline String GetDefaultNotesUrl() const;
private:
	inline String GetDefaultNotes() const;
private:
	inline double GetDefaultCheckInterval() const;
private:
	inline double GetDefaultRetryInterval() const;
private:
	inline double GetDefaultFlappingThreshold() const;
private:
	inline double GetDefaultFlappingCurrent() const;
private:
	inline double GetDefaultFlappingThresholdLow() const;
private:
	inline double GetDefaultFlappingThresholdHigh() const;
private:
	inline Array::Ptr GetDefaultFlappingIgnoreStates() const;
private:
	inline CheckResult::Ptr GetDefaultLastCheckResult() const;
private:
	inline Dictionary::Ptr GetDefaultExecutions() const;
private:
	inline Dictionary::Ptr GetDefaultPendingExecutions() const;
private:
	inline int GetDefaultSuppressedNotifications() const;
private:
	inline int GetDefaultMaxCheckAttempts() const;
private:
	inline int GetDefaultFlappingIndex() const;
private:
	inline int GetDefaultFlappingBuffer() const;
private:
	inline int GetDefaultDowntimeDepth() const;
private:
	inline int GetDefaultAcknowledgementRaw() const;
private:
	inline int GetDefaultSeverity() const;
private:
	inline int GetDefaultCheckAttempt() const;
private:
	inline int GetDefaultFlappingIgnoreStatesFilter() const;
private:
	inline ServiceState GetDefaultStateRaw() const;
private:
	inline ServiceState GetDefaultFlappingLastState() const;
private:
	inline ServiceState GetDefaultStateBeforeSuppression() const;
private:
	inline ServiceState GetDefaultLastHardStateRaw() const;
private:
	inline ServiceState GetDefaultLastStateRaw() const;
private:
	inline StateType GetDefaultLastStateType() const;
private:
	inline StateType GetDefaultStateType() const;
private:
	inline Timestamp GetDefaultLastHardStateChange() const;
private:
	inline Timestamp GetDefaultLastStateChange() const;
private:
	inline Timestamp GetDefaultLastStateUnreachable() const;
private:
	inline Timestamp GetDefaultPreviousStateChange() const;
private:
	inline Timestamp GetDefaultNextUpdate() const;
private:
	inline Timestamp GetDefaultAcknowledgementExpiry() const;
private:
	inline Timestamp GetDefaultAcknowledgementLastChange() const;
private:
	inline Timestamp GetDefaultLastCheckStarted() const;
private:
	inline Timestamp GetDefaultNextCheck() const;
private:
	inline Timestamp GetDefaultLastCheck() const;
private:
	inline Timestamp GetDefaultFlappingLastChange() const;
private:
	inline bool GetDefaultFlapping() const;
private:
	inline bool GetDefaultForceNextCheck() const;
private:
	inline bool GetDefaultHandled() const;
private:
	inline bool GetDefaultProblem() const;
private:
	inline bool GetDefaultForceNextNotification() const;
private:
	inline bool GetDefaultEnableEventHandler() const;
private:
	inline bool GetDefaultEnableNotifications() const;
private:
	inline bool GetDefaultLastReachable() const;
private:
	inline bool GetDefaultEnableFlapping() const;
private:
	inline bool GetDefaultEnablePassiveChecks() const;
private:
	inline bool GetDefaultEnablePerfdata() const;
private:
	inline bool GetDefaultEnableActiveChecks() const;
private:
	inline bool GetDefaultVolatile() const;
private:
	inline unsigned short GetDefaultLastSoftStatesRaw() const;
private:
	inline unsigned short GetDefaultLastHardStatesRaw() const;
protected:
	virtual void ValidateCheckTimeout(const Lazy<Value>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCommandEndpointRaw(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCheckCommandRaw(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCheckPeriodRaw(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEventCommandRaw(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateIconImageAlt(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateIconImage(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateActionUrl(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateNotesUrl(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateNotes(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCheckInterval(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateRetryInterval(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingThreshold(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingCurrent(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingThresholdLow(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingThresholdHigh(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingIgnoreStates(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastCheckResult(const Lazy<CheckResult::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateExecutions(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePendingExecutions(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSuppressedNotifications(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateMaxCheckAttempts(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingIndex(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingBuffer(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateDowntimeDepth(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateAcknowledgementRaw(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSeverity(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCheckAttempt(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingIgnoreStatesFilter(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateStateRaw(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingLastState(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateStateBeforeSuppression(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastHardStateRaw(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastStateRaw(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastStateType(const Lazy<StateType>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateStateType(const Lazy<StateType>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastHardStateChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastStateChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastStateUnreachable(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePreviousStateChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateNextUpdate(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateAcknowledgementExpiry(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateAcknowledgementLastChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastCheckStarted(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateNextCheck(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastCheck(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlappingLastChange(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlapping(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateForceNextCheck(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateHandled(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateProblem(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateForceNextNotification(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableEventHandler(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableNotifications(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastReachable(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableFlapping(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnablePassiveChecks(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnablePerfdata(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableActiveChecks(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateVolatile(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastSoftStatesRaw(const Lazy<unsigned short>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastHardStatesRaw(const Lazy<unsigned short>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<Value> m_CheckTimeout;
	AtomicOrLocked<String> m_CommandEndpointRaw;
	AtomicOrLocked<String> m_CheckCommandRaw;
	AtomicOrLocked<String> m_CheckPeriodRaw;
	AtomicOrLocked<String> m_EventCommandRaw;
	AtomicOrLocked<String> m_IconImageAlt;
	AtomicOrLocked<String> m_IconImage;
	AtomicOrLocked<String> m_ActionUrl;
	AtomicOrLocked<String> m_NotesUrl;
	AtomicOrLocked<String> m_Notes;
	AtomicOrLocked<double> m_CheckInterval;
	AtomicOrLocked<double> m_RetryInterval;
	AtomicOrLocked<double> m_FlappingThreshold;
	AtomicOrLocked<double> m_FlappingCurrent;
	AtomicOrLocked<double> m_FlappingThresholdLow;
	AtomicOrLocked<double> m_FlappingThresholdHigh;
	AtomicOrLocked<Array::Ptr> m_FlappingIgnoreStates;
	AtomicOrLocked<CheckResult::Ptr> m_LastCheckResult;
	AtomicOrLocked<Dictionary::Ptr> m_Executions;
	AtomicOrLocked<Dictionary::Ptr> m_PendingExecutions;
	AtomicOrLocked<int> m_SuppressedNotifications;
	AtomicOrLocked<int> m_MaxCheckAttempts;
	AtomicOrLocked<int> m_FlappingIndex;
	AtomicOrLocked<int> m_FlappingBuffer;
	AtomicOrLocked<int> m_AcknowledgementRaw;
	AtomicOrLocked<int> m_CheckAttempt;
	AtomicOrLocked<int> m_FlappingIgnoreStatesFilter;
	AtomicOrLocked<ServiceState> m_StateRaw;
	AtomicOrLocked<ServiceState> m_FlappingLastState;
	AtomicOrLocked<ServiceState> m_StateBeforeSuppression;
	AtomicOrLocked<ServiceState> m_LastHardStateRaw;
	AtomicOrLocked<ServiceState> m_LastStateRaw;
	AtomicOrLocked<StateType> m_LastStateType;
	AtomicOrLocked<StateType> m_StateType;
	AtomicOrLocked<Timestamp> m_LastHardStateChange;
	AtomicOrLocked<Timestamp> m_LastStateChange;
	AtomicOrLocked<Timestamp> m_LastStateUnreachable;
	AtomicOrLocked<Timestamp> m_PreviousStateChange;
	AtomicOrLocked<Timestamp> m_AcknowledgementExpiry;
	AtomicOrLocked<Timestamp> m_AcknowledgementLastChange;
	AtomicOrLocked<Timestamp> m_LastCheckStarted;
	AtomicOrLocked<Timestamp> m_NextCheck;
	AtomicOrLocked<Timestamp> m_FlappingLastChange;
	AtomicOrLocked<bool> m_Flapping;
	AtomicOrLocked<bool> m_ForceNextCheck;
	AtomicOrLocked<bool> m_ForceNextNotification;
	AtomicOrLocked<bool> m_EnableEventHandler;
	AtomicOrLocked<bool> m_EnableNotifications;
	AtomicOrLocked<bool> m_LastReachable;
	AtomicOrLocked<bool> m_EnableFlapping;
	AtomicOrLocked<bool> m_EnablePassiveChecks;
	AtomicOrLocked<bool> m_EnablePerfdata;
	AtomicOrLocked<bool> m_EnableActiveChecks;
	AtomicOrLocked<bool> m_Volatile;
	AtomicOrLocked<unsigned short> m_LastSoftStatesRaw;
	AtomicOrLocked<unsigned short> m_LastHardStatesRaw;
public:
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnCheckTimeoutChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnCommandEndpointRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnCheckCommandRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnCheckPeriodRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEventCommandRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnIconImageAltChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnIconImageChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnActionUrlChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnNotesUrlChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnNotesChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnCheckIntervalChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnRetryIntervalChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingThresholdChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingCurrentChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingThresholdLowChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingThresholdHighChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingIgnoreStatesChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastCheckResultChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnExecutionsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnPendingExecutionsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnSuppressedNotificationsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnMaxCheckAttemptsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingIndexChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingBufferChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnDowntimeDepthChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnAcknowledgementRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnSeverityChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnCheckAttemptChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingIgnoreStatesFilterChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnStateRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingLastStateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnStateBeforeSuppressionChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastHardStateRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastStateRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastStateTypeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnStateTypeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastHardStateChangeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastStateChangeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastStateUnreachableChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnPreviousStateChangeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnNextUpdateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnAcknowledgementExpiryChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnAcknowledgementLastChangeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastCheckStartedChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnNextCheckChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastCheckChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingLastChangeChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnFlappingChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnForceNextCheckChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnHandledChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnProblemChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnForceNextNotificationChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEnableEventHandlerChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEnableNotificationsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastReachableChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEnableFlappingChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEnablePassiveChecksChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEnablePerfdataChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnEnableActiveChecksChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnVolatileChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastSoftStatesRawChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Checkable>&, const Value&)> OnLastHardStatesRawChanged;
};

}
#endif /* CHECKABLE_TI */
