#ifndef HOST_TI
#define HOST_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "icinga/checkable.hpp"

#include "icinga/customvarobject.hpp"

namespace icinga
{

class Host;

template<>
class TypeImpl<Host> : public TypeImpl<Checkable>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<Host>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	const std::unordered_set<Type*>& GetLoadDependencies() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<Host> : public Checkable
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<Host>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateDisplayName(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateAddress(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateAddress6(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateGroups(const Lazy<Array::Ptr>& avalue, const ValidationUtils& utils);
	void SimpleValidateState(const Lazy<HostState>& value, const ValidationUtils& utils);
	void SimpleValidateLastState(const Lazy<HostState>& value, const ValidationUtils& utils);
	void SimpleValidateLastHardState(const Lazy<HostState>& value, const ValidationUtils& utils);
	void SimpleValidateLastStateUp(const Lazy<Timestamp>& value, const ValidationUtils& utils);
	void SimpleValidateLastStateDown(const Lazy<Timestamp>& value, const ValidationUtils& utils);
public:
	ObjectImpl<Host>();
public:
	~ObjectImpl<Host>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetDisplayName() const;
public:
	String GetAddress() const;
public:
	String GetAddress6() const;
public:
	Array::Ptr GetGroups() const;
public:
	virtual HostState GetState() const = 0;
public:
	virtual HostState GetLastState() const = 0;
public:
	virtual HostState GetLastHardState() const = 0;
public:
	Timestamp GetLastStateUp() const;
public:
	Timestamp GetLastStateDown() const;
public:
	void SetDisplayName(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetAddress(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetAddress6(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetGroups(const Array::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetState(const HostState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastState(const HostState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastHardState(const HostState& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastStateUp(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetLastStateDown(const Timestamp& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void TrackGroups(const Array::Ptr& oldValue, const Array::Ptr& newValue);
protected:
	void Start(bool runtimeCreated = false) override;
	void Stop(bool runtimeRemoved = false) override;
public:
	virtual void NotifyDisplayName(const Value& cookie = Empty);
public:
	virtual void NotifyAddress(const Value& cookie = Empty);
public:
	virtual void NotifyAddress6(const Value& cookie = Empty);
public:
	virtual void NotifyGroups(const Value& cookie = Empty);
public:
	virtual void NotifyState(const Value& cookie = Empty);
public:
	virtual void NotifyLastState(const Value& cookie = Empty);
public:
	virtual void NotifyLastHardState(const Value& cookie = Empty);
public:
	virtual void NotifyLastStateUp(const Value& cookie = Empty);
public:
	virtual void NotifyLastStateDown(const Value& cookie = Empty);
private:
	inline String GetDefaultDisplayName() const;
private:
	inline String GetDefaultAddress() const;
private:
	inline String GetDefaultAddress6() const;
private:
	inline Array::Ptr GetDefaultGroups() const;
private:
	inline HostState GetDefaultState() const;
private:
	inline HostState GetDefaultLastState() const;
private:
	inline HostState GetDefaultLastHardState() const;
private:
	inline Timestamp GetDefaultLastStateUp() const;
private:
	inline Timestamp GetDefaultLastStateDown() const;
protected:
	virtual void ValidateDisplayName(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateAddress(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateAddress6(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateGroups(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateState(const Lazy<HostState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastState(const Lazy<HostState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastHardState(const Lazy<HostState>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastStateUp(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastStateDown(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_DisplayName;
	AtomicOrLocked<String> m_Address;
	AtomicOrLocked<String> m_Address6;
	AtomicOrLocked<Array::Ptr> m_Groups;
	AtomicOrLocked<Timestamp> m_LastStateUp;
	AtomicOrLocked<Timestamp> m_LastStateDown;
public:
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnDisplayNameChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnAddressChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnAddress6Changed;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnGroupsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&, const Value&)> OnGroupsChangedWithOldValue;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnStateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnLastStateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnLastHardStateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnLastStateUpChanged;
	static boost::signals2::signal<void (const intrusive_ptr<Host>&, const Value&)> OnLastStateDownChanged;
};

}
#endif /* HOST_TI */
