#include "base/exception.hpp"
#include "base/objectlock.hpp"
#include "base/utility.hpp"
#include "base/convert.hpp"
#include "base/debug.hpp"
#include "base/dependencygraph.hpp"
#include "base/logger.hpp"
#include "base/function.hpp"
#include "base/configobject.hpp"
#include "base/configtype.hpp"
#ifdef _MSC_VER
#pragma warning( push )
#pragma warning( disable : 4244 )
#pragma warning( disable : 4800 )
#endif /* _MSC_VER */

#include "icinga/servicegroup.hpp"

namespace icinga
{

TypeImpl<Service>::TypeImpl()
{ }

TypeImpl<Service>::~TypeImpl()
{ }

String TypeImpl<Service>::GetName() const
{
	return "Service";
}

int TypeImpl<Service>::GetAttributes() const
{
	return 0;
}

Type::Ptr TypeImpl<Service>::GetBaseType() const
{
	return Checkable::TypeInstance;
}

int TypeImpl<Service>::GetFieldId(const String& name) const
{
	int offset = Checkable::TypeInstance->GetFieldCount();

	switch (static_cast<int>(Utility::SDBM(name, 6))) {
		case -1851841644:
			if (name == "groups")
				return offset + 2;

			break;
		case -1785555177:
			if (name == "host_name")
				return offset + 1;

			break;
		case -1553228489:
			if (name == "display_name")
				return offset + 0;

			break;
		case -931315503:
			if (name == "state")
				return offset + 4;

			break;
		case 501261128:
			if (name == "host")
				return offset + 3;

			break;
		case 928442879:
			if (name == "last_hard_state")
				return offset + 6;

			break;
		case 928442890:
			if (name == "last_state")
				return offset + 5;
			if (name == "last_state_ok")
				return offset + 7;
			if (name == "last_state_warning")
				return offset + 8;
			if (name == "last_state_critical")
				return offset + 9;
			if (name == "last_state_unknown")
				return offset + 10;

			break;
	}

	return Checkable::TypeInstance->GetFieldId(name);
}

Field TypeImpl<Service>::GetFieldInfo(int id) const
{
	int real_id = id - Checkable::TypeInstance->GetFieldCount();
	if (real_id < 0) { return Checkable::TypeInstance->GetFieldInfo(id); }
	switch (real_id) {
		case 0:
			return {0, "String", "display_name", "display_name", nullptr, 2, 0};
		case 1:
			return {1, "String", "host_name", "host_name", "Host", 1282, 0};
		case 2:
			return {2, "Array", "groups", "groups", "ServiceGroup", 66818, 1};
		case 3:
			return {3, "Host", "host", "host", nullptr, 577, 0};
		case 4:
			return {4, "Number", "state", "state", nullptr, 73, 0};
		case 5:
			return {5, "Number", "last_state", "last_state", nullptr, 73, 0};
		case 6:
			return {6, "Number", "last_hard_state", "last_hard_state", nullptr, 73, 0};
		case 7:
			return {7, "Timestamp", "last_state_ok", "last_state_ok", nullptr, 4, 0};
		case 8:
			return {8, "Timestamp", "last_state_warning", "last_state_warning", nullptr, 4, 0};
		case 9:
			return {9, "Timestamp", "last_state_critical", "last_state_critical", nullptr, 4, 0};
		case 10:
			return {10, "Timestamp", "last_state_unknown", "last_state_unknown", nullptr, 4, 0};
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

int TypeImpl<Service>::GetFieldCount() const
{
	return 11 + Checkable::TypeInstance->GetFieldCount();
}

ObjectFactory TypeImpl<Service>::GetFactory() const
{
	return TypeHelper<Service, false>::GetFactory();
}

const std::unordered_set<Type*>& TypeImpl<Service>::GetLoadDependencies() const
{
	static const auto deps ([] {
		auto typeApiListener (GetByName("ApiListener").get());
		VERIFY(typeApiListener);
		VERIFY(ConfigObject::TypeInstance->IsAssignableFrom(typeApiListener));

		auto typeEndpoint (GetByName("Endpoint").get());
		VERIFY(typeEndpoint);
		VERIFY(ConfigObject::TypeInstance->IsAssignableFrom(typeEndpoint));

		auto typeHost (GetByName("Host").get());
		VERIFY(typeHost);
		VERIFY(ConfigObject::TypeInstance->IsAssignableFrom(typeHost));

		auto typeZone (GetByName("Zone").get());
		VERIFY(typeZone);
		VERIFY(ConfigObject::TypeInstance->IsAssignableFrom(typeZone));

		return std::unordered_set<Type*>{ typeApiListener, typeEndpoint, typeHost, typeZone, };
	}());

	return deps;
}

int TypeImpl<Service>::GetActivationPriority() const
{
	return 0;
}

void TypeImpl<Service>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	int real_id = fieldId - Checkable::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { Checkable::TypeInstance->RegisterAttributeHandler(fieldId, callback); return; }
	switch (real_id) {
		case 0:
			ObjectImpl<Service>::OnDisplayNameChanged.connect(callback);
			break;
		case 1:
			ObjectImpl<Service>::OnHostNameChanged.connect(callback);
			break;
		case 2:
			ObjectImpl<Service>::OnGroupsChanged.connect(callback);
			break;
		case 3:
			ObjectImpl<Service>::OnHostChanged.connect(callback);
			break;
		case 4:
			ObjectImpl<Service>::OnStateChanged.connect(callback);
			break;
		case 5:
			ObjectImpl<Service>::OnLastStateChanged.connect(callback);
			break;
		case 6:
			ObjectImpl<Service>::OnLastHardStateChanged.connect(callback);
			break;
		case 7:
			ObjectImpl<Service>::OnLastStateOKChanged.connect(callback);
			break;
		case 8:
			ObjectImpl<Service>::OnLastStateWarningChanged.connect(callback);
			break;
		case 9:
			ObjectImpl<Service>::OnLastStateCriticalChanged.connect(callback);
			break;
		case 10:
			ObjectImpl<Service>::OnLastStateUnknownChanged.connect(callback);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Service>::Validate(int types, const ValidationUtils& utils)
{
	Checkable::Validate(types, utils);

	if (2 & types)
		ValidateDisplayName(Lazy<String>([this]() { return GetDisplayName(); }), utils);
	if (2 & types)
		ValidateHostName(Lazy<String>([this]() { return GetHostName(); }), utils);
	if (2 & types)
		ValidateGroups(Lazy<Array::Ptr>([this]() { return GetGroups(); }), utils);
	if (1 & types)
		ValidateHost(Lazy<Host::Ptr>([this]() { return GetHost(); }), utils);
	if (1 & types)
		ValidateState(Lazy<ServiceState>([this]() { return GetState(); }), utils);
	if (1 & types)
		ValidateLastState(Lazy<ServiceState>([this]() { return GetLastState(); }), utils);
	if (1 & types)
		ValidateLastHardState(Lazy<ServiceState>([this]() { return GetLastHardState(); }), utils);
	if (4 & types)
		ValidateLastStateOK(Lazy<Timestamp>([this]() { return GetLastStateOK(); }), utils);
	if (4 & types)
		ValidateLastStateWarning(Lazy<Timestamp>([this]() { return GetLastStateWarning(); }), utils);
	if (4 & types)
		ValidateLastStateCritical(Lazy<Timestamp>([this]() { return GetLastStateCritical(); }), utils);
	if (4 & types)
		ValidateLastStateUnknown(Lazy<Timestamp>([this]() { return GetLastStateUnknown(); }), utils);
}

void ObjectImpl<Service>::SimpleValidateDisplayName(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateHostName(const Lazy<String>& value, const ValidationUtils& utils)
{
	if (value().IsEmpty())
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "host_name" }, "Attribute must not be empty."));

			if (!value().IsEmpty() && !utils.ValidateName("Host", value()))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "host_name" }, "Object '" + value() + "' of type 'Host' does not exist."));
}

void ObjectImpl<Service>::SimpleValidateGroups(const Lazy<Array::Ptr>& avalue, const ValidationUtils& utils)
{
	if (!avalue())
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "groups" }, "Attribute must not be empty."));

	if (avalue()) {
		ObjectLock olock(avalue());
		for (const Value& value : avalue()) {
			if (!value.IsEmpty() && !value.IsString())
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "groups" }, "It is not allowed to specify '" + value + "' of type '" + value.GetTypeName() + "' as 'ServiceGroup' name. Expected type of 'ServiceGroup' name: 'String'."));
			if (value.IsEmpty() || !utils.ValidateName("ServiceGroup", value))
				BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "groups" }, "Object '" + value + "' of type 'ServiceGroup' does not exist."));
		}
	}
}

void ObjectImpl<Service>::SimpleValidateHost(const Lazy<Host::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateState(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateLastState(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateLastHardState(const Lazy<ServiceState>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateLastStateOK(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateLastStateWarning(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateLastStateCritical(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Service>::SimpleValidateLastStateUnknown(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

ObjectImpl<Service>::ObjectImpl()
{
	SetDisplayName(GetDefaultDisplayName(), true);
	SetHostName(GetDefaultHostName(), true);
	SetGroups(GetDefaultGroups(), true);
	SetHost(GetDefaultHost(), true);
	SetState(GetDefaultState(), true);
	SetLastState(GetDefaultLastState(), true);
	SetLastHardState(GetDefaultLastHardState(), true);
	SetLastStateOK(GetDefaultLastStateOK(), true);
	SetLastStateWarning(GetDefaultLastStateWarning(), true);
	SetLastStateCritical(GetDefaultLastStateCritical(), true);
	SetLastStateUnknown(GetDefaultLastStateUnknown(), true);
}

ObjectImpl<Service>::~ObjectImpl()
{ }

void ObjectImpl<Service>::SetField(int id, const Value& value, bool suppress_events, const Value& cookie)
{
	int real_id = id - Checkable::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { Checkable::SetField(id, value, suppress_events, cookie); return; }
	switch (real_id) {
		case 0:
			SetDisplayName(value, suppress_events, cookie);
			break;
		case 1:
			SetHostName(value, suppress_events, cookie);
			break;
		case 2:
			SetGroups(value, suppress_events, cookie);
			break;
		case 3:
			SetHost(value, suppress_events, cookie);
			break;
		case 4:
			SetState(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 5:
			SetLastState(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 6:
			SetLastHardState(static_cast<ServiceState>(static_cast<int>(value)), suppress_events, cookie);
			break;
		case 7:
			SetLastStateOK(value, suppress_events, cookie);
			break;
		case 8:
			SetLastStateWarning(value, suppress_events, cookie);
			break;
		case 9:
			SetLastStateCritical(value, suppress_events, cookie);
			break;
		case 10:
			SetLastStateUnknown(value, suppress_events, cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<Service>::GetField(int id) const
{
	int real_id = id - Checkable::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return Checkable::GetField(id); }
	switch (real_id) {
		case 0:
			return GetDisplayName();
		case 1:
			return GetHostName();
		case 2:
			return GetGroups();
		case 3:
			return GetHost();
		case 4:
			return GetState();
		case 5:
			return GetLastState();
		case 6:
			return GetLastHardState();
		case 7:
			return GetLastStateOK();
		case 8:
			return GetLastStateWarning();
		case 9:
			return GetLastStateCritical();
		case 10:
			return GetLastStateUnknown();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Service>::ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	int real_id = id - Checkable::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { Checkable::ValidateField(id, lvalue, utils); return; }
	switch (real_id) {
		case 0:
			ValidateDisplayName(lvalue, utils);
			break;
		case 1:
			ValidateHostName(lvalue, utils);
			break;
		case 2:
			ValidateGroups(lvalue, utils);
			break;
		case 3:
			ValidateHost(lvalue, utils);
			break;
		case 4:
			ValidateState(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 5:
			ValidateLastState(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 6:
			ValidateLastHardState(static_cast<Lazy<ServiceState> >(static_cast<Lazy<int> >(lvalue)), utils);
			break;
		case 7:
			ValidateLastStateOK(lvalue, utils);
			break;
		case 8:
			ValidateLastStateWarning(lvalue, utils);
			break;
		case 9:
			ValidateLastStateCritical(lvalue, utils);
			break;
		case 10:
			ValidateLastStateUnknown(lvalue, utils);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Service>::NotifyField(int id, const Value& cookie)
{
	int real_id = id - Checkable::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { Checkable::NotifyField(id, cookie); return; }
	switch (real_id) {
		case 0:
			NotifyDisplayName(cookie);
			break;
		case 1:
			NotifyHostName(cookie);
			break;
		case 2:
			NotifyGroups(cookie);
			break;
		case 3:
			NotifyHost(cookie);
			break;
		case 4:
			NotifyState(cookie);
			break;
		case 5:
			NotifyLastState(cookie);
			break;
		case 6:
			NotifyLastHardState(cookie);
			break;
		case 7:
			NotifyLastStateOK(cookie);
			break;
		case 8:
			NotifyLastStateWarning(cookie);
			break;
		case 9:
			NotifyLastStateCritical(cookie);
			break;
		case 10:
			NotifyLastStateUnknown(cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Object::Ptr ObjectImpl<Service>::NavigateField(int id) const
{
	int real_id = id - Checkable::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return Checkable::NavigateField(id); }
	switch (real_id) {
		case 3:
			return NavigateHost();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

String ObjectImpl<Service>::GetDisplayName() const
{

			String displayName = m_DisplayName.load();
			if (displayName.IsEmpty())
				return GetShortName();
			else
				return displayName;
		
}

String ObjectImpl<Service>::GetHostName() const
{
	return m_HostName.load();
}

Array::Ptr ObjectImpl<Service>::GetGroups() const
{
	return m_Groups.load();
}

ServiceState ObjectImpl<Service>::GetState() const
{

			return GetStateRaw();
		
}

ServiceState ObjectImpl<Service>::GetLastState() const
{

			return GetLastStateRaw();
		
}

ServiceState ObjectImpl<Service>::GetLastHardState() const
{

			return GetLastHardStateRaw();
		
}

Timestamp ObjectImpl<Service>::GetLastStateOK() const
{
	return m_LastStateOK.load();
}

Timestamp ObjectImpl<Service>::GetLastStateWarning() const
{
	return m_LastStateWarning.load();
}

Timestamp ObjectImpl<Service>::GetLastStateCritical() const
{
	return m_LastStateCritical.load();
}

Timestamp ObjectImpl<Service>::GetLastStateUnknown() const
{
	return m_LastStateUnknown.load();
}

void ObjectImpl<Service>::SetDisplayName(const String& value, bool suppress_events, const Value& cookie)
{
	m_DisplayName.store(value);
	if (!suppress_events) {
		NotifyDisplayName(cookie);
	}

}

void ObjectImpl<Service>::SetHostName(const String& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetHostName();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_HostName.store(value);
	if (!dobj || dobj->IsActive())
		TrackHostName(oldValue, value);
	if (!suppress_events) {
		NotifyHostName(cookie);
	}

}

void ObjectImpl<Service>::SetGroups(const Array::Ptr& value, bool suppress_events, const Value& cookie)
{
	Value oldValue = GetGroups();
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	m_Groups.store(value);
	if (!dobj || dobj->IsActive())
		TrackGroups(oldValue, value);
	if (!suppress_events) {
		NotifyGroups(cookie);
		if (!dobj || dobj->IsActive())
			OnGroupsChangedWithOldValue(static_cast<Service *>(this), oldValue, value);
	}

}

void ObjectImpl<Service>::SetHost(const Host::Ptr& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyHost(cookie);
	}

}

void ObjectImpl<Service>::SetState(const ServiceState& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyState(cookie);
	}

}

void ObjectImpl<Service>::SetLastState(const ServiceState& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyLastState(cookie);
	}

}

void ObjectImpl<Service>::SetLastHardState(const ServiceState& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyLastHardState(cookie);
	}

}

void ObjectImpl<Service>::SetLastStateOK(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastStateOK.store(value);
	if (!suppress_events) {
		NotifyLastStateOK(cookie);
	}

}

void ObjectImpl<Service>::SetLastStateWarning(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastStateWarning.store(value);
	if (!suppress_events) {
		NotifyLastStateWarning(cookie);
	}

}

void ObjectImpl<Service>::SetLastStateCritical(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastStateCritical.store(value);
	if (!suppress_events) {
		NotifyLastStateCritical(cookie);
	}

}

void ObjectImpl<Service>::SetLastStateUnknown(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastStateUnknown.store(value);
	if (!suppress_events) {
		NotifyLastStateUnknown(cookie);
	}

}

void ObjectImpl<Service>::TrackHostName(const String& oldValue, const String& newValue)
{
	if (!oldValue.IsEmpty())
		DependencyGraph::RemoveDependency(this, ConfigObject::GetObject<Host>(oldValue).get());
	if (!newValue.IsEmpty())
		DependencyGraph::AddDependency(this, ConfigObject::GetObject<Host>(newValue).get());
}

void ObjectImpl<Service>::TrackGroups(const Array::Ptr& oldValue, const Array::Ptr& newValue)
{
	if (oldValue) {
		ObjectLock olock(oldValue);
		for (auto& ref : oldValue) {
			DependencyGraph::RemoveDependency(this, ConfigObject::GetObject<ServiceGroup>(ref).get());
		}
	}
	if (newValue) {
		ObjectLock olock(newValue);
		for (auto& ref : newValue) {
			DependencyGraph::AddDependency(this, ConfigObject::GetObject<ServiceGroup>(ref).get());
		}
	}
}

Object::Ptr ObjectImpl<Service>::NavigateHost() const
{
	
			return GetHost();
		
}

void ObjectImpl<Service>::Start(bool runtimeCreated)
{
	Checkable::Start(runtimeCreated);

	TrackHostName(Empty, GetHostName());
	TrackGroups(Empty, GetGroups());
}

void ObjectImpl<Service>::Stop(bool runtimeRemoved)
{
	Checkable::Stop(runtimeRemoved);

	TrackHostName(GetHostName(), Empty);
	TrackGroups(GetGroups(), Empty);
}

void ObjectImpl<Service>::NotifyDisplayName(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnDisplayNameChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyHostName(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnHostNameChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyGroups(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnGroupsChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyHost(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnHostChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyState(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnStateChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyLastState(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyLastHardState(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastHardStateChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyLastStateOK(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateOKChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyLastStateWarning(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateWarningChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyLastStateCritical(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateCriticalChanged(static_cast<Service *>(this), cookie);
}

void ObjectImpl<Service>::NotifyLastStateUnknown(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastStateUnknownChanged(static_cast<Service *>(this), cookie);
}

String ObjectImpl<Service>::GetDefaultDisplayName() const
{
	return String();
}

String ObjectImpl<Service>::GetDefaultHostName() const
{
	return String();
}

Array::Ptr ObjectImpl<Service>::GetDefaultGroups() const
{
	 return new Array(); 
}

Host::Ptr ObjectImpl<Service>::GetDefaultHost() const
{
	return Host::Ptr();
}

ServiceState ObjectImpl<Service>::GetDefaultState() const
{
	return ServiceState();
}

ServiceState ObjectImpl<Service>::GetDefaultLastState() const
{
	return ServiceState();
}

ServiceState ObjectImpl<Service>::GetDefaultLastHardState() const
{
	return ServiceState();
}

Timestamp ObjectImpl<Service>::GetDefaultLastStateOK() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Service>::GetDefaultLastStateWarning() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Service>::GetDefaultLastStateCritical() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Service>::GetDefaultLastStateUnknown() const
{
	return Timestamp();
}


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnDisplayNameChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnHostNameChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnGroupsChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&, const Value&)> ObjectImpl<Service>::OnGroupsChangedWithOldValue;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnHostChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnStateChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnLastStateChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnLastHardStateChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnLastStateOKChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnLastStateWarningChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnLastStateCriticalChanged;


boost::signals2::signal<void (const intrusive_ptr<Service>&, const Value&)> ObjectImpl<Service>::OnLastStateUnknownChanged;

void ObjectImpl<Service>::ValidateDisplayName(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateDisplayName(lvalue, utils);
}

void ObjectImpl<Service>::ValidateGroups(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateGroups(lvalue, utils);
}

void ObjectImpl<Service>::ValidateHost(const Lazy<Host::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateHost(lvalue, utils);
}

void ObjectImpl<Service>::ValidateHostName(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateHostName(lvalue, utils);
}

void ObjectImpl<Service>::ValidateLastHardState(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastHardState(lvalue, utils);
}

void ObjectImpl<Service>::ValidateLastState(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastState(lvalue, utils);
}

void ObjectImpl<Service>::ValidateLastStateCritical(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateCritical(lvalue, utils);
}

void ObjectImpl<Service>::ValidateLastStateOK(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateOK(lvalue, utils);
}

void ObjectImpl<Service>::ValidateLastStateUnknown(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateUnknown(lvalue, utils);
}

void ObjectImpl<Service>::ValidateLastStateWarning(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastStateWarning(lvalue, utils);
}

void ObjectImpl<Service>::ValidateState(const Lazy<ServiceState>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateState(lvalue, utils);
}

}
#ifdef _MSC_VER
#pragma warning ( pop )
#endif /* _MSC_VER */
