#ifndef SERVICEGROUP_TI
#define SERVICEGROUP_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "icinga/customvarobject.hpp"

namespace icinga
{

class ServiceGroup;

template<>
class TypeImpl<ServiceGroup> : public TypeImpl<CustomVarObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<ServiceGroup>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<ServiceGroup> : public CustomVarObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<ServiceGroup>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateDisplayName(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateNotes(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateNotesUrl(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateActionUrl(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateGroups(const Lazy<Array::Ptr>& avalue, const ValidationUtils& utils);
public:
	ObjectImpl<ServiceGroup>();
public:
	~ObjectImpl<ServiceGroup>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetDisplayName() const;
public:
	String GetNotes() const;
public:
	String GetNotesUrl() const;
public:
	String GetActionUrl() const;
public:
	Array::Ptr GetGroups() const;
public:
	void SetDisplayName(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetNotes(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetNotesUrl(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetActionUrl(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetGroups(const Array::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void TrackGroups(const Array::Ptr& oldValue, const Array::Ptr& newValue);
protected:
	void Start(bool runtimeCreated = false) override;
	void Stop(bool runtimeRemoved = false) override;
public:
	virtual void NotifyDisplayName(const Value& cookie = Empty);
public:
	virtual void NotifyNotes(const Value& cookie = Empty);
public:
	virtual void NotifyNotesUrl(const Value& cookie = Empty);
public:
	virtual void NotifyActionUrl(const Value& cookie = Empty);
public:
	virtual void NotifyGroups(const Value& cookie = Empty);
private:
	inline String GetDefaultDisplayName() const;
private:
	inline String GetDefaultNotes() const;
private:
	inline String GetDefaultNotesUrl() const;
private:
	inline String GetDefaultActionUrl() const;
private:
	inline Array::Ptr GetDefaultGroups() const;
protected:
	virtual void ValidateDisplayName(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateNotes(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateNotesUrl(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateActionUrl(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateGroups(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_DisplayName;
	AtomicOrLocked<String> m_Notes;
	AtomicOrLocked<String> m_NotesUrl;
	AtomicOrLocked<String> m_ActionUrl;
	AtomicOrLocked<Array::Ptr> m_Groups;
public:
	static boost::signals2::signal<void (const intrusive_ptr<ServiceGroup>&, const Value&)> OnDisplayNameChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ServiceGroup>&, const Value&)> OnNotesChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ServiceGroup>&, const Value&)> OnNotesUrlChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ServiceGroup>&, const Value&)> OnActionUrlChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ServiceGroup>&, const Value&)> OnGroupsChanged;
};

}
#endif /* SERVICEGROUP_TI */
