#ifndef ICINGADB_TI
#define ICINGADB_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

#include "base/tlsutility.hpp"

namespace icinga
{

class IcingaDB;

template<>
class TypeImpl<IcingaDB> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<IcingaDB>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<IcingaDB> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<IcingaDB>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateUsername(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePassword(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCertPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateKeyPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCaPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCrlPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCipherList(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateTlsProtocolmin(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateEnvironmentId(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateLastdumpTook(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateLastdumpEnd(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateOngoingDumpStart(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateConnectTimeout(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateDbIndex(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidatePort(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateEnableTls(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateInsecureNoverify(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<IcingaDB>();
public:
	~ObjectImpl<IcingaDB>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHost() const;
public:
	String GetPath() const;
public:
	String GetUsername() const;
public:
	String GetPassword() const;
public:
	String GetCertPath() const;
public:
	String GetKeyPath() const;
public:
	String GetCaPath() const;
public:
	String GetCrlPath() const;
public:
	String GetCipherList() const;
public:
	String GetTlsProtocolmin() const;
public:
	virtual String GetEnvironmentId() const = 0;
public:
	double GetLastdumpTook() const;
public:
	double GetLastdumpEnd() const;
public:
	double GetOngoingDumpStart() const;
public:
	double GetConnectTimeout() const;
public:
	int GetDbIndex() const;
public:
	int GetPort() const;
public:
	bool GetEnableTls() const;
public:
	bool GetInsecureNoverify() const;
public:
	void SetHost(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetUsername(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPassword(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCertPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetKeyPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCaPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCrlPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCipherList(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetTlsProtocolmin(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnvironmentId(const String& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void SetLastdumpTook(double value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void SetLastdumpEnd(double value, bool suppress_events = false, const Value& cookie = Empty);
protected:
	void SetOngoingDumpStart(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetConnectTimeout(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetDbIndex(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPort(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableTls(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetInsecureNoverify(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHost(const Value& cookie = Empty);
public:
	virtual void NotifyPath(const Value& cookie = Empty);
public:
	virtual void NotifyUsername(const Value& cookie = Empty);
public:
	virtual void NotifyPassword(const Value& cookie = Empty);
public:
	virtual void NotifyCertPath(const Value& cookie = Empty);
public:
	virtual void NotifyKeyPath(const Value& cookie = Empty);
public:
	virtual void NotifyCaPath(const Value& cookie = Empty);
public:
	virtual void NotifyCrlPath(const Value& cookie = Empty);
public:
	virtual void NotifyCipherList(const Value& cookie = Empty);
public:
	virtual void NotifyTlsProtocolmin(const Value& cookie = Empty);
public:
	virtual void NotifyEnvironmentId(const Value& cookie = Empty);
protected:
	virtual void NotifyLastdumpTook(const Value& cookie = Empty);
protected:
	virtual void NotifyLastdumpEnd(const Value& cookie = Empty);
protected:
	virtual void NotifyOngoingDumpStart(const Value& cookie = Empty);
public:
	virtual void NotifyConnectTimeout(const Value& cookie = Empty);
public:
	virtual void NotifyDbIndex(const Value& cookie = Empty);
public:
	virtual void NotifyPort(const Value& cookie = Empty);
public:
	virtual void NotifyEnableTls(const Value& cookie = Empty);
public:
	virtual void NotifyInsecureNoverify(const Value& cookie = Empty);
private:
	inline String GetDefaultHost() const;
private:
	inline String GetDefaultPath() const;
private:
	inline String GetDefaultUsername() const;
private:
	inline String GetDefaultPassword() const;
private:
	inline String GetDefaultCertPath() const;
private:
	inline String GetDefaultKeyPath() const;
private:
	inline String GetDefaultCaPath() const;
private:
	inline String GetDefaultCrlPath() const;
private:
	inline String GetDefaultCipherList() const;
private:
	inline String GetDefaultTlsProtocolmin() const;
private:
	inline String GetDefaultEnvironmentId() const;
private:
	inline double GetDefaultLastdumpTook() const;
private:
	inline double GetDefaultLastdumpEnd() const;
private:
	inline double GetDefaultOngoingDumpStart() const;
private:
	inline double GetDefaultConnectTimeout() const;
private:
	inline int GetDefaultDbIndex() const;
private:
	inline int GetDefaultPort() const;
private:
	inline bool GetDefaultEnableTls() const;
private:
	inline bool GetDefaultInsecureNoverify() const;
protected:
	virtual void ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateUsername(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePassword(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCertPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateKeyPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCaPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCrlPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCipherList(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateTlsProtocolmin(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnvironmentId(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastdumpTook(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateLastdumpEnd(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateOngoingDumpStart(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateConnectTimeout(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateDbIndex(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePort(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableTls(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateInsecureNoverify(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Host;
	AtomicOrLocked<String> m_Path;
	AtomicOrLocked<String> m_Username;
	AtomicOrLocked<String> m_Password;
	AtomicOrLocked<String> m_CertPath;
	AtomicOrLocked<String> m_KeyPath;
	AtomicOrLocked<String> m_CaPath;
	AtomicOrLocked<String> m_CrlPath;
	AtomicOrLocked<String> m_CipherList;
	AtomicOrLocked<String> m_TlsProtocolmin;
	AtomicOrLocked<double> m_LastdumpTook;
	AtomicOrLocked<double> m_LastdumpEnd;
	AtomicOrLocked<double> m_OngoingDumpStart;
	AtomicOrLocked<double> m_ConnectTimeout;
	AtomicOrLocked<int> m_DbIndex;
	AtomicOrLocked<int> m_Port;
	AtomicOrLocked<bool> m_EnableTls;
	AtomicOrLocked<bool> m_InsecureNoverify;
public:
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnHostChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnUsernameChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnPasswordChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnCertPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnKeyPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnCaPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnCrlPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnCipherListChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnTlsProtocolminChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnEnvironmentIdChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnLastdumpTookChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnLastdumpEndChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnOngoingDumpStartChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnConnectTimeoutChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnDbIndexChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnPortChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnEnableTlsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<IcingaDB>&, const Value&)> OnInsecureNoverifyChanged;
};

}
#endif /* ICINGADB_TI */
