#ifndef GELFWRITER_TI
#define GELFWRITER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class GelfWriter;

template<>
class TypeImpl<GelfWriter> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<GelfWriter>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<GelfWriter> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<GelfWriter>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePort(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSource(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCaPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateCertPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateKeyPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateEnableSendPerfdata(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateConnected(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateShouldConnect(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableHa(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableTls(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateInsecureNoverify(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<GelfWriter>();
public:
	~ObjectImpl<GelfWriter>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHost() const;
public:
	String GetPort() const;
public:
	String GetSource() const;
public:
	String GetCaPath() const;
public:
	String GetCertPath() const;
public:
	String GetKeyPath() const;
public:
	bool GetEnableSendPerfdata() const;
public:
	bool GetConnected() const;
public:
	bool GetShouldConnect() const;
public:
	bool GetEnableHa() const;
public:
	bool GetEnableTls() const;
public:
	bool GetInsecureNoverify() const;
public:
	void SetHost(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPort(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSource(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCaPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetCertPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetKeyPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableSendPerfdata(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetConnected(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetShouldConnect(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableHa(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableTls(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetInsecureNoverify(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHost(const Value& cookie = Empty);
public:
	virtual void NotifyPort(const Value& cookie = Empty);
public:
	virtual void NotifySource(const Value& cookie = Empty);
public:
	virtual void NotifyCaPath(const Value& cookie = Empty);
public:
	virtual void NotifyCertPath(const Value& cookie = Empty);
public:
	virtual void NotifyKeyPath(const Value& cookie = Empty);
public:
	virtual void NotifyEnableSendPerfdata(const Value& cookie = Empty);
public:
	virtual void NotifyConnected(const Value& cookie = Empty);
public:
	virtual void NotifyShouldConnect(const Value& cookie = Empty);
public:
	virtual void NotifyEnableHa(const Value& cookie = Empty);
public:
	virtual void NotifyEnableTls(const Value& cookie = Empty);
public:
	virtual void NotifyInsecureNoverify(const Value& cookie = Empty);
private:
	inline String GetDefaultHost() const;
private:
	inline String GetDefaultPort() const;
private:
	inline String GetDefaultSource() const;
private:
	inline String GetDefaultCaPath() const;
private:
	inline String GetDefaultCertPath() const;
private:
	inline String GetDefaultKeyPath() const;
private:
	inline bool GetDefaultEnableSendPerfdata() const;
private:
	inline bool GetDefaultConnected() const;
private:
	inline bool GetDefaultShouldConnect() const;
private:
	inline bool GetDefaultEnableHa() const;
private:
	inline bool GetDefaultEnableTls() const;
private:
	inline bool GetDefaultInsecureNoverify() const;
protected:
	virtual void ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePort(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSource(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCaPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateCertPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateKeyPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableSendPerfdata(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateConnected(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateShouldConnect(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableHa(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableTls(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateInsecureNoverify(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Host;
	AtomicOrLocked<String> m_Port;
	AtomicOrLocked<String> m_Source;
	AtomicOrLocked<String> m_CaPath;
	AtomicOrLocked<String> m_CertPath;
	AtomicOrLocked<String> m_KeyPath;
	AtomicOrLocked<bool> m_EnableSendPerfdata;
	AtomicOrLocked<bool> m_Connected;
	AtomicOrLocked<bool> m_ShouldConnect;
	AtomicOrLocked<bool> m_EnableHa;
	AtomicOrLocked<bool> m_EnableTls;
	AtomicOrLocked<bool> m_InsecureNoverify;
public:
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnHostChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnPortChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnSourceChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnCaPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnCertPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnKeyPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnEnableSendPerfdataChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnConnectedChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnShouldConnectChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnEnableHaChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnEnableTlsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GelfWriter>&, const Value&)> OnInsecureNoverifyChanged;
};

}
#endif /* GELFWRITER_TI */
