#ifndef GRAPHITEWRITER_TI
#define GRAPHITEWRITER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class GraphiteWriter;

template<>
class TypeImpl<GraphiteWriter> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<GraphiteWriter>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<GraphiteWriter> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<GraphiteWriter>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePort(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateHostNameTemplate(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateServiceNameTemplate(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateEnableSendThresholds(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableSendMetadata(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateConnected(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateShouldConnect(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableHa(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<GraphiteWriter>();
public:
	~ObjectImpl<GraphiteWriter>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHost() const;
public:
	String GetPort() const;
public:
	String GetHostNameTemplate() const;
public:
	String GetServiceNameTemplate() const;
public:
	bool GetEnableSendThresholds() const;
public:
	bool GetEnableSendMetadata() const;
public:
	bool GetConnected() const;
public:
	bool GetShouldConnect() const;
public:
	bool GetEnableHa() const;
public:
	void SetHost(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPort(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetHostNameTemplate(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetServiceNameTemplate(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableSendThresholds(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableSendMetadata(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetConnected(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetShouldConnect(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableHa(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHost(const Value& cookie = Empty);
public:
	virtual void NotifyPort(const Value& cookie = Empty);
public:
	virtual void NotifyHostNameTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyServiceNameTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyEnableSendThresholds(const Value& cookie = Empty);
public:
	virtual void NotifyEnableSendMetadata(const Value& cookie = Empty);
public:
	virtual void NotifyConnected(const Value& cookie = Empty);
public:
	virtual void NotifyShouldConnect(const Value& cookie = Empty);
public:
	virtual void NotifyEnableHa(const Value& cookie = Empty);
private:
	inline String GetDefaultHost() const;
private:
	inline String GetDefaultPort() const;
private:
	inline String GetDefaultHostNameTemplate() const;
private:
	inline String GetDefaultServiceNameTemplate() const;
private:
	inline bool GetDefaultEnableSendThresholds() const;
private:
	inline bool GetDefaultEnableSendMetadata() const;
private:
	inline bool GetDefaultConnected() const;
private:
	inline bool GetDefaultShouldConnect() const;
private:
	inline bool GetDefaultEnableHa() const;
protected:
	virtual void ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePort(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateHostNameTemplate(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateServiceNameTemplate(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableSendThresholds(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableSendMetadata(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateConnected(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateShouldConnect(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableHa(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Host;
	AtomicOrLocked<String> m_Port;
	AtomicOrLocked<String> m_HostNameTemplate;
	AtomicOrLocked<String> m_ServiceNameTemplate;
	AtomicOrLocked<bool> m_EnableSendThresholds;
	AtomicOrLocked<bool> m_EnableSendMetadata;
	AtomicOrLocked<bool> m_Connected;
	AtomicOrLocked<bool> m_ShouldConnect;
	AtomicOrLocked<bool> m_EnableHa;
public:
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnHostChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnPortChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnHostNameTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnServiceNameTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnEnableSendThresholdsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnEnableSendMetadataChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnConnectedChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnShouldConnectChanged;
	static boost::signals2::signal<void (const intrusive_ptr<GraphiteWriter>&, const Value&)> OnEnableHaChanged;
};

}
#endif /* GRAPHITEWRITER_TI */
