#ifndef INFLUXDBCOMMONWRITER_TI
#define INFLUXDBCOMMONWRITER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class InfluxdbCommonWriter;

template<>
struct TypeHelper<InfluxdbCommonWriter, false>
{
	static ObjectFactory GetFactory()
	{
		return nullptr;
	}
};

template<>
class TypeImpl<InfluxdbCommonWriter> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<InfluxdbCommonWriter>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<InfluxdbCommonWriter> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<InfluxdbCommonWriter>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePort(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslCaCert(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslCert(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateSslKey(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateHostTemplate(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateServiceTemplate(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateFlushInterval(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateFlushThreshold(const Lazy<int>& value, const ValidationUtils& utils);
	void SimpleValidateSslEnable(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateSslInsecureNoverify(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableSendThresholds(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableSendMetadata(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableHa(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<InfluxdbCommonWriter>();
public:
	~ObjectImpl<InfluxdbCommonWriter>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHost() const;
public:
	String GetPort() const;
public:
	String GetSslCaCert() const;
public:
	String GetSslCert() const;
public:
	String GetSslKey() const;
public:
	Dictionary::Ptr GetHostTemplate() const;
public:
	Dictionary::Ptr GetServiceTemplate() const;
public:
	int GetFlushInterval() const;
public:
	int GetFlushThreshold() const;
public:
	bool GetSslEnable() const;
public:
	bool GetSslInsecureNoverify() const;
public:
	bool GetEnableSendThresholds() const;
public:
	bool GetEnableSendMetadata() const;
public:
	bool GetEnableHa() const;
public:
	void SetHost(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPort(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslCaCert(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslCert(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslKey(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetHostTemplate(const Dictionary::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetServiceTemplate(const Dictionary::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlushInterval(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetFlushThreshold(int value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslEnable(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetSslInsecureNoverify(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableSendThresholds(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableSendMetadata(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableHa(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHost(const Value& cookie = Empty);
public:
	virtual void NotifyPort(const Value& cookie = Empty);
public:
	virtual void NotifySslCaCert(const Value& cookie = Empty);
public:
	virtual void NotifySslCert(const Value& cookie = Empty);
public:
	virtual void NotifySslKey(const Value& cookie = Empty);
public:
	virtual void NotifyHostTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyServiceTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyFlushInterval(const Value& cookie = Empty);
public:
	virtual void NotifyFlushThreshold(const Value& cookie = Empty);
public:
	virtual void NotifySslEnable(const Value& cookie = Empty);
public:
	virtual void NotifySslInsecureNoverify(const Value& cookie = Empty);
public:
	virtual void NotifyEnableSendThresholds(const Value& cookie = Empty);
public:
	virtual void NotifyEnableSendMetadata(const Value& cookie = Empty);
public:
	virtual void NotifyEnableHa(const Value& cookie = Empty);
private:
	inline String GetDefaultHost() const;
private:
	inline String GetDefaultPort() const;
private:
	inline String GetDefaultSslCaCert() const;
private:
	inline String GetDefaultSslCert() const;
private:
	inline String GetDefaultSslKey() const;
private:
	inline Dictionary::Ptr GetDefaultHostTemplate() const;
private:
	inline Dictionary::Ptr GetDefaultServiceTemplate() const;
private:
	inline int GetDefaultFlushInterval() const;
private:
	inline int GetDefaultFlushThreshold() const;
private:
	inline bool GetDefaultSslEnable() const;
private:
	inline bool GetDefaultSslInsecureNoverify() const;
private:
	inline bool GetDefaultEnableSendThresholds() const;
private:
	inline bool GetDefaultEnableSendMetadata() const;
private:
	inline bool GetDefaultEnableHa() const;
protected:
	virtual void ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePort(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslCaCert(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslCert(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslKey(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateHostTemplate(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateServiceTemplate(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlushInterval(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateFlushThreshold(const Lazy<int>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslEnable(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateSslInsecureNoverify(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableSendThresholds(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableSendMetadata(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableHa(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Host;
	AtomicOrLocked<String> m_Port;
	AtomicOrLocked<String> m_SslCaCert;
	AtomicOrLocked<String> m_SslCert;
	AtomicOrLocked<String> m_SslKey;
	AtomicOrLocked<Dictionary::Ptr> m_HostTemplate;
	AtomicOrLocked<Dictionary::Ptr> m_ServiceTemplate;
	AtomicOrLocked<int> m_FlushInterval;
	AtomicOrLocked<int> m_FlushThreshold;
	AtomicOrLocked<bool> m_SslEnable;
	AtomicOrLocked<bool> m_SslInsecureNoverify;
	AtomicOrLocked<bool> m_EnableSendThresholds;
	AtomicOrLocked<bool> m_EnableSendMetadata;
	AtomicOrLocked<bool> m_EnableHa;
public:
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnHostChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnPortChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnSslCaCertChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnSslCertChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnSslKeyChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnHostTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnServiceTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnFlushIntervalChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnFlushThresholdChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnSslEnableChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnSslInsecureNoverifyChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnEnableSendThresholdsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnEnableSendMetadataChanged;
	static boost::signals2::signal<void (const intrusive_ptr<InfluxdbCommonWriter>&, const Value&)> OnEnableHaChanged;
};

}
#endif /* INFLUXDBCOMMONWRITER_TI */
