#ifndef OPENTSDBWRITER_TI
#define OPENTSDBWRITER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

namespace icinga
{

class OpenTsdbWriter;

template<>
class TypeImpl<OpenTsdbWriter> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<OpenTsdbWriter>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<OpenTsdbWriter> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<OpenTsdbWriter>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePort(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateHostTemplate(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateServiceTemplate(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils);
	void SimpleValidateEnableHa(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateEnableGenericMetrics(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateConnected(const Lazy<bool>& value, const ValidationUtils& utils);
	void SimpleValidateShouldConnect(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<OpenTsdbWriter>();
public:
	~ObjectImpl<OpenTsdbWriter>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHost() const;
public:
	String GetPort() const;
public:
	Dictionary::Ptr GetHostTemplate() const;
public:
	Dictionary::Ptr GetServiceTemplate() const;
public:
	bool GetEnableHa() const;
public:
	bool GetEnableGenericMetrics() const;
public:
	bool GetConnected() const;
public:
	bool GetShouldConnect() const;
public:
	void SetHost(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPort(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetHostTemplate(const Dictionary::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetServiceTemplate(const Dictionary::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableHa(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableGenericMetrics(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetConnected(bool value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetShouldConnect(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHost(const Value& cookie = Empty);
public:
	virtual void NotifyPort(const Value& cookie = Empty);
public:
	virtual void NotifyHostTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyServiceTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyEnableHa(const Value& cookie = Empty);
public:
	virtual void NotifyEnableGenericMetrics(const Value& cookie = Empty);
public:
	virtual void NotifyConnected(const Value& cookie = Empty);
public:
	virtual void NotifyShouldConnect(const Value& cookie = Empty);
private:
	inline String GetDefaultHost() const;
private:
	inline String GetDefaultPort() const;
private:
	inline Dictionary::Ptr GetDefaultHostTemplate() const;
private:
	inline Dictionary::Ptr GetDefaultServiceTemplate() const;
private:
	inline bool GetDefaultEnableHa() const;
private:
	inline bool GetDefaultEnableGenericMetrics() const;
private:
	inline bool GetDefaultConnected() const;
private:
	inline bool GetDefaultShouldConnect() const;
protected:
	virtual void ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePort(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateHostTemplate(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateServiceTemplate(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableHa(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableGenericMetrics(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateConnected(const Lazy<bool>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateShouldConnect(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Host;
	AtomicOrLocked<String> m_Port;
	AtomicOrLocked<Dictionary::Ptr> m_HostTemplate;
	AtomicOrLocked<Dictionary::Ptr> m_ServiceTemplate;
	AtomicOrLocked<bool> m_EnableHa;
	AtomicOrLocked<bool> m_EnableGenericMetrics;
	AtomicOrLocked<bool> m_Connected;
	AtomicOrLocked<bool> m_ShouldConnect;
public:
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnHostChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnPortChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnHostTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnServiceTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnEnableHaChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnEnableGenericMetricsChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnConnectedChanged;
	static boost::signals2::signal<void (const intrusive_ptr<OpenTsdbWriter>&, const Value&)> OnShouldConnectChanged;
};

}
#endif /* OPENTSDBWRITER_TI */
