#ifndef PERFDATAWRITER_TI
#define PERFDATAWRITER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

#include "base/application.hpp"

namespace icinga
{

class PerfdataWriter;

template<>
class TypeImpl<PerfdataWriter> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<PerfdataWriter>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<PerfdataWriter> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<PerfdataWriter>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidateHostPerfdataPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateServicePerfdataPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateHostTempPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateServiceTempPath(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateHostFormatTemplate(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateServiceFormatTemplate(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateRotationInterval(const Lazy<double>& value, const ValidationUtils& utils);
	void SimpleValidateEnableHa(const Lazy<bool>& value, const ValidationUtils& utils);
public:
	ObjectImpl<PerfdataWriter>();
public:
	~ObjectImpl<PerfdataWriter>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetHostPerfdataPath() const;
public:
	String GetServicePerfdataPath() const;
public:
	String GetHostTempPath() const;
public:
	String GetServiceTempPath() const;
public:
	String GetHostFormatTemplate() const;
public:
	String GetServiceFormatTemplate() const;
public:
	double GetRotationInterval() const;
public:
	bool GetEnableHa() const;
public:
	void SetHostPerfdataPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetServicePerfdataPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetHostTempPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetServiceTempPath(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetHostFormatTemplate(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetServiceFormatTemplate(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetRotationInterval(double value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetEnableHa(bool value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyHostPerfdataPath(const Value& cookie = Empty);
public:
	virtual void NotifyServicePerfdataPath(const Value& cookie = Empty);
public:
	virtual void NotifyHostTempPath(const Value& cookie = Empty);
public:
	virtual void NotifyServiceTempPath(const Value& cookie = Empty);
public:
	virtual void NotifyHostFormatTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyServiceFormatTemplate(const Value& cookie = Empty);
public:
	virtual void NotifyRotationInterval(const Value& cookie = Empty);
public:
	virtual void NotifyEnableHa(const Value& cookie = Empty);
private:
	inline String GetDefaultHostPerfdataPath() const;
private:
	inline String GetDefaultServicePerfdataPath() const;
private:
	inline String GetDefaultHostTempPath() const;
private:
	inline String GetDefaultServiceTempPath() const;
private:
	inline String GetDefaultHostFormatTemplate() const;
private:
	inline String GetDefaultServiceFormatTemplate() const;
private:
	inline double GetDefaultRotationInterval() const;
private:
	inline bool GetDefaultEnableHa() const;
protected:
	virtual void ValidateHostPerfdataPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateServicePerfdataPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateHostTempPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateServiceTempPath(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateHostFormatTemplate(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateServiceFormatTemplate(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateRotationInterval(const Lazy<double>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateEnableHa(const Lazy<bool>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_HostPerfdataPath;
	AtomicOrLocked<String> m_ServicePerfdataPath;
	AtomicOrLocked<String> m_HostTempPath;
	AtomicOrLocked<String> m_ServiceTempPath;
	AtomicOrLocked<String> m_HostFormatTemplate;
	AtomicOrLocked<String> m_ServiceFormatTemplate;
	AtomicOrLocked<double> m_RotationInterval;
	AtomicOrLocked<bool> m_EnableHa;
public:
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnHostPerfdataPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnServicePerfdataPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnHostTempPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnServiceTempPathChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnHostFormatTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnServiceFormatTemplateChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnRotationIntervalChanged;
	static boost::signals2::signal<void (const intrusive_ptr<PerfdataWriter>&, const Value&)> OnEnableHaChanged;
};

}
#endif /* PERFDATAWRITER_TI */
