#ifndef APIUSER_TI
#define APIUSER_TI

#include "base/object.hpp"
#include "base/type.hpp"
#include "base/value.hpp"
#include "base/array.hpp"
#include "base/atomic.hpp"
#include "base/dictionary.hpp"
#include <boost/signals2.hpp>

#include "base/configobject.hpp"

#include "base/function.hpp"

namespace icinga
{

class ApiUser;

template<>
class TypeImpl<ApiUser> : public TypeImpl<ConfigObject>
{
public:
	DECLARE_PTR_TYPEDEFS(TypeImpl<ApiUser>);

	TypeImpl();
	~TypeImpl() override;

	String GetName() const override;
	int GetAttributes() const override;
	Type::Ptr GetBaseType() const override;
	int GetFieldId(const String& name) const override;
	Field GetFieldInfo(int id) const override;
	int GetFieldCount() const override;
	ObjectFactory GetFactory() const override;
	int GetActivationPriority() const override;
public:
	void RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback) override;
};


template<>
class ObjectImpl<ApiUser> : public ConfigObject
{
public:
	DECLARE_PTR_TYPEDEFS(ObjectImpl<ApiUser>);

	void Validate(int types, const ValidationUtils& utils) override;
	void SimpleValidatePassword(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePasswordHash(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidateClientCN(const Lazy<String>& value, const ValidationUtils& utils);
	void SimpleValidatePermissions(const Lazy<Array::Ptr>& avalue, const ValidationUtils& utils);
public:
	ObjectImpl<ApiUser>();
public:
	~ObjectImpl<ApiUser>() override;
public:
	void SetField(int id, const Value& value, bool suppress_events = false, const Value& cookie = Empty) override;
public:
	Value GetField(int id) const override;
public:
	void ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils) override;
public:
	void NotifyField(int id, const Value& cookie = Empty) override;
public:
	Object::Ptr NavigateField(int id) const override;
public:
	String GetPassword() const;
public:
	String GetPasswordHash() const;
public:
	String GetClientCN() const;
public:
	Array::Ptr GetPermissions() const;
public:
	void SetPassword(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPasswordHash(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetClientCN(const String& value, bool suppress_events = false, const Value& cookie = Empty);
public:
	void SetPermissions(const Array::Ptr& value, bool suppress_events = false, const Value& cookie = Empty);
protected:
public:
	virtual void NotifyPassword(const Value& cookie = Empty);
public:
	virtual void NotifyPasswordHash(const Value& cookie = Empty);
public:
	virtual void NotifyClientCN(const Value& cookie = Empty);
public:
	virtual void NotifyPermissions(const Value& cookie = Empty);
private:
	inline String GetDefaultPassword() const;
private:
	inline String GetDefaultPasswordHash() const;
private:
	inline String GetDefaultClientCN() const;
private:
	inline Array::Ptr GetDefaultPermissions() const;
protected:
	virtual void ValidatePassword(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePasswordHash(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidateClientCN(const Lazy<String>& lvalue, const ValidationUtils& utils);
protected:
	virtual void ValidatePermissions(const Lazy<Array::Ptr>& lvalue, const ValidationUtils& utils);
private:
	AtomicOrLocked<String> m_Password;
	AtomicOrLocked<String> m_PasswordHash;
	AtomicOrLocked<String> m_ClientCN;
	AtomicOrLocked<Array::Ptr> m_Permissions;
public:
	static boost::signals2::signal<void (const intrusive_ptr<ApiUser>&, const Value&)> OnPasswordChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ApiUser>&, const Value&)> OnPasswordHashChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ApiUser>&, const Value&)> OnClientCNChanged;
	static boost::signals2::signal<void (const intrusive_ptr<ApiUser>&, const Value&)> OnPermissionsChanged;
};

}
#endif /* APIUSER_TI */
