#include "base/exception.hpp"
#include "base/objectlock.hpp"
#include "base/utility.hpp"
#include "base/convert.hpp"
#include "base/debug.hpp"
#include "base/dependencygraph.hpp"
#include "base/logger.hpp"
#include "base/function.hpp"
#include "base/configobject.hpp"
#include "base/configtype.hpp"
#ifdef _MSC_VER
#pragma warning( push )
#pragma warning( disable : 4244 )
#pragma warning( disable : 4800 )
#endif /* _MSC_VER */

namespace icinga
{

TypeImpl<Endpoint>::TypeImpl()
{ }

TypeImpl<Endpoint>::~TypeImpl()
{ }

String TypeImpl<Endpoint>::GetName() const
{
	return "Endpoint";
}

int TypeImpl<Endpoint>::GetAttributes() const
{
	return 0;
}

Type::Ptr TypeImpl<Endpoint>::GetBaseType() const
{
	return ConfigObject::TypeInstance;
}

int TypeImpl<Endpoint>::GetFieldId(const String& name) const
{
	int offset = ConfigObject::TypeInstance->GetFieldCount();

	switch (static_cast<int>(Utility::SDBM(name, 7))) {
		case -2006476959:
			if (name == "last_message_received")
				return offset + 11;
			if (name == "last_message_sent")
				return offset + 12;

			break;
		case -1984735321:
			if (name == "messages_sent_per_second")
				return offset + 2;
			if (name == "messages_received_per_second")
				return offset + 3;
			if (name == "messages_received_per_type")
				return offset + 8;

			break;
		case -1844041471:
			if (name == "port")
				return offset + 0;

			break;
		case -1674083201:
			if (name == "seconds_processing_messages")
				return offset + 7;

			break;
		case -1053842266:
			if (name == "log_duration")
				return offset + 6;

			break;
		case -1030437066:
			if (name == "capabilities")
				return offset + 16;

			break;
		case -814341782:
			if (name == "connected")
				return offset + 13;
			if (name == "connecting")
				return offset + 15;

			break;
		case -759907650:
			if (name == "bytes_received_per_second")
				return offset + 5;

			break;
		case -759907649:
			if (name == "bytes_sent_per_second")
				return offset + 4;

			break;
		case 303636071:
			if (name == "syncing")
				return offset + 14;

			break;
		case 308691128:
			if (name == "local_log_position")
				return offset + 9;

			break;
		case 501261128:
			if (name == "host")
				return offset + 1;

			break;
		case 1883089209:
			if (name == "remote_log_position")
				return offset + 10;

			break;
		case 2009033958:
			if (name == "icinga_version")
				return offset + 17;

			break;
	}

	return ConfigObject::TypeInstance->GetFieldId(name);
}

Field TypeImpl<Endpoint>::GetFieldInfo(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount();
	if (real_id < 0) { return ConfigObject::TypeInstance->GetFieldInfo(id); }
	switch (real_id) {
		case 0:
			return {0, "String", "port", "port", nullptr, 258, 0};
		case 1:
			return {1, "String", "host", "host", nullptr, 2, 0};
		case 2:
			return {2, "Number", "messages_sent_per_second", "messages_sent_per_second", nullptr, 1089, 0};
		case 3:
			return {3, "Number", "messages_received_per_second", "messages_received_per_second", nullptr, 1089, 0};
		case 4:
			return {4, "Number", "bytes_sent_per_second", "bytes_sent_per_second", nullptr, 1089, 0};
		case 5:
			return {5, "Number", "bytes_received_per_second", "bytes_received_per_second", nullptr, 1089, 0};
		case 6:
			return {6, "Number", "log_duration", "log_duration", nullptr, 2, 0};
		case 7:
			return {7, "Number", "seconds_processing_messages", "seconds_processing_messages", nullptr, 1089, 0};
		case 8:
			return {8, "Dictionary", "messages_received_per_type", "messages_received_per_type", nullptr, 1089, 0};
		case 9:
			return {9, "Timestamp", "local_log_position", "local_log_position", nullptr, 4, 0};
		case 10:
			return {10, "Timestamp", "remote_log_position", "remote_log_position", nullptr, 4, 0};
		case 11:
			return {11, "Timestamp", "last_message_received", "last_message_received", nullptr, 1, 0};
		case 12:
			return {12, "Timestamp", "last_message_sent", "last_message_sent", nullptr, 1, 0};
		case 13:
			return {13, "Boolean", "connected", "connected", nullptr, 1089, 0};
		case 14:
			return {14, "Boolean", "syncing", "syncing", nullptr, 1025, 0};
		case 15:
			return {15, "Boolean", "connecting", "connecting", nullptr, 1025, 0};
		case 16:
			return {16, "uint_fast64_t", "capabilities", "capabilities", nullptr, 4, 0};
		case 17:
			return {17, "unsigned long", "icinga_version", "icinga_version", nullptr, 4, 0};
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

int TypeImpl<Endpoint>::GetFieldCount() const
{
	return 18 + ConfigObject::TypeInstance->GetFieldCount();
}

ObjectFactory TypeImpl<Endpoint>::GetFactory() const
{
	return TypeHelper<Endpoint, false>::GetFactory();
}

const std::unordered_set<Type*>& TypeImpl<Endpoint>::GetLoadDependencies() const
{
	static const auto deps ([] {
		auto typeZone (GetByName("Zone").get());
		VERIFY(typeZone);
		VERIFY(ConfigObject::TypeInstance->IsAssignableFrom(typeZone));

		return std::unordered_set<Type*>{ typeZone, };
	}());

	return deps;
}

int TypeImpl<Endpoint>::GetActivationPriority() const
{
	return 0;
}

void TypeImpl<Endpoint>::RegisterAttributeHandler(int fieldId, const Type::AttributeHandler& callback)
{
	int real_id = fieldId - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::TypeInstance->RegisterAttributeHandler(fieldId, callback); return; }
	switch (real_id) {
		case 0:
			ObjectImpl<Endpoint>::OnPortChanged.connect(callback);
			break;
		case 1:
			ObjectImpl<Endpoint>::OnHostChanged.connect(callback);
			break;
		case 2:
			ObjectImpl<Endpoint>::OnMessagesSentPerSecondChanged.connect(callback);
			break;
		case 3:
			ObjectImpl<Endpoint>::OnMessagesReceivedPerSecondChanged.connect(callback);
			break;
		case 4:
			ObjectImpl<Endpoint>::OnBytesSentPerSecondChanged.connect(callback);
			break;
		case 5:
			ObjectImpl<Endpoint>::OnBytesReceivedPerSecondChanged.connect(callback);
			break;
		case 6:
			ObjectImpl<Endpoint>::OnLogDurationChanged.connect(callback);
			break;
		case 7:
			ObjectImpl<Endpoint>::OnSecondsProcessingMessagesChanged.connect(callback);
			break;
		case 8:
			ObjectImpl<Endpoint>::OnMessagesReceivedPerTypeChanged.connect(callback);
			break;
		case 9:
			ObjectImpl<Endpoint>::OnLocalLogPositionChanged.connect(callback);
			break;
		case 10:
			ObjectImpl<Endpoint>::OnRemoteLogPositionChanged.connect(callback);
			break;
		case 11:
			ObjectImpl<Endpoint>::OnLastMessageReceivedChanged.connect(callback);
			break;
		case 12:
			ObjectImpl<Endpoint>::OnLastMessageSentChanged.connect(callback);
			break;
		case 13:
			ObjectImpl<Endpoint>::OnConnectedChanged.connect(callback);
			break;
		case 14:
			ObjectImpl<Endpoint>::OnSyncingChanged.connect(callback);
			break;
		case 15:
			ObjectImpl<Endpoint>::OnConnectingChanged.connect(callback);
			break;
		case 16:
			ObjectImpl<Endpoint>::OnCapabilitiesChanged.connect(callback);
			break;
		case 17:
			ObjectImpl<Endpoint>::OnIcingaVersionChanged.connect(callback);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Endpoint>::Validate(int types, const ValidationUtils& utils)
{
	ConfigObject::Validate(types, utils);

	if (2 & types)
		ValidatePort(Lazy<String>([this]() { return GetPort(); }), utils);
	if (2 & types)
		ValidateHost(Lazy<String>([this]() { return GetHost(); }), utils);
	if (1 & types)
		ValidateMessagesSentPerSecond(Lazy<double>([this]() { return GetMessagesSentPerSecond(); }), utils);
	if (1 & types)
		ValidateMessagesReceivedPerSecond(Lazy<double>([this]() { return GetMessagesReceivedPerSecond(); }), utils);
	if (1 & types)
		ValidateBytesSentPerSecond(Lazy<double>([this]() { return GetBytesSentPerSecond(); }), utils);
	if (1 & types)
		ValidateBytesReceivedPerSecond(Lazy<double>([this]() { return GetBytesReceivedPerSecond(); }), utils);
	if (2 & types)
		ValidateLogDuration(Lazy<double>([this]() { return GetLogDuration(); }), utils);
	if (1 & types)
		ValidateSecondsProcessingMessages(Lazy<double>([this]() { return GetSecondsProcessingMessages(); }), utils);
	if (1 & types)
		ValidateMessagesReceivedPerType(Lazy<Dictionary::Ptr>([this]() { return GetMessagesReceivedPerType(); }), utils);
	if (4 & types)
		ValidateLocalLogPosition(Lazy<Timestamp>([this]() { return GetLocalLogPosition(); }), utils);
	if (4 & types)
		ValidateRemoteLogPosition(Lazy<Timestamp>([this]() { return GetRemoteLogPosition(); }), utils);
	if (1 & types)
		ValidateLastMessageReceived(Lazy<Timestamp>([this]() { return GetLastMessageReceived(); }), utils);
	if (1 & types)
		ValidateLastMessageSent(Lazy<Timestamp>([this]() { return GetLastMessageSent(); }), utils);
	if (1 & types)
		ValidateConnected(Lazy<bool>([this]() { return GetConnected(); }), utils);
	if (1 & types)
		ValidateSyncing(Lazy<bool>([this]() { return GetSyncing(); }), utils);
	if (1 & types)
		ValidateConnecting(Lazy<bool>([this]() { return GetConnecting(); }), utils);
	if (4 & types)
		ValidateCapabilities(Lazy<uint_fast64_t>([this]() { return GetCapabilities(); }), utils);
	if (4 & types)
		ValidateIcingaVersion(Lazy<unsigned long>([this]() { return GetIcingaVersion(); }), utils);
}

void ObjectImpl<Endpoint>::SimpleValidatePort(const Lazy<String>& value, const ValidationUtils& utils)
{
	if (value().IsEmpty())
		BOOST_THROW_EXCEPTION(ValidationError(dynamic_cast<ConfigObject *>(this), { "port" }, "Attribute must not be empty."));

}

void ObjectImpl<Endpoint>::SimpleValidateHost(const Lazy<String>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateMessagesSentPerSecond(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateMessagesReceivedPerSecond(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateBytesSentPerSecond(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateBytesReceivedPerSecond(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateLogDuration(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateSecondsProcessingMessages(const Lazy<double>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateMessagesReceivedPerType(const Lazy<Dictionary::Ptr>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateLocalLogPosition(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateRemoteLogPosition(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateLastMessageReceived(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateLastMessageSent(const Lazy<Timestamp>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateConnected(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateSyncing(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateConnecting(const Lazy<bool>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateCapabilities(const Lazy<uint_fast64_t>& value, const ValidationUtils& utils)
{
}

void ObjectImpl<Endpoint>::SimpleValidateIcingaVersion(const Lazy<unsigned long>& value, const ValidationUtils& utils)
{
}

ObjectImpl<Endpoint>::ObjectImpl()
{
	SetPort(GetDefaultPort(), true);
	SetHost(GetDefaultHost(), true);
	SetMessagesSentPerSecond(GetDefaultMessagesSentPerSecond(), true);
	SetMessagesReceivedPerSecond(GetDefaultMessagesReceivedPerSecond(), true);
	SetBytesSentPerSecond(GetDefaultBytesSentPerSecond(), true);
	SetBytesReceivedPerSecond(GetDefaultBytesReceivedPerSecond(), true);
	SetLogDuration(GetDefaultLogDuration(), true);
	SetSecondsProcessingMessages(GetDefaultSecondsProcessingMessages(), true);
	SetMessagesReceivedPerType(GetDefaultMessagesReceivedPerType(), true);
	SetLocalLogPosition(GetDefaultLocalLogPosition(), true);
	SetRemoteLogPosition(GetDefaultRemoteLogPosition(), true);
	SetLastMessageReceived(GetDefaultLastMessageReceived(), true);
	SetLastMessageSent(GetDefaultLastMessageSent(), true);
	SetConnected(GetDefaultConnected(), true);
	SetSyncing(GetDefaultSyncing(), true);
	SetConnecting(GetDefaultConnecting(), true);
	SetCapabilities(GetDefaultCapabilities(), true);
	SetIcingaVersion(GetDefaultIcingaVersion(), true);
}

ObjectImpl<Endpoint>::~ObjectImpl()
{ }

void ObjectImpl<Endpoint>::SetField(int id, const Value& value, bool suppress_events, const Value& cookie)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::SetField(id, value, suppress_events, cookie); return; }
	switch (real_id) {
		case 0:
			SetPort(value, suppress_events, cookie);
			break;
		case 1:
			SetHost(value, suppress_events, cookie);
			break;
		case 2:
			SetMessagesSentPerSecond(value, suppress_events, cookie);
			break;
		case 3:
			SetMessagesReceivedPerSecond(value, suppress_events, cookie);
			break;
		case 4:
			SetBytesSentPerSecond(value, suppress_events, cookie);
			break;
		case 5:
			SetBytesReceivedPerSecond(value, suppress_events, cookie);
			break;
		case 6:
			SetLogDuration(value, suppress_events, cookie);
			break;
		case 7:
			SetSecondsProcessingMessages(value, suppress_events, cookie);
			break;
		case 8:
			SetMessagesReceivedPerType(value, suppress_events, cookie);
			break;
		case 9:
			SetLocalLogPosition(value, suppress_events, cookie);
			break;
		case 10:
			SetRemoteLogPosition(value, suppress_events, cookie);
			break;
		case 11:
			SetLastMessageReceived(value, suppress_events, cookie);
			break;
		case 12:
			SetLastMessageSent(value, suppress_events, cookie);
			break;
		case 13:
			SetConnected(value, suppress_events, cookie);
			break;
		case 14:
			SetSyncing(value, suppress_events, cookie);
			break;
		case 15:
			SetConnecting(value, suppress_events, cookie);
			break;
		case 16:
			SetCapabilities(value, suppress_events, cookie);
			break;
		case 17:
			SetIcingaVersion(value, suppress_events, cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Value ObjectImpl<Endpoint>::GetField(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObject::GetField(id); }
	switch (real_id) {
		case 0:
			return GetPort();
		case 1:
			return GetHost();
		case 2:
			return GetMessagesSentPerSecond();
		case 3:
			return GetMessagesReceivedPerSecond();
		case 4:
			return GetBytesSentPerSecond();
		case 5:
			return GetBytesReceivedPerSecond();
		case 6:
			return GetLogDuration();
		case 7:
			return GetSecondsProcessingMessages();
		case 8:
			return GetMessagesReceivedPerType();
		case 9:
			return GetLocalLogPosition();
		case 10:
			return GetRemoteLogPosition();
		case 11:
			return GetLastMessageReceived();
		case 12:
			return GetLastMessageSent();
		case 13:
			return GetConnected();
		case 14:
			return GetSyncing();
		case 15:
			return GetConnecting();
		case 16:
			return GetCapabilities();
		case 17:
			return GetIcingaVersion();
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Endpoint>::ValidateField(int id, const Lazy<Value>& lvalue, const ValidationUtils& utils)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::ValidateField(id, lvalue, utils); return; }
	switch (real_id) {
		case 0:
			ValidatePort(lvalue, utils);
			break;
		case 1:
			ValidateHost(lvalue, utils);
			break;
		case 2:
			ValidateMessagesSentPerSecond(lvalue, utils);
			break;
		case 3:
			ValidateMessagesReceivedPerSecond(lvalue, utils);
			break;
		case 4:
			ValidateBytesSentPerSecond(lvalue, utils);
			break;
		case 5:
			ValidateBytesReceivedPerSecond(lvalue, utils);
			break;
		case 6:
			ValidateLogDuration(lvalue, utils);
			break;
		case 7:
			ValidateSecondsProcessingMessages(lvalue, utils);
			break;
		case 8:
			ValidateMessagesReceivedPerType(lvalue, utils);
			break;
		case 9:
			ValidateLocalLogPosition(lvalue, utils);
			break;
		case 10:
			ValidateRemoteLogPosition(lvalue, utils);
			break;
		case 11:
			ValidateLastMessageReceived(lvalue, utils);
			break;
		case 12:
			ValidateLastMessageSent(lvalue, utils);
			break;
		case 13:
			ValidateConnected(lvalue, utils);
			break;
		case 14:
			ValidateSyncing(lvalue, utils);
			break;
		case 15:
			ValidateConnecting(lvalue, utils);
			break;
		case 16:
			ValidateCapabilities(lvalue, utils);
			break;
		case 17:
			ValidateIcingaVersion(lvalue, utils);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

void ObjectImpl<Endpoint>::NotifyField(int id, const Value& cookie)
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { ConfigObject::NotifyField(id, cookie); return; }
	switch (real_id) {
		case 0:
			NotifyPort(cookie);
			break;
		case 1:
			NotifyHost(cookie);
			break;
		case 2:
			NotifyMessagesSentPerSecond(cookie);
			break;
		case 3:
			NotifyMessagesReceivedPerSecond(cookie);
			break;
		case 4:
			NotifyBytesSentPerSecond(cookie);
			break;
		case 5:
			NotifyBytesReceivedPerSecond(cookie);
			break;
		case 6:
			NotifyLogDuration(cookie);
			break;
		case 7:
			NotifySecondsProcessingMessages(cookie);
			break;
		case 8:
			NotifyMessagesReceivedPerType(cookie);
			break;
		case 9:
			NotifyLocalLogPosition(cookie);
			break;
		case 10:
			NotifyRemoteLogPosition(cookie);
			break;
		case 11:
			NotifyLastMessageReceived(cookie);
			break;
		case 12:
			NotifyLastMessageSent(cookie);
			break;
		case 13:
			NotifyConnected(cookie);
			break;
		case 14:
			NotifySyncing(cookie);
			break;
		case 15:
			NotifyConnecting(cookie);
			break;
		case 16:
			NotifyCapabilities(cookie);
			break;
		case 17:
			NotifyIcingaVersion(cookie);
			break;
		default:
			throw std::runtime_error("Invalid field ID.");
	}
}

Object::Ptr ObjectImpl<Endpoint>::NavigateField(int id) const
{
	int real_id = id - ConfigObject::TypeInstance->GetFieldCount(); 
	if (real_id < 0) { return ConfigObject::NavigateField(id); }
	throw std::runtime_error("Invalid field ID.");
}

String ObjectImpl<Endpoint>::GetPort() const
{
	return m_Port.load();
}

String ObjectImpl<Endpoint>::GetHost() const
{
	return m_Host.load();
}

double ObjectImpl<Endpoint>::GetLogDuration() const
{
	return m_LogDuration.load();
}

Timestamp ObjectImpl<Endpoint>::GetLocalLogPosition() const
{
	return m_LocalLogPosition.load();
}

Timestamp ObjectImpl<Endpoint>::GetRemoteLogPosition() const
{
	return m_RemoteLogPosition.load();
}

Timestamp ObjectImpl<Endpoint>::GetLastMessageReceived() const
{
	return m_LastMessageReceived.load();
}

Timestamp ObjectImpl<Endpoint>::GetLastMessageSent() const
{
	return m_LastMessageSent.load();
}

bool ObjectImpl<Endpoint>::GetSyncing() const
{
	return m_Syncing.load();
}

bool ObjectImpl<Endpoint>::GetConnecting() const
{
	return m_Connecting.load();
}

uint_fast64_t ObjectImpl<Endpoint>::GetCapabilities() const
{
	return m_Capabilities.load();
}

unsigned long ObjectImpl<Endpoint>::GetIcingaVersion() const
{
	return m_IcingaVersion.load();
}

void ObjectImpl<Endpoint>::SetPort(const String& value, bool suppress_events, const Value& cookie)
{
	m_Port.store(value);
	if (!suppress_events) {
		NotifyPort(cookie);
	}

}

void ObjectImpl<Endpoint>::SetHost(const String& value, bool suppress_events, const Value& cookie)
{
	m_Host.store(value);
	if (!suppress_events) {
		NotifyHost(cookie);
	}

}

void ObjectImpl<Endpoint>::SetMessagesSentPerSecond(double value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyMessagesSentPerSecond(cookie);
	}

}

void ObjectImpl<Endpoint>::SetMessagesReceivedPerSecond(double value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyMessagesReceivedPerSecond(cookie);
	}

}

void ObjectImpl<Endpoint>::SetBytesSentPerSecond(double value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyBytesSentPerSecond(cookie);
	}

}

void ObjectImpl<Endpoint>::SetBytesReceivedPerSecond(double value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyBytesReceivedPerSecond(cookie);
	}

}

void ObjectImpl<Endpoint>::SetLogDuration(double value, bool suppress_events, const Value& cookie)
{
	m_LogDuration.store(value);
	if (!suppress_events) {
		NotifyLogDuration(cookie);
	}

}

void ObjectImpl<Endpoint>::SetSecondsProcessingMessages(double value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifySecondsProcessingMessages(cookie);
	}

}

void ObjectImpl<Endpoint>::SetMessagesReceivedPerType(const Dictionary::Ptr& value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyMessagesReceivedPerType(cookie);
	}

}

void ObjectImpl<Endpoint>::SetLocalLogPosition(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LocalLogPosition.store(value);
	if (!suppress_events) {
		NotifyLocalLogPosition(cookie);
	}

}

void ObjectImpl<Endpoint>::SetRemoteLogPosition(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_RemoteLogPosition.store(value);
	if (!suppress_events) {
		NotifyRemoteLogPosition(cookie);
	}

}

void ObjectImpl<Endpoint>::SetLastMessageReceived(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastMessageReceived.store(value);
	if (!suppress_events) {
		NotifyLastMessageReceived(cookie);
	}

}

void ObjectImpl<Endpoint>::SetLastMessageSent(const Timestamp& value, bool suppress_events, const Value& cookie)
{
	m_LastMessageSent.store(value);
	if (!suppress_events) {
		NotifyLastMessageSent(cookie);
	}

}

void ObjectImpl<Endpoint>::SetConnected(bool value, bool suppress_events, const Value& cookie)
{


	if (!suppress_events) {
		NotifyConnected(cookie);
	}

}

void ObjectImpl<Endpoint>::SetSyncing(bool value, bool suppress_events, const Value& cookie)
{
	m_Syncing.store(value);
	if (!suppress_events) {
		NotifySyncing(cookie);
	}

}

void ObjectImpl<Endpoint>::SetConnecting(bool value, bool suppress_events, const Value& cookie)
{
	m_Connecting.store(value);
	if (!suppress_events) {
		NotifyConnecting(cookie);
	}

}

void ObjectImpl<Endpoint>::SetCapabilities(const uint_fast64_t& value, bool suppress_events, const Value& cookie)
{
	m_Capabilities.store(value);
	if (!suppress_events) {
		NotifyCapabilities(cookie);
	}

}

void ObjectImpl<Endpoint>::SetIcingaVersion(const unsigned long& value, bool suppress_events, const Value& cookie)
{
	m_IcingaVersion.store(value);
	if (!suppress_events) {
		NotifyIcingaVersion(cookie);
	}

}

void ObjectImpl<Endpoint>::NotifyPort(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnPortChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyHost(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnHostChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyMessagesSentPerSecond(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnMessagesSentPerSecondChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyMessagesReceivedPerSecond(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnMessagesReceivedPerSecondChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyBytesSentPerSecond(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnBytesSentPerSecondChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyBytesReceivedPerSecond(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnBytesReceivedPerSecondChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyLogDuration(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLogDurationChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifySecondsProcessingMessages(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnSecondsProcessingMessagesChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyMessagesReceivedPerType(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnMessagesReceivedPerTypeChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyLocalLogPosition(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLocalLogPositionChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyRemoteLogPosition(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnRemoteLogPositionChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyLastMessageReceived(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastMessageReceivedChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyLastMessageSent(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnLastMessageSentChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyConnected(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnConnectedChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifySyncing(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnSyncingChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyConnecting(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnConnectingChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyCapabilities(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnCapabilitiesChanged(static_cast<Endpoint *>(this), cookie);
}

void ObjectImpl<Endpoint>::NotifyIcingaVersion(const Value& cookie)
{
	auto *dobj = dynamic_cast<ConfigObject *>(this);
	if (!dobj || dobj->IsActive())
		OnIcingaVersionChanged(static_cast<Endpoint *>(this), cookie);
}

String ObjectImpl<Endpoint>::GetDefaultPort() const
{
	 return "5665"; 
}

String ObjectImpl<Endpoint>::GetDefaultHost() const
{
	return String();
}

double ObjectImpl<Endpoint>::GetDefaultMessagesSentPerSecond() const
{
	return double();
}

double ObjectImpl<Endpoint>::GetDefaultMessagesReceivedPerSecond() const
{
	return double();
}

double ObjectImpl<Endpoint>::GetDefaultBytesSentPerSecond() const
{
	return double();
}

double ObjectImpl<Endpoint>::GetDefaultBytesReceivedPerSecond() const
{
	return double();
}

double ObjectImpl<Endpoint>::GetDefaultLogDuration() const
{
	 return 86400; 
}

double ObjectImpl<Endpoint>::GetDefaultSecondsProcessingMessages() const
{
	return double();
}

Dictionary::Ptr ObjectImpl<Endpoint>::GetDefaultMessagesReceivedPerType() const
{
	return Dictionary::Ptr();
}

Timestamp ObjectImpl<Endpoint>::GetDefaultLocalLogPosition() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Endpoint>::GetDefaultRemoteLogPosition() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Endpoint>::GetDefaultLastMessageReceived() const
{
	return Timestamp();
}

Timestamp ObjectImpl<Endpoint>::GetDefaultLastMessageSent() const
{
	return Timestamp();
}

bool ObjectImpl<Endpoint>::GetDefaultConnected() const
{
	return bool();
}

bool ObjectImpl<Endpoint>::GetDefaultSyncing() const
{
	return bool();
}

bool ObjectImpl<Endpoint>::GetDefaultConnecting() const
{
	return bool();
}

uint_fast64_t ObjectImpl<Endpoint>::GetDefaultCapabilities() const
{
	 return 0; 
}

unsigned long ObjectImpl<Endpoint>::GetDefaultIcingaVersion() const
{
	 return 0; 
}


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnPortChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnHostChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnMessagesSentPerSecondChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnMessagesReceivedPerSecondChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnBytesSentPerSecondChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnBytesReceivedPerSecondChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnLogDurationChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnSecondsProcessingMessagesChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnMessagesReceivedPerTypeChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnLocalLogPositionChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnRemoteLogPositionChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnLastMessageReceivedChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnLastMessageSentChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnConnectedChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnSyncingChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnConnectingChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnCapabilitiesChanged;


boost::signals2::signal<void (const intrusive_ptr<Endpoint>&, const Value&)> ObjectImpl<Endpoint>::OnIcingaVersionChanged;

void ObjectImpl<Endpoint>::ValidateBytesReceivedPerSecond(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateBytesReceivedPerSecond(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateBytesSentPerSecond(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateBytesSentPerSecond(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateCapabilities(const Lazy<uint_fast64_t>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateCapabilities(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateConnected(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateConnected(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateConnecting(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateConnecting(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateHost(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateHost(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateIcingaVersion(const Lazy<unsigned long>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateIcingaVersion(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateLastMessageReceived(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastMessageReceived(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateLastMessageSent(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLastMessageSent(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateLocalLogPosition(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLocalLogPosition(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateLogDuration(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateLogDuration(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateMessagesReceivedPerSecond(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateMessagesReceivedPerSecond(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateMessagesReceivedPerType(const Lazy<Dictionary::Ptr>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateMessagesReceivedPerType(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateMessagesSentPerSecond(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateMessagesSentPerSecond(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidatePort(const Lazy<String>& lvalue, const ValidationUtils& utils)
{
	SimpleValidatePort(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateRemoteLogPosition(const Lazy<Timestamp>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateRemoteLogPosition(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateSecondsProcessingMessages(const Lazy<double>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateSecondsProcessingMessages(lvalue, utils);
}

void ObjectImpl<Endpoint>::ValidateSyncing(const Lazy<bool>& lvalue, const ValidationUtils& utils)
{
	SimpleValidateSyncing(lvalue, utils);
}

}
#ifdef _MSC_VER
#pragma warning ( pop )
#endif /* _MSC_VER */
